<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabPlayer {

	/**
	 * Icons library meta files.
	 * @var array
	 */
	private static $icons_meta = [
		'speaker.json',
		'material.json'
	];

	/**
	 * Controls for Design tab.
	 * @return void
	 */
	public static function controls() {

		$tabs = Plugin::get_tabs();

		$options = Settings::get_instance()->options;

		$fields = array();
		$fields = array_merge(
			self::fields_player_general_style( $fields, $options ),
			self::fields_player_style( $fields, $options ),
			self::fields_player_waveform( $fields, $options ),
			self::fields_player_play_pause( $fields, $options ),
			self::fields_player_repeat( $fields ),
			self::fields_player_timing( $fields, $options ),
			self::fields_player_mute( $fields ),
			self::fields_player_download( $fields, $options ),
			self::fields_player_speed( $fields, $options ),
			self::fields_player_share( $fields ),
			self::fields_player_thumbnail( $fields, $options ),
			self::fields_player_initial_settings( $fields, $options )
		);

		$tabs[ 'player' ][ 'fields' ] = $fields;

		Plugin::set_tabs( $tabs );

	}

	/**
	 * General player style
	 *
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_general_style( array $fields, array $options ): array {

		// Available player styles
		$style_options = [
			'speaker-wavesurfer'            => esc_html__( 'Wavesurfer player', 'speaker' ),
			'speaker-mediaelement'          => esc_html__( 'MediaElement player', 'speaker' ),
			"speaker-wp-default"            => esc_html__( 'WordPress default player', 'speaker' ),
			"speaker-chrome"                => esc_html__( 'Chrome Style player', 'speaker' ),
			"speaker-browser-default"       => esc_html__( 'Browser default player', 'speaker' ),
		];

		$fields[ 'player_position' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Position', 'speaker' ),
			'description'       => esc_html__( 'Select the Player position or use shortcode.', 'speaker' ),
			'default'           => $options[ 'position' ] ?? 'before-content',
			'options'           => [
				"before-content"    => esc_html__( 'Before Content', 'speaker' ),
				"after-content"     => esc_html__( 'After Content', 'speaker' ),
				"top-fixed"         => esc_html__( 'Top Fixed', 'speaker' ),
				"bottom-fixed"      => esc_html__( 'Bottom Fixed', 'speaker' ),
				"before-title"      => esc_html__( 'Before Title', 'speaker' ),
				"after-title"       => esc_html__( 'After Title', 'speaker' ),
				"before-filter"     => esc_html__( 'Before in Custom Filter', 'speaker' ),
				"after-filter"      => esc_html__( 'After in Custom Filter', 'speaker' ),
				"action"            => esc_html__( 'WordPress Hook(action)', 'speaker' ),
				"shortcode"         => esc_html__( 'Shortcode', 'speaker' ) . '&nbsp;<b>[speaker]</b>',
			]
		];

		// TODO: SpeakerUtilities
		$fields[ 'custom_filter' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Custom filter name', 'speaker' ),
			'description'       => esc_html__( 'Specify a theme filter or custom filter before(or after) which the player will be added', 'speaker' ),
			'default'           => 'the_content',
		];

		// TODO: SpeakerUtilities
		$fields[ 'custom_action' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Custom action name', 'speaker' ),
			'description'       => esc_html__( 'Custom WordPress action for which the player will be added', 'speaker' ),
			'default'           => '',
		];

		$fields[ 'player_style' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Player Style', 'speaker' ),
			'description'       => esc_html__( 'Select one of the Player styles', 'speaker' ),
			'default'           => self::player_style(),
			'options'           => $style_options
		];

		return $fields;

	}

	/**
	 * Player style
	 * @return mixed|string
	 */
	private static function player_style() {

		$options = Settings::get_instance()->options;

		// Default player style from old version
		if ( isset( $options[ 'style' ] ) ) {

			if ( in_array( $options[ 'style' ], [ 'speaker-round', 'speaker-round', 'speaker-squared' ] ) ) {
				$player_style_default = 'speaker-mediaelement';
			} else {
				$player_style_default = $options[ 'style' ];
			}

		} else {

			$player_style_default = 'speaker-wavesurfer';

		}

		return $player_style_default;

	}

	/**
	 * Player box style
	 *
	 * @param array $fields     Settings fields
	 * @param array $options    Plugins settings
	 *
	 * @return array
	 */
	private static function fields_player_style( array $fields, array $options ): array {

		$prefix = 'player';
		$old_player_color = $options[ 'bgcolor' ] ?? 'rgba(2, 83, 238, 1)';

		$fields[ $prefix . '_margin' ] = [
			'type' => 'sides',
			'label'             => esc_html__( 'Margin', 'speaker' ) . ':',
			'description'       => esc_html__( 'Margin', 'speaker' ),
			'default'           => [
				'top' => '0',
				'right' => '0',
				'bottom' => '20',
				'left' => '0',
				'unit' => 'px',
				'linked' => true
			],
		];

		$fields[ $prefix . '_padding'] = [
			'type' => 'sides',
			'label'             => esc_html__( 'Padding', 'speaker' ) . ':',
			'description'       => esc_html__( 'Padding', 'speaker' ),
			'default'           => [
				'top' => '0',
				'right' => '0',
				'bottom' => '0',
				'left' => '0',
				'unit' => 'px',
				'linked' => true
			],
		];

		$fields[ $prefix . '_border_radius'] = [
			'type' => 'sides',
			'label'             => esc_html__( 'Border radius', 'speaker' ),
			'description'       => esc_html__( 'Border radius', 'speaker' ),
			'default'           => self::default_border_radius( $options ),
		];

		$key = 'row_gap';
		$row_gap = isset( $options[ $key ] ) && $options[ $key ] ? $options[ $key ] : 20;
		$fields[ $key ] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Row gap', 'speaker' ),
			'description'       => wp_sprintf(
			/* translators: %ps: row gap value */
				esc_html__( 'The vertical gap between the player and other elements: speed bar, download bar, share bar and text block. Current gap is: %s px', 'speaker' ),
				'<strong>' . esc_attr( $row_gap ) . '</strong>'
			),
			'default'           => $row_gap,
			'min'               => 0,
			'max'               => 100,
			'step'              => 1,
		];

		$fields[ $prefix . '_color'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Background color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Background color', 'speaker' ),
			'description'       => esc_html__( 'Select background player color', 'speaker' ),
			'default'           => 'rgba(2, 83, 238, .09)',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$prefix = 'wavesurfer';


		$fields[ 'player_color_controls'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Controls color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Controls color', 'speaker' ),
			'description'       => esc_html__( 'Select the player controls color', 'speaker' ),
			'default'           => $old_player_color,
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$fields[ 'player_color_controls_hover'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Controls color hover', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Controls color hover', 'speaker' ),
			'description'       => esc_html__( 'Select the player controls color on hover', 'speaker' ),
			'default'           => '#ffffff',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$fields[ $prefix . '_color_hover'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Controls background hover color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Controls background hover color', 'speaker' ),
			'description'       => esc_html__( 'Select controls background color on hover', 'speaker' ),
			'default'           => $old_player_color,
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$key = 'player_icon_size';
		$default = $options[ $key ] ?? 16;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Controls size', 'speaker' ) . ':',
			'description'       => wp_sprintf(
				/* translators: %s: current value */
				esc_html__( 'Icons size: %spx', 'speaker' ),
				'<strong>' . $default . '</strong>'
			),
			'min'               => 4,
			'max'               => 32,
			'step'              => 1,
			'default'           => $default,
		];

		$fields[ 'player_color_text' ] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Text color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Text color', 'speaker' ),
			'description'       => esc_html__( 'Select the player text color', 'speaker' ),
			'default'           => '#446cb8',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$key = 'player_font_size';
		$default = $options[ $key ] ?? 12;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Font size', 'speaker' ) . ':',
			'description'       => wp_sprintf(
			/* translators: %s: current value */
				esc_html__( 'Font size: %spx', 'speaker' ),
				'<strong>' . $default . '</strong>'
			),
			'min'               => 4,
			'max'               => 32,
			'step'              => 1,
			'default'           => $default,
		];

		return $fields;

	}

	/**
	 * Player thumbnail
	 *
	 * @param array $fields     Settings fields
	 * @param array $options    Plugins settings
	 *
	 * @return array
	 */
	private static function fields_player_thumbnail( array $fields, array $options ): array {

		$fields[ 'player_divider_thumbnail' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_thumbnail' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Thumbnail', 'speaker' ),
			'description'       => esc_html__( 'Select the thumbnail position', 'speaker' ),
			'default'           => 'thumbnail',
			'options'           => [
				'none'              => esc_html__( 'None', 'speaker' ),
				'background'        => esc_html__( 'Background', 'speaker' ),
				'above'             => esc_html__( 'Above', 'speaker' ),
				'below'             => esc_html__( 'Below', 'speaker' ),
				'in-play'           => esc_html__( 'In play button', 'speaker' ),
			],
		];

		$fields[ 'wavesurfer_thumbnail_overlay' ] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Thumbnail overlay', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Thumbnail overlay color', 'speaker' ),
			'description'       => esc_html__( 'Select the thumbnail overlay color', 'speaker' ),
			'default'           => 'rgba(232, 238, 253, 0.85)',
		];

		$key = 'player_thumbnail_height';
		$default = $options[ $key ] ?? 250;
		$fields[ $key ] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Thumbnail height', 'speaker' ),
			'description'       => wp_sprintf(
				/* translators: %s: current value */
				esc_html__( 'Select the thumbnail height in pixels. Current: %spx', 'speaker' ),
				'<strong>' . esc_attr( $default ) . '</strong>'
			),
			'default'           => $default,
			'min'           => 0,
			'max'           => 600,
			'step'          => 1,
		];

		return $fields;

	}

	/**
	 * Waveform settings and Progress bar settings
	 * @param array $fields     Settings fields
	 * @param array $options    Plugins settings
	 *
	 * @return array
	 */
	private static function fields_player_waveform( array $fields, array $options ): array {

		$prefix = 'wavesurfer';

		$fields[ 'divider_waveform' ] = [
			'type'              => 'divider',
		];

		$fields[ $prefix . '_waveform' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Wave', 'speaker' ),
			'description'       => esc_html__( 'Show waveform', 'speaker' ),
			'default'           => 'on',
		];

		$old_player_color = $options[ 'bgcolor' ] ?? 'rgba(2, 83, 238, 1)';
		$fields[ 'player_color_wave'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Track color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Track color', 'speaker' ),
			'description'       => esc_html__( 'Select the wave color', 'speaker' ),
			'default'           => 'rgba(2, 83, 238, .5)',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$fields[ 'player_color_progress'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Progress color', 'speaker' ) . ':',
			'placeholder'       => esc_html__( 'Progress color', 'speaker' ),
			'description'       => esc_html__( 'Select the player progress color', 'speaker' ),
			'default'           => $old_player_color,
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$fields[ $prefix . '_color_cursor' ] = [
			'type'  => 'colorpicker',
			'label' => esc_html__( 'Cursor color', 'speaker' ) . ':',
			'placeholder' => esc_html__( 'Cursor color', 'speaker' ),
			'description' => esc_html__( 'Select the player cursor color', 'speaker' ),
			'default' => $old_player_color,
			'attr' => [
				'readonly' => 'readonly',
			]
		];

		$key = $prefix . '_wave_height';
		$default = $options[ $key ] ?? 40;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Waveform height', 'speaker' ) . ':',
			'description'       => esc_html__( 'Waveform height duration: ', 'speaker' ) .
			                       '<strong>' . $default . '</strong>px',
			'min'               => 0,
			'max'               => 200,
			'step'              => 1,
			'default'           => $default,
			'discrete'          => true,
		];

		$key = $prefix . '_bar_radius';
		$default = $options[ $key ] ?? 2;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Wave bar radius', 'speaker' ) . ':',
			'description'       => esc_html__( 'Wave bar radius: ', 'speaker' ) .
			                       '<strong>' . $default . '</strong>px',
			'min'               => 0,
			'max'               => 10,
			'step'              => 1,
			'default'           => $default,
			'discrete'          => true,
		];

		$key = $prefix . '_bar_width';
		$default = $options[ $key ] ?? 2;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Wave bar width', 'speaker' ) . ':',
			'description'       => esc_html__( 'Wave bar width: ', 'speaker' ) .
			                       '<strong>' . $default . '</strong>px',
			'min'               => 0,
			'max'               => 10,
			'step'              => 1,
			'default'           => $default,
			'discrete'          => true,
		];

		$key = $prefix . '_bar_min_height';
		$default = $options[ $key ] ?? 2;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Wave bar min-height', 'speaker' ) . ':',
			'description'       => esc_html__( 'Wave bar min-height: ', 'speaker' ) .
			                       '<strong>' . $default . '</strong>px',
			'min'               => 0,
			'max'               => 10,
			'step'              => 1,
			'default'           => $default,
			'discrete'          => true,
		];

		$key = $prefix . '_bar_gap';
		$default = $options[ $key ] ?? 1;
		$fields[$key] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Wave bar gap', 'speaker' ) . ':',
			'description'       => esc_html__( 'Wave bar gap: ', 'speaker' ) .
			                       '<strong>' . $default . '</strong>px',
			'min'               => 0,
			'max'               => 10,
			'step'              => 1,
			'default'           => $default,
			'discrete'          => true,
		];

		return $fields;

	}

	/**
	 * Player play/pause fields
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_play_pause( array $fields, array $options ): array {

		$fields[ 'divider_play_pause' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_icon_play' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Play icon', 'speaker' ),
			'description'       => esc_html__( 'Play icon for wave player', 'speaker' ),
			'default'           => 'speaker/play.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'player_icon_pause' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Pause icon', 'speaker' ),
			'description'       => esc_html__( 'Pause icon for wave player', 'speaker' ),
			'default'           => 'speaker/pause.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'wavesurfer_large_play_pause' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Large play button', 'speaker' ),
			'description'       => esc_html__( 'Show large play button', 'speaker' ),
			'default'           => 'on',
		];

		$key = 'wavesurfer_thumbnail_width';
		$default = $options[ $key ] ?? 100;
		$fields[ $key ] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Large play button width', 'speaker' ),
			'description'       => wp_sprintf(
				/* translators: %s: current value */
				esc_html__( 'Set the large play button width in pixels. Current: %spx', 'speaker' ),
				'<strong>' . esc_attr( $default ) . '</strong>'
			),
			'default'           => 100,
			'min'               => 0,
			'max'               => 200,
			'step'              => 1,
		];

		return $fields;

	}

	/**
	 * Player repeat fields
	 *
	 * @param array $fields
	 *
	 * @return array
	 */
	private static function fields_player_repeat( array $fields ): array {

		$fields[ 'divider_repeat' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_repeat' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Repeat button', 'speaker' ),
			'description'       => esc_html__( 'Show repeat button', 'speaker' ),
			'default'           => 'on',
		];

		$fields[ 'player_icon_repeat_on' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Repeat ON icon', 'speaker' ),
			'description'       => esc_html__( 'Repeat icon for wave player', 'speaker' ),
			'default'           => 'speaker/repeat-on.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'player_icon_repeat_off' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Repeat OFF icon', 'speaker' ),
			'description'       => esc_html__( 'Repeat icon for wave player', 'speaker' ),
			'default'           => 'speaker/repeat-off.svg',
			'meta'              => self::$icons_meta
		];

		return $fields;

	}

	/**
	 * Player duration and elapsed fields
	 *
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_timing( array $fields, array $options ): array {

		$fields[ 'divider_timing' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_elapsed' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Current time', 'speaker' ),
			'description'       => esc_html__( 'Show current time', 'speaker' ),
			'default'           => 'on',
		];

		$fields[ 'player_duration' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Duration', 'speaker' ),
			'description'       => esc_html__( 'Show duration', 'speaker' ),
			'default'           => 'on',
		];

		$default_player_preload = $options[ 'preload' ] ?? 'metadata';
		$fields[ 'player_preload' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Preload', 'speaker' ),
			'description'       => esc_html__( 'The preload attribute specifies if and how the audio file should be loaded when the page loads.', 'speaker' ),
			'default'           => $default_player_preload ,
			'options'           => [
				"none"      => esc_html__( 'None', 'speaker' ),
				"metadata"  => esc_html__( 'Metadata', 'speaker' ),
				"auto"      => esc_html__( 'Auto', 'speaker' ),
				"backend"   => esc_html__( 'Backend', 'speaker' ),
			]
		];

		return $fields;

	}

	/**
	 * Player mute fields
	 *
	 * @param array $fields
	 *
	 * @return array
	 */
	private static function fields_player_mute( array $fields ): array {

		$fields[ 'divider_mute' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_mute' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Mute button', 'speaker' ),
			'description'       => esc_html__( 'Show mute button', 'speaker' ),
			'default'           => 'on',
		];

		$fields[ 'player_icon_mute' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Mute icon', 'speaker' ),
			'description'       => esc_html__( 'Mute icon for wave player', 'speaker' ),
			'default'           => 'speaker/mute-on.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'player_icon_volume' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Volume icon', 'speaker' ),
			'description'       => esc_html__( 'Volume icon for wave player', 'speaker' ),
			'default'           => 'speaker/mute-off.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'player_volume' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Volume bar', 'speaker' ),
			'description'       => esc_html__( 'Show volume bar', 'speaker' ),
			'default'           => 'on',
		];

		return $fields;

	}

	/**
	 * Player download fields
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_download( array $fields, array $options ): array {

		$fields[ 'divider_download' ] = [
			'type'              => 'divider',
		];

		$default_player_download = isset( $options[ 'link' ] ) && $options[ 'link' ] === 'none' ? 'off' : 'on';
		$fields[ 'player_download' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Download button', 'speaker' ),
			'description'       => esc_html__( 'Show download button', 'speaker' ),
			'default'           => $default_player_download,
		];

		$default_player_download_link = $options[ 'link' ] ?? 'inline';
		$fields[ 'player_download_link' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Download button', 'speaker' ),
			'description'       => esc_html__( 'Position of the Download audio link', 'speaker' ),
			'default'           => $default_player_download_link,
			'options'           => [
				"frontend-above"        => esc_html__( 'Above player ', 'speaker' ),
				"frontend"              => esc_html__( 'Below player ', 'speaker' ),
				"inline"                => esc_html__( 'Player layout', 'speaker' )
			]
		];

		$fields[ 'player_icon_download' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Download icon', 'speaker' ),
			'description'       => esc_html__( 'Download icon for wave player', 'speaker' ),
			'default'           => 'speaker/download.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'download_label' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Download label', 'speaker' ),
			'description'       => esc_html__( 'Specify the label for download link', 'speaker' ),
			'default'           => esc_html__( 'Download:', 'speaker' )
		];

		$default_player_download_link_backend = isset( $options[ 'link' ] ) && in_array( $options[ 'link' ], [ 'backend', 'backend-and-frontend' ] ) ? 'on' : 'off';
		$fields[ 'player_download_link_backend' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Backend download link', 'speaker' ),
			'description'       => esc_html__( 'Show download link in backend', 'speaker' ),
			'default'           => $default_player_download_link_backend
		];

		return $fields;

	}

	/**
	 * Player speed fields
	 *
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_speed( array $fields, array $options ): array {

		$fields[ 'divider_speed' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_speed' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Speed button', 'speaker' ),
			'description'       => esc_html__( 'Show speed button', 'speaker' ),
			'default'           => 'on',
		];

		$default_speed_controls = $options[ 'speed_controls' ] ?? 'off';
		$fields[ 'player_speed_controls' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Speed control positions', 'speaker' ),
			'description'       => esc_html__( 'Speed controls positions for the audio player', 'speaker' ),
			'default'           => $default_speed_controls,
			'options'           => [
				'off'   => __( 'Player layout', 'speaker' ),
				'above' => __( 'Above player', 'speaker' ),
				'on'    => __( 'Below player', 'speaker' )
			]
		];

		$fields[ 'player_icon_speed' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Speed icon', 'speaker' ),
			'description'       => esc_html__( 'Speed icon for wave player', 'speaker' ),
			'default'           => 'speaker/speed.svg',
			'meta'              => self::$icons_meta
		];

		$default_speed_title = $options[ 'speed_title' ] ?? esc_html__( 'Playback speed:', 'speaker' );
		$fields[ 'player_speed_title' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Speed block title', 'speaker' ),
			'description'       => esc_html__( 'Specify the title for speeds section ', 'speaker' ),
			'default'           => $default_speed_title
		];

		$default_speed = $options[ 'speed' ] ?? '0.5, 0.75, 1, 1.25, 1.5';
		$fields[ 'player_speed_list' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Available speed', 'speaker' ),
			'description'       => esc_html__( 'Specify speeds separated by commas. Speed must be in range from 0.1 to 16. Use period for decimal numbers, for example: 1.2, 1.5, 1.75', 'speaker' ),
			'default'           => $default_speed,
		];

		return $fields;

	}

	/**
	 * Player share fields
	 *
	 * @param array $fields
	 *
	 * @return array
	 */
	private static function fields_player_share( array $fields ): array {

		$fields[ 'divider_share' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_share' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Share button', 'speaker' ),
			'description'       => esc_html__( 'Show share button', 'speaker' ),
			'default'           => 'on',
		];

		$fields[ 'share_socials' ] = [
			'type'              => 'chosen',
			'label'             => esc_html__( 'Socials', 'speaker' ),
			'placeholder'       => esc_html__( 'Choose socials', 'speaker' ),
			'description'       => esc_html__( 'Select social networks and instant messengers that will be used in the sharing interface', 'speaker' ),
			'default'           => [ 'facebook', 'twitter', 'reddit' ],
			'options'           => [
				'facebook' => 'Facebook',
				'twitter' => 'X',
				'linkedin' => 'LinkedIn',
				'pinterest' => 'Pinterest',
				'reddit' => 'Reddit',
				'tumblr' => 'Tumblr',
				'whatsapp' => 'WhatsApp',
				'telegram' => 'Telegram',
				'digg' => 'Digg',
				'stumbleupon' => 'StumbleUpon',
				'pocket' => 'Pocket',
				'xing' => 'Xing',
				'email' => 'Email',
				'print' => 'Print',
			],
			'attr'              => [
				'placeholder' => esc_html__( 'Choose socials', 'speaker' ),
				'multiple' => 'multiple',
			]
		];

		$fields[ 'share_type' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Type', 'speaker' ),
			'description'       => esc_html__( 'Choose share interface type', 'speaker' ),
			'default'           => 'none',
			'options'           => [
				'none'      => __( 'None', 'speaker' ),
				'icons'     => __( 'Icons', 'speaker' ),
				'list'      => __( 'List', 'speaker' ),
				'below'     => __( 'Below player', 'speaker' ),
				'above'     => __( 'Above player', 'speaker' )
			],
		];

		$fields[ 'player_icon_share' ] = [
			'type'              => 'icon',
			'label'             => esc_html__( 'Share icon', 'speaker' ),
			'description'       => esc_html__( 'Share icon for wave player', 'speaker' ),
			'default'           => 'speaker/share.svg',
			'meta'              => self::$icons_meta
		];

		$fields[ 'share_label' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Share label', 'speaker' ),
			'description'       => esc_html__( 'Specify the label for share button', 'speaker' ),
			'default'           => esc_html__( 'Share:', 'speaker' )
		];

		$fields[ 'share_title' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Share title', 'speaker' ),
			/* translators: %s is replaced by actual WordPress post title */
			'description'       => esc_html__( 'Title for shared entry. %s is replaced by actual WordPress post title', 'speaker' ),
			/* translators: %s is replaced by actual WordPress post title */
			'default'           => esc_html__( 'Listen to the audio version of the post %s', 'speaker' ),
		];

		return $fields;

	}

	/**
	 * Player initial settings
	 * @param array $fields
	 * @param array $options
	 *
	 * @return array
	 */
	private static function fields_player_initial_settings( array $fields, array $options ): array {

		$fields[ 'divider_initial' ] = [
			'type'              => 'divider',
		];

		$fields[ 'player_autoplay' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Autoplay', 'speaker' ),
			'description'       => esc_html__( 'Autoplay the audio after page load. May not work in some browsers due to Browser Autoplay Policy. More details for ', 'speaker' ) .
			                       '<a href="https://developers.google.com/web/updates/2017/09/autoplay-policy-changes" target="_blank" rel="noreferrer">' . esc_html__( 'WebKit Browsers', 'speaker' ) . '</a>' .
			                       esc_html__( ' and ', 'speaker' ) .
			                       '<a href="https://hacks.mozilla.org/2019/02/firefox-66-to-block-automatically-playing-audible-video-and-audio/" target="_blank" rel="noreferrer">' . esc_html__( 'Firefox', 'speaker' ) . '</a>',
			'default'           => $options[ 'autoplay' ] ?? 'off',
		];

		$fields[ 'player_loop' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Loop', 'speaker' ),
			'description'       => esc_html__( 'Loop the audio playback', 'speaker' ),
			'default'           => $options[ 'loop' ] ?? 'off',
		];

        $fields[ 'ios_unmute' ] = [
            'type'              => 'switcher',
            'label'             => esc_html__( 'iOS unmute', 'speaker' ),
            'description'       => esc_html__( 'Unmute the audio on iOS devices if device mute button is active', 'speaker' ),
            'default'           => $options[ 'iOS_unmute' ] ?? 'off',
        ];

		return $fields;

	}

	/**
	 * Default border radius compatible with Speaker 4-
	 *
	 * @param $options
	 *
	 * @return array
	 */
	private static function default_border_radius( $options ): array {

		$old_player_style = $options[ 'style' ] ?? false;

		switch ( $old_player_style ) {

			case 'speaker-round':
				return [
					'top' => 100,
					'right' => 100,
					'bottom' => 100,
					'left' => 100,
				];

			case 'speaker-rounded':
				return [
					'top' => 8,
					'right' => 8,
					'bottom' => 8,
					'left' => 8,
				];

			case 'speaker-squared':
				return [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
				];

			default:
				return [
					'top' => 20,
					'right' => 20,
					'bottom' => 20,
					'left' => 20,
				];

		}

	}

}
