<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabPostTypes {

	/**
	 * Controls for Design tab.
	 * @return void
	 */
	public static function controls() {

		$tabs = Plugin::get_tabs();
		$fields = array();

        $fields[ 'cpt_support' ] = [
            'type'              => 'custom_type',
            'render'            => [ TabPostTypes::class, 'post_types_chosen' ],
            'label'             => esc_html__( 'Post types', 'speaker' ),
            'description'       => esc_html__( 'Select post types for which the plugin will work.', 'speaker' ),
            'default'           => [ 'post', 'page' ],
            'options'           => self::get_cpt(),
            'attr'              => [
	            'multiple' => 'multiple',
            ]
        ];

		$fields[ 'default_templates' ] = [
			'type'              => 'custom_type',
			'render'            => [ TabPostTypes::class, 'default_templates' ],
			'label'             => esc_html__( 'Speech Templates', 'speaker' ),
			'default'           => '',
		];

		$tabs[ 'post_types' ][ 'fields' ] = $fields;
		Plugin::set_tabs( $tabs );

	}

	/**
	 * Return list of Custom Post Types.
	 *
	 * @param array $cpt Array with posts types to exclude.
	 *
	 * @return array
	 * @access private
	 */
	private static function get_cpt( array $cpt = [] ): array {

		$defaults = [
			'exclude' => [
				'attachment',
				'elementor_library',
				'notification'
			],
		];
		$cpt = array_merge( $defaults, $cpt );

		$post_types_objects = get_post_types(
			[
				'public' => true,
			], 'objects'
		);

		/**
		 * Filters the list of post type objects used by plugin.
		 *
		 * @since 3.0.0
		 *
		 * @param array $post_types_objects List of post type objects used by plugin.
		 **/
		$post_types_objects = apply_filters( 'speaker/post_type_objects', $post_types_objects );

		$cpt['options'] = [];

		foreach ( $post_types_objects as $cpt_slug => $post_type ) {

			if ( in_array( $cpt_slug, $cpt['exclude'], true ) ) {
				continue;
			}

			$cpt['options'][ $cpt_slug ] = $post_type->labels->name;

		}

		return $cpt['options'];

	}

	/**
	 * Render Default Speech Templates field.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	public static function default_templates() {

		/** Prepare options for ST select. */
		$options = MetaBox::get_instance()->get_st_options();

		?>
		<div class="mdc-data-table">
			<table id="mdp-custom-posts-tbl" class="mdc-data-table__table" aria-label="<?php esc_attr_e( 'Default Speech Templates', 'speaker' ); ?>">
				<thead>
				<tr class="mdc-data-table__header-row">
					<th class="mdc-data-table__header-cell" role="columnheader" scope="col"><?php esc_html_e( 'Post Type', 'speaker' ); ?></th>
					<th class="mdc-data-table__header-cell" role="columnheader" scope="col"><?php esc_html_e( 'Default Speech Template', 'speaker' ); ?></th>
				</tr>
				</thead>
				<tbody class="mdc-data-table__content">
				<?php foreach ( self::get_cpt() as $key => $post_type ) :
					?>
					<tr class="mdc-data-table__row" data-post-type="<?php esc_attr_e( $key ); ?>">
						<td class="mdp-post-type mdp-sc-name mdc-data-table__cell">
                            <span data-post-type="<?php esc_attr_e( $key ); ?>">
                                <?php esc_html_e( $post_type ); ?>
                                <span>(<?php esc_html_e( $key ); ?>)</span>
                            </span>
						</td>
						<td class="mdp-sc-name mdc-data-table__cell">

							<?php

							/** Return default ST for current post type. */
							$default = MetaBox::get_instance()->get_default_st( $key );

							/** Render Speech Template Select. */
							UI::get_instance()->render_select(
								$options,
								$default, // Selected option.
								esc_html__( 'Speech Template', 'speaker' ),
								'',
								[
									'name' => 'mdp_speaker_default_speech_template_for__' . $key,
									'id' => 'mdp-speaker-default-speech-template-for--' . $key
								] );

							?>

						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>
		</div>
		<?php

	}

	/**
     * Post types chosen
     * Don't replace to regular chosen
     *
	 * @return void
	 */
    public static function post_types_chosen() {

	    $options = Settings::get_instance()->options;
	    $default = $field[ 'default' ] ?? array();
	    $selected = isset( $options[ 'cpt_support' ] ) && $options[ 'cpt_support' ] ? $options[ 'cpt_support' ] : $default;

	    UI::get_instance()->render_chosen(
		    self::get_cpt(),
		    $selected,
		    esc_html__( 'Select post types for which the plugin will work.', 'speaker' ),
		    [
			    'multiple' => 'multiple',
			    'name' => 'mdp_speaker_post_types_settings[cpt_support]',
			    'id' => 'mdp_speaker_post_types_settings_cpt_support'
		    ]
	    );

    }

}


