<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\TabActivation;
use Merkulove\Speaker\Unity\Plugin as Plugin;
use WP_User;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class UserSettings {

	/**
	 * @var UserSettings
	 */
	private static $instance;

	private static $options;

	/**
	 * UserSettings constructor.
	 */
	public function __construct() {

		$options = Settings::get_instance()->options;
		if ( ! SpeakerHelper::is_key_exists() ) { return; }
        if ( ! TabActivation::get_instance()->is_activated() ) { return; }

		self::$options = $options;

        /** Add Options */
		add_action('show_user_profile', [ $this, 'userMetaBirthdayForm' ] ); // editing your own profile
		add_action('edit_user_profile', [ $this, 'userMetaBirthdayForm' ] ); // editing another user

        /** Save user meta */
		add_action('personal_options_update', [ $this, 'userMetaBirthdaySave' ] );
		add_action('edit_user_profile_update', [ $this, 'userMetaBirthdaySave' ] );

		/** Add scripts and styles */
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_styles' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

	}

	/**
     * Add styles
	 * @return void
	 */
    public function enqueue_styles() {

	    $screen = get_current_screen();
	    if ( $screen === null ) { return; }
	    if ( ! in_array( $screen->base, [ 'user-edit', 'profile' ] ) ) { return; }

        wp_enqueue_style( 'mdp-speaker-admin-user-edit', Plugin::get_url() . 'css/speaker-admin-user-edit' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

    }

	/**
     * Add scripts
	 * @return void
	 */
    public function enqueue_scripts() {

	    $screen = get_current_screen();

	    if ( $screen === null ) { return; }
	    if ( ! in_array( $screen->base, [ 'user-edit', 'profile' ] ) ) { return; }

        wp_enqueue_script( 'mdp-speaker-admin-user-edit', Plugin::get_url() . 'js/speaker-admin-user-edit' . Plugin::get_suffix() . '.js', array(), Plugin::get_version() );

		wp_localize_script(
			'mdp-speaker-admin-user-edit',
			'mdpSpeakerAdminPost',
			[
				'metaboxOptions' => [
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce' => wp_create_nonce( 'speaker-nonce-metabox' ),
					'pluginUrl' => Plugin::get_url(),
				],
				'translations' => [

					'play_rate' => __( 'The ratio of clicks on the play button to the total number of visitors', 'speaker' ),
					'time_average' => __( 'The average number of seconds played', 'speaker' ),
					'init' => __( 'The number of times the player was initialized', 'speaker' ),
					'play' => __( 'The number of times the play button was clicked', 'speaker' ),
					'pause' => __( 'The number of times the pause button was clicked', 'speaker' ),
					'time' => __( 'The total number of seconds played', 'speaker' ),
					'bounce_rate' => __( 'The ratio of visitors who did not listen to the end of the audio', 'speaker' ),

					'noVoices' => __( 'No one found', 'speaker' ),
					'placeholderValue' => __( 'Choose an option', 'speaker' ),
					'searchPlaceholderValue' => __( 'Language or locale code', 'speaker' ),
					'loadingText' => __( 'Loading...', 'speaker' ),
					'noResultsText' => __( 'No results found', 'speaker' ),
					'noChoicesText' => __( 'No choices to choose from', 'speaker' ),
					'addItemText' => __( 'Press Enter', 'speaker' ),
					'voiceSelectLabel' => __( 'Post voice', 'speaker' ),

					'error' => __( 'Error', 'speaker' ),
					'done' => __( 'Done!', 'speaker' ),
					'requestError' => __( 'Request error', 'speaker' ),
					'requestAborted' => __( 'Request aborted', 'speaker' ),

					'youCannotDuplicate' => __( 'You cannot duplicate this template', 'speaker' ),

				],
			]
		);

    }

	/**
     * Add settings to user profile
	 * @param WP_User $user
	 */
	public function userMetaBirthdayForm( WP_User $user ) {

		$global_voice = self::$options[ 'language' ];

		$user_voice = get_user_meta( $user->ID, 'mdp_speaker_author_voice', true );
		if ( ! $user_voice ) { $user_voice = $global_voice; }

		// Remove user meta if global voice is selected
		if ( $user_voice === $global_voice ) {
			delete_user_meta( $user->ID, 'mdp_speaker_author_voice' );
		}

        echo wp_sprintf(
            '<section class="speaker-author-voice-section">
                <h2>Speaker</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="mdp_speaker_author_voice">%1$s</label></th>
                        <td>
	                        <div id="mdp-speaker-post-voice" class="choices-wrapper" data-global-voice="%5$s" data-post-voice="%3$s">
	                            <select id="mdp_speaker_author_voice" name="mdp_speaker_author_voice"></select>	
	                        </div>
                            <p class="description">
                            	%2$s
                            	<a id="mdp_speaker_author_voice_reset">%4$s</a>
                            </p>
                        </td>
                    </tr>
                </table>
            </section>',
            esc_html__( 'Author Voice', 'speaker'),
            esc_html__( 'Set custom voice if this user is the author of the post.', 'speaker' ),
	        esc_attr( $user_voice ),
	        esc_html__( 'Reset to global voice' ),
	        esc_attr( $global_voice )
        );

	}

	/**
     * Save user meta
	 * @param $userId
	 */
	public function userMetaBirthdaySave($userId) {

		if ( !current_user_can('edit_user', $userId) ) { return; }

		update_user_meta( $userId, 'mdp_speaker_author_voice', $_REQUEST[ 'mdp_speaker_author_voice' ] );

	}

	/**
	 * @return UserSettings
	 */
	public static function get_instance(): UserSettings{

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
