<?php
/**
 * Speaker - PHP 7
 * Extension for Speaker plugin to support PHP 7.4
 *
 * @package         Speaker - PHP7
 * @encoding        UTF-8
 * @version         1.0.1
 * @copyright       (C) 2018-2024 Merkulove ( https://merku.love/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmytro Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 *
 * @wordpress-plugin
 * Plugin Name: Speaker - PHP7
 * Plugin URI: https://1.envato.market/speaker
 * Description: Extension for Speaker plugin to support PHP 7.4
 * Version: 1.0.0
 * Requires at least: 5.5
 * Requires PHP: 7.4
 * Author: Merkulove
 * Author URI: https://1.envato.market/cc-merkulove
 * License: Envato License https://1.envato.market/KYbje
 * License URI: https://1.envato.market/KYbje
 * Text Domain: speaker
 * Domain Path: /languages
 * Tested up to: 6.8
 * Requires Plugins: speaker
 */

namespace Merkulove;

/** Exit if accessed directly. */
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

class SpeakerPHP7
{

    /** @var string - Plugin name. */
    private static $plugin_name = 'Speaker PHP7';

    /** @var array - WP Die arguments. */
    private static $wp_die_args = [
        'back_link' => true
    ];

    /**
     * On activation hook.
     */
    public static function on_activation()
    {

        $wp_die_title = self::$plugin_name . ' - ' . esc_html('Activation Error');

        if (!current_user_can('activate_plugins')) {
            wp_die(
                wp_sprintf(
                    '<b>%s</b>: %s',
                    self::$plugin_name,
                    esc_html('You do not have sufficient permissions to activate this plugin.')
                ),
                $wp_die_title,
                self::$wp_die_args
            );
        }

        if (!is_plugin_active('speaker/speaker.php')) {
            wp_die(
                wp_sprintf(
                    '<strong>%s</strong>: %s',
                    self::$plugin_name,
                    esc_html('This plugin requires Speaker plugin to be installed and activated. Please install and activate Speaker plugin first.')
                ),
                $wp_die_title,
                self::$wp_die_args
            );
        }

        if (version_compare(PHP_VERSION, '7.4', '<')) {
            wp_die(
                wp_sprintf(
                    '<strong>%s</strong>: %s',
                    self::$plugin_name,
                    esc_html('This plugin requires PHP 7.4 or higher. Please update your PHP first')
                ),
                $wp_die_title,
                self::$wp_die_args
            );
        }

        if (version_compare(get_bloginfo('version'), '5.5', '<')) {
            wp_die(
                wp_sprintf(
                    '<strong>%s</strong>: %s',
                    self::$plugin_name,
                    esc_html('This plugin requires WordPress 5.5 or higher. Please update your WordPress first')
                ),
                $wp_die_title,
                self::$wp_die_args
            );
        }

    }

    /**
     * On deactivation hook.
     */
    public static function on_deactivation()
    {
        if (!current_user_can('activate_plugins')) {
            wp_die(
                wp_sprintf(
                    '<strong>%s</strong>: %s',
                    self::$plugin_name,
                    esc_html('You do not have sufficient permissions to deactivate this plugin.')
                ),
                self::$plugin_name . ' - ' . esc_html('Deactivation Error'),
                self::$wp_die_args
            );
        }
    }

    public static function on_plugins_loaded()
    {

        add_action('init', function () {


            if (!function_exists('is_plugin_active')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            /**
             * Critical checks + deactivation
             */

            if (!is_plugin_active('speaker/speaker.php')) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-error is-dismissible"><p><strong>%s</strong>: %s</p></div>',
                        self::$plugin_name,
                        esc_html('This plugin requires Speaker plugin to be installed and activated. Please install and activate Speaker plugin first.'),
                    );
                });
                deactivate_plugins(plugin_basename(__FILE__));
                return;
            }

            if (version_compare(PHP_VERSION, '7.4', '<')) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-error is-dismissible"><p><strong>%s</strong>: %s %s. %s</p></div>',
                        self::$plugin_name,
                        esc_html('This plugin requires PHP 7.4 or higher. Your current PHP version is'),
                        esc_attr(PHP_VERSION),
                        esc_html('Please update your PHP first.'),
                    );
                });
                deactivate_plugins(plugin_basename(__FILE__));
                return;
            }

            /**
             * Important Optional checks without deactivation
             */

            if (version_compare(PHP_VERSION, '8.2', '>')) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-warning is-dismissible"><p><strong>%s</strong>: %s %s. %s</p></div>',
                        self::$plugin_name,
                        esc_html('This plugin is not compatible with PHP'),
                        esc_attr(PHP_VERSION),
                        esc_html('The Speaker 4+ by default is compatible with PHP 8.2 or higher.'),
                    );
                });
            }

            if (version_compare(get_bloginfo('version'), '5.5', '<')) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-warning is-dismissible"><p><strong>%s</strong>: %s %s. %s</p></div>',
                        self::$plugin_name,
                        esc_html('This plugin requires WordPress 5.5 or higher. Your current WordPress version is'),
                        esc_attr(get_bloginfo('version')),
                        esc_html('Please update your WordPress first.'),
                    );
                });
            }

            if (version_compare(get_plugin_data(WP_PLUGIN_DIR . '/speaker/speaker.php')['Version'], '4.0.0', '<')) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-warning is-dismissible"><p><strong>%s</strong>: %s %s. %s</p></div>',
                        self::$plugin_name,
                        esc_html('This plugin requires Speaker plugin version 4.0.0 or higher. Your current Speaker plugin version is'),
                        esc_attr(get_plugin_data(WP_PLUGIN_DIR . '/speaker/speaker.php')['Version']),
                        esc_html('Please update your Speaker plugin first.'),
                    );
                });
            }

        });

    }

}

/** Register activation and deactivation hooks. */
register_activation_hook(__FILE__, [SpeakerPHP7::class, 'on_activation']);
register_deactivation_hook(__FILE__, [SpeakerPHP7::class, 'on_deactivation']);

/** Run Plugin class once after activated plugins have loaded. */
add_action('plugins_loaded', [SpeakerPHP7::class, 'on_plugins_loaded']);
