<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class Analytics {

	/**
	 * The one true Analytics.
	 * @var Analytics
	 **/
	private static $instance;

	/**
	 * Sets up a new Analytics instance.
	 * @access public
	 **/
	private function __construct() {

		$options = Settings::get_instance()->options;
		if ( $options[ 'analytics' ] !== 'on' ) { return; }

		add_action( 'wp_ajax_nopriv_speaker_analytics', [ $this, 'speaker_analytics' ] );
		add_action( 'wp_ajax_speaker_analytics', [ $this, 'speaker_analytics' ] );

	}

	/**
	 * Handler for ajax request.
	 * @return void
	 */
	public function speaker_analytics() {

		check_ajax_referer( 'mdp_speaker_analytics', 'nonce' );

		$post_id = isset( $_POST['id'] ) ? intval( $_POST['id'] ) : 0;
		$event = isset( $_POST['event'] ) ? sanitize_text_field( $_POST['event'] ) : '';
		$time = isset( $_POST['time'] ) ? intval( $_POST['time'] ) : 0;

		switch ( $event ) {

			case 'init':
				$init_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_init', true );
				update_post_meta( $post_id, 'mdp_speaker_analytics_init', $init_counter ? $init_counter + 1 : 1 );
				break;

			case 'play':
				$play_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_play', true );
				update_post_meta( $post_id, 'mdp_speaker_analytics_play', $play_counter ? $play_counter + 1 : 1 );
				break;

			case 'pause':
				$pause_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_pause', true );
				update_post_meta( $post_id, 'mdp_speaker_analytics_pause', $pause_counter ? $pause_counter + 1 : 1 );
				break;

			case 'time':
				$time_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_time', true );
				update_post_meta( $post_id, 'mdp_speaker_analytics_time', $time_counter ? $time_counter + $time : $time );
				break;

			case 'end':
				$end_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_end', true );
				update_post_meta( $post_id, 'mdp_speaker_analytics_end', $end_counter ? $end_counter + 1 : 1 );
				break;

			default:
				break;

		}

		wp_die();

	}

	/**
	 * Calculate analytics for post.
	 *
	 * @param int $post_id
	 *
	 * @return array
	 */
	private function calculate_analytics( int $post_id ): array {

		// Raw data.
		$init_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_init', true ) ?: 0;
		$play_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_play', true ) ?: 0;
		$pause_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_pause', true ) ?: 0;
		$end_counter = get_post_meta( $post_id, 'mdp_speaker_analytics_end', true ) ?: 0;
		$time = get_post_meta( $post_id, 'mdp_speaker_analytics_time', true ) ?: 0;

		// Calculated data.
		$play_rate = $init_counter !== 0 ? round( $play_counter / $init_counter, 2 ) * 100 : 0;
		$average_time = $play_counter !== 0 ? round( $time / $play_counter, 2 ) : 0;
		$bounce_rate = $play_counter !== 0 ? 100 - round( ( $end_counter / $play_counter ) * 100 ) : 0;

		return [
			'init' => $init_counter,
			'play' => $play_counter,
			'pause' => $pause_counter,
			'end' => $end_counter,
			'time' => $time,
			'play_rate' => $play_rate,
			'average_time' => $average_time,
			'bounce_rate' => $bounce_rate,
		];

	}

	/**
	 * Analytics box for metabox.
	 * @return void
	 */
	public function analytics_metabox() {

		$options = Settings::get_instance()->options;
		if ( $options[ 'analytics' ] !== 'on' ) { return; }
		if ( $options[ 'analytics_metabox' ] !== 'on' ) { return; }

		$a = $this->calculate_analytics( get_the_ID() );

		echo wp_sprintf(
			'<div class="speaker-analytics">
				<div class="speaker-analytics-group">
					<div class="speaker-analytics--play-rate">
						<span class="speaker-analytics-count">%1$s</span>
						<span class="speaker-analytics-label">%2$s<span class="dashicons dashicons-info-outline"></span></span>
					</div>
					<div class="speaker-analytics--time-average">
						<span class="speaker-analytics-count">%3$s</span>
						<span class="speaker-analytics-label">%4$s<span class="dashicons dashicons-info-outline"></span></span>
					</div>
				</div>
				<div class="speaker-analytics-accordion-content" style="display: none">
					<div class="speaker-analytics-group">
						<div class="speaker-analytics--time">
							<span class="speaker-analytics-count">%11$s</span>
							<span class="speaker-analytics-label">%12$s<span class="dashicons dashicons-info-outline"></span></span>
						</div>
						<div class="speaker-analytics--bounce-rate">
							<span class="speaker-analytics-count">%13$s</span>
							<span class="speaker-analytics-label">%14$s<span class="dashicons dashicons-info-outline"></span></span>
						</div>
					</div>
	
					<div class="speaker-analytics-group">
						<div class="speaker-analytics--init">
							<span class="speaker-analytics-count">%5$s</span>
							<span class="speaker-analytics-label">%6$s<span class="dashicons dashicons-info-outline"></span></span>
						</div>
						<div class="speaker-analytics--play">
							<span class="speaker-analytics-count">%7$s</span>
							<span class="speaker-analytics-label">%8$s<span class="dashicons dashicons-info-outline"></span></span>
						</div>
						<div class="speaker-analytics--pause">
							<span class="speaker-analytics-count">%9$s</span>
							<span class="speaker-analytics-label">%10$s<span class="dashicons dashicons-info-outline"></span></span>
						</div>
					</div>
				</div>
				
			</div>',
			esc_attr( $a[ 'play_rate' ] ) . '%',
			esc_html__( 'Playback ratio', 'speaker' ),
			esc_attr( self::timeConverter( $a[ 'average_time' ] ) ),
			esc_html__( 'Average played', 'speaker' ),
			esc_attr( self::number_converter( $a[ 'init' ] ) ),
			esc_html__( 'Ready', 'speaker' ),
			esc_attr( self::number_converter( $a[ 'play' ] ) ),
			esc_html__( 'Play', 'speaker' ),
			esc_attr( self::number_converter( $a[ 'pause' ] ) ),
			esc_html__( 'Pause', 'speaker' ),
			esc_attr( self::timeConverter( $a[ 'time' ] ) ),
			esc_html__( 'Total played', 'speaker' ),
			esc_attr( $a[ 'bounce_rate' ] ) . '%',
			esc_html__( 'Bounce rate', 'speaker' ),
		);

	}

	/**
	 * Analytics column
	 *
	 * @param $post_id
	 *
	 * @return void
	 */
	public function analytics_column( $post_id ) {

		$options = Settings::get_instance()->options;
		if ( $options[ 'analytics' ] !== 'on' ) { return; }
		if ( $options[ 'analytics_column' ] !== 'on' ) { return; }

		/** Show player if we have audio. */
		$f_time = SpeakerHelper::is_multipage( $post_id ) ? AudioFile::audio_exists( $post_id, 1 ) : AudioFile::audio_exists( $post_id );
		if ( ! $f_time ) { return; }

		$column_markup = [];
		$a = $this->calculate_analytics( $post_id );
		$analytics_data_types = $options[ 'analytics_column_data' ] ?? array();
		foreach ( $analytics_data_types as $type ) {

			switch ( $type ) {

				case 'play_rate':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Playback ratio', 'speaker' ),
						esc_attr( $a[ 'play_rate' ] ) . '%'
					);
					break;

				case 'time_average':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Average played', 'speaker' ),
						esc_attr( self::timeConverter( $a[ 'average_time' ] ) )
					);
					break;

				case 'init':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Ready', 'speaker' ),
						esc_attr( self::number_converter( $a[ 'init' ] ) )
					);
					break;

				case 'play':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Played', 'speaker' ),
						esc_attr( self::number_converter( $a[ 'play' ] ) )
					);
					break;

				case 'pause':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Paused', 'speaker' ),
						esc_attr( self::number_converter( $a[ 'pause' ] ) )
					);
					break;

				case 'time':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Total played', 'speaker' ),
						esc_attr( self::timeConverter( $a[ 'time' ] ) )
					);
					break;

				case 'bounce_rate':

					$column_markup[] = wp_sprintf(
						'<span class="speaker-analytics-data %s" aria-label="%s"><span>%s</span></span>',
						esc_attr( str_replace( '_', '-', $type ) ),
						esc_html__( 'Bounce rate', 'speaker' ),
						esc_attr( $a[ 'bounce_rate' ] ) . '%'
					);
					break;

			}

		}

		echo wp_sprintf(
			'<div class="speaker-analytics-column">%s</div>',
			implode( '', $column_markup )
		);

	}

	/**
	 * Convert time to human-readable format.
	 * @param $time
	 *
	 * @return false|string
	 */
	private static function timeConverter( $time ) {

		if ( $time > 99 * 60 ) {

			$hours = $time / 3600;

			if ( $hours > 999999 ) {

				$hours = $hours / 1000000;
				$time_converted = number_format( (float)$hours, 1, '.', '') . 'M ' . __('hours', 'speaker');

			} else if ( $hours > 999 ) {

				$hours = $hours / 1000;
				$time_converted = number_format( (float)$hours, 1, '.', '') . 'k ' . __('hours', 'speaker');

			} else {

				$time_converted = (int)( $hours ) . ':' . gmdate( 'i:s', $time );

			}

		} else if ( $time > 99 ) {

			$time_converted = gmdate( 'i:s', $time );

		} else {

			$time_converted = $time . __( 'sec.', 'speaker' );

		}

		return $time_converted;

	}

	/**
	 * Convert number to human-readable format.
	 *
	 * @param $number
	 *
	 * @return mixed|string
	 */
	private function number_converter( $number ) {

		if ( $number > 999999 ) {

			$number = $number / 1000000;
			$number_converted = number_format( (float)$number, 1, '.', '') . 'M';

		} else if ( $number > 999 ) {

			$number = $number / 1000;
			$number_converted = number_format( (float)$number, 1, '.', '') . 'k';

		} else {

			$number_converted = $number;

		}

		return $number_converted;

	}

	/**
	 * Main Analytics Instance.
	 * @return Analytics
	 */
	public static function get_instance(): Analytics {

		/** @noinspection SelfClassReferencingInspection */
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Analytics ) ) {

			self::$instance = new Analytics;

		}

		return self::$instance;

	}

}
