<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Elementor;
use Merkulove\Speaker\Unity\WPBakery;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class Caster {

	/**
     * @access private
	 * @var Caster
	 **/
	private static $instance;

    /**
     * Set up the plugin.
     *
     * @return void
     **/
    public function setup() {

	    /** Includes the autoloader for libraries installed with Composer. */
        if ( SpeakerHelper::is_key_exists() ) {
            require_once SpeakerHelper::get_vendor_path();
        }

		/** Apply WordPress plugin filters */
		Filters::get_instance();

		/** Make WordPress plugin actions */
		Actions::get_instance();

        /** Define hooks that runs on both the front-end and the dashboard. */
        $this->both_hooks();

        /** Define public hooks. */
        $this->public_hooks();

        /** Define admin hooks. */
        $this->admin_hooks();

    }

    /**
     * Define hooks that runs on both the front-end and the dashboard.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function both_hooks() {

	    /** Add RSS feeds */
	    RSS::get_instance();

		/** Speaker analytics */
		Analytics::get_instance();

	    /** Adds all the necessary shortcodes. */
	    Shortcodes::get_instance();

		/** Load Speaker Elementor addon. */
	    Elementor::get_instance()->setup();

		/** Load Speaker WPBakery addon. */
	    WPBakery::get_instance()->setup();

    }

	/**
	 * Reset API Key on fatal error.
	 *
	 * @since  3.0.0
	 * @access public
	 * @return void
	 **/
	public function reset_api_key( $message = null ) {

		/** Remove API Key. */
		$options = get_option( 'mdp_speaker_general_settings' );
        if ( ! isset( $options['dnd-api-key'] ) ) { return; }

        $options['dnd-api-key'] = '';

        /** Save new value. */
        update_option( 'mdp_speaker_general_settings', $options );

        /** Go to first tab. */
        $redirect_url = admin_url( '/admin.php?page=mdp_speaker_settings' );
        if ( $message ) {
            $redirect_url .= '&message=' . urlencode( $message ) . '&speaker_error=true';
        }

        wp_redirect( $redirect_url );
        exit;

	}

    /**
     * Register all the hooks related to the public-facing functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function public_hooks() {

        /** Work only on frontend area. */
        if ( is_admin() ) { return; }

	    /** Load CSS for Frontend Area. */
	    FrontStyles::get_instance();

	    /** Load JavaScripts for Frontend Area. */
	    FrontScripts::get_instance();

	    /** Add player code to page. */
	    PlayerPositions::get_instance();

	    /** Speaker use custom page template to parse content without garbage. */
	    add_filter( 'template_include', [ SpeakerCaster::class, 'speaker_page_template'], PHP_INT_MAX );

	    /** Hide admin bar for Speech Template Editor. */
	    SpeakerCaster::hide_admin_bar();

	    /** Add Schema markup */
	    add_action( 'wp_head', [ SpeakerCaster::get_instance(), 'structured_data' ] );

	    /** Add accessibility elements for audio shortcode */
	    add_action( 'wp_audio_shortcode', [ SpeakerCaster::get_instance(), 'audio_shortcode_filter' ] );

    }

    /**
     * Register all of the hooks related to the admin area functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function admin_hooks() {

        /** Work only in admin area. */
        if ( ! is_admin() ) { return; }

	    /** Reset API Key on fatal error. */
	    if ( isset( $_GET['reset-api-key'] ) && $_GET['reset-api-key'] ) {
		    $this->reset_api_key();
	    }

		/** Notices for admin area */
		Notice::get_instance();

	    /** Create folder for audio files. */
	    $this->create_speaker_folder();

	    /** Add Ajax handlers and before_delete_post action. */
	    SpeakerCaster::get_instance()->add_actions();

	    /** Dashboard widgets for Speaker */
	    DashboardWidget::get_instance();

	    /** Add Meta Box for selected post types. */
	    MetaBox::get_instance();

	    /** Add admin styles. */
	    AdminStyles::get_instance();

	    /** Add admin javascript. */
	    AdminScripts::get_instance();

	    /** Catch Google Drive authorization code */
	    StorageGoogle::get_instance();

		/** Speech Templates */
		SpeechTemplates::get_instance();

		/** Add columns and actions for post list */
		PostList::get_instance();

		/** Add hooks and actions for post edit */
		PostEdit::get_instance();

	    /** Add settings for user */
	    UserSettings::get_instance();

    }

	/**
	 * Create folder for audio files.
	 *
	 * @since 3.0.0
	 * @access private
	 * @return void
	 **/
	public function create_speaker_folder() {

		/** Create /wp-content/uploads/speaker/ folder. */
		wp_mkdir_p( trailingslashit( wp_upload_dir()['basedir'] ) . 'speaker' );

	}

	/**
	 * This method used in register_activation_hook
	 * Everything written here will be done during plugin activation.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function activation_hook() {

		/** Activation hook */

	}



	/**
	 * Main Caster Instance.
	 * Insures that only one instance of Caster exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Caster
	 **/
	public static function get_instance(): Caster {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
