<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Google\ApiCore\ApiException;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class MetaBoxOptions {

	/**
	 * The one true MetaBoxOptions.
	 *
	 * @var MetaBoxOptions
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new MetaBoxOptions instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct() {

		add_action( 'wp_ajax_speaker_metabox_options', [ $this, 'speaker_metabox_options' ] );

	}

    public function speaker_metabox_options() {

//        check_ajax_referer( 'speaker-nonce-metabox' );

	    if ( ! empty( $_GET ) ) {

			wp_send_json( $this->process_get_request() );

	    } else if ( ! empty( $_POST ) ) {

			$result = $this->process_post_request();
			wp_send_json(
				[
					'status' => $result !== false ? 'success' : 'error',
					'message' => $result !== false ?
						esc_html__( 'OK', 'speaker' ) :
						__( 'Something went wrong. Please try again later.', 'speaker' ),
				]
			);

		}

        wp_die();

    }

	private function process_get_request() {

		$type = $_GET['type'] ?? '';
		$type = sanitize_text_field( $type );

		$key = $_GET['key'] ?? '';
		$key = sanitize_text_field( $key );

		switch ( $type ) {

			case 'metadata':

				break;

			case 'option':

				break;

			case 'custom':

				if ( $key === 'voices-list' ) {

					$voices = get_transient( 'speaker_list_voices' );

					$rendered_voices = [];
					foreach ( $voices as $voice ) :

						/** Skip already rendered voices. */
						if ( in_array( $voice->getName(), $rendered_voices ) ) {
							continue;
						} else {
							$rendered_voices[] = [
								'voice' => $voice->getName(),
								'language' => $voice->getLanguageCodes()[0],
								'gender' => TabVoice::get_voice_gender( $voice ),
							];
						}

					endforeach;

					return $rendered_voices;

				}

				if ( $key === 'lang-list' ) {

					return Language::$languages;

				}

				break;

			default:

				break;

		}

	}

	private static function process_post_request() {

		$post_id = sanitize_text_field( $_POST['post_id'] ?? '' );
		$type = sanitize_text_field( $_POST['type'] ?? '' );
		$key = sanitize_text_field( $_POST['key'] ?? '' );
		$value = sanitize_text_field( $_POST['value'] ?? '' );

		if ( $key === '' ) { return ''; }

		$update_status = false;
		switch ( $type ) {

			case 'metadata':

				$update_status = update_post_meta( (int)$post_id, $key, $value );

				break;

			case 'custom':

				if ( $key !== 'mdp_speaker_speech_templates' ) { break; }

				$templates = get_option( $key, [] );
				foreach ( $templates as $template ) {

					if ( $template[ 'id' ] !== $value ) { continue; }

					$duplicated_template = $template;
					$duplicated_template[ 'id' ] = uniqid();
					$duplicated_template[ 'name' ] .= ' ' . __( 'Copy', 'speaker' );

					$templates[] = $duplicated_template;
					$update_status = update_option( $key, $templates );
					break;

				}
				break;

			default:

				break;

		}

		return $update_status;

	}


	/**
	 * Main MetaBoxOptions Instance.
	 *
	 * @return MetaBoxOptions
	 **/
	public static function get_instance(): MetaBoxOptions {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
