<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

final class Player {

	/**
	 * Check conditions and show player
	 *
	 * @param int $id
	 * @param bool $ob
	 *
	 * @return false|string
	 */
    public static function the_player( int $id = 0, bool $ob = true ) {

        if ( ! $id ) return false;

        /** Get page index */
	    $page_index = $_GET[ 'page' ] ?? 0;
        if ( ! $page_index && SpeakerHelper::is_multipage( $id ) ) { $page_index = 1; }

        /** Show player if we have audio. */
        if ( ! AudioFile::audio_exists( $id, $page_index ) ) { return false; }

        /** Don't show player if we parse content. */
        if ( isset( $_GET['speaker-ssml'] ) || isset( $_GET['speaker-template'] ) ) { return false; }

        /** Don't show player if in Speech Template Editor. */
        if ( isset( $_GET['speaker_speech_template'] ) && '1' === $_GET['speaker_speech_template'] ) { return false; }

	    /** Render or return player */
        if ( $ob ) {

	        ob_start();
            echo self::player_markup( $id, $page_index );
	        return ob_get_clean();

        } else {

            return self::player_markup( $id, $page_index );

        }

    }

	/**
     * Front-end Player HTML markup
	 * @param int $id
	 * @param int $page_index
	 *
	 * @return mixed|null
	 */
    private static function player_markup( int $id, int $page_index = 0 ) {

	    /** Options shorthand */
	    $options = Settings::get_instance()->options;

	    /** Prepare variables for render */
	    $post = get_post( $id );

	    $audio_from_meta = get_post_meta( $id, 'mdp_speaker_post_audio', true );
	    $audio_url = $audio_from_meta && $audio_from_meta !== '' ? $audio_from_meta : SpeakerCaster::get_instance()->get_audio_url( $id, $page_index );
	    $loop = $options[ 'player_loop' ] === 'on' ? 'loop="on" ' : ' ';
	    $autoplay = $options[ 'player_autoplay' ] === 'on' ? 'autoplay="on" ' : ' ';
	    $preload = $options[ 'player_preload' ] !== 'backend' ? ' preload="' . $options[ 'player_preload' ] . '"' : '';

		$player_placeholder = '<div class="mdp-speaker-wavesurfer-player mdp-speaker-wavesurfer-placeholder"></div>';

	    /** @noinspection HtmlUnknownAttribute */
	    $player_markup = wp_sprintf( '<div class="mdp-speaker-wrapper">
            %1$s
            %2$s
            %3$s
            %4$s
            <div class="mdp-speaker-box %5$s">
                <div class="mdp-speaker-player" %6$s %7$s>
                    %8$s
                </div>
            </div>
            %9$s
            %10$s
            %11$s
            %12$s
			</div>',
		    $options[ 'before_player_switcher' ] === 'on' ? wp_kses_post( $options[ 'before_player' ] ) : '',
		    $options[ 'player_share' ] === 'on' && $options[ 'share_type' ] === 'above' ? self::share_box( $options ) : '',
		    $options[ 'player_download_link' ] === 'frontend-above' && in_array( $options[ 'player_download' ], [ 'on', 'link' ] ) ? self::download_box( $audio_url, $id, $post ) : '',
		    $options[ 'player_speed_controls' ] === 'above' ? self::speed_bar_markup( $options ) : '',
		    esc_attr( self::classes( $options ) ),
		    self::data_attributes( $options, $id, $post, $audio_url ),
		    self::inline_style( $options, $id ),
		    $options[ 'player_style' ] !== 'speaker-wavesurfer' ?
			    do_shortcode( '[audio src="' . $audio_url . '" ' . $autoplay . $loop . $preload . ']' ) : $player_placeholder,
		    $options[ 'player_speed_controls' ] === 'on' ? self::speed_bar_markup( $options ) : '',
		    $options[ 'player_download_link' ] === 'frontend'  && in_array( $options[ 'player_download' ], [ 'on', 'link' ] ) ? self::download_box( $audio_url, $id, $post ) : '',
		    $options[ 'player_share' ] === 'on' && $options[ 'share_type' ] === 'below' ? self::share_box( $options ) : '',
		    $options[ 'after_player_switcher' ] === 'on' ? wp_kses_post( $options[ 'after_player' ] ) : ''
	    );

        return apply_filters( 'speaker_frontend_player', $player_markup, $id );

    }

	/**
     * Speed bar
	 *
	 * @param $options
	 *
	 * @return string
	 */
    private static function speed_bar_markup( $options ): string {

		// Don't show speed bar if we have only one speed
		if ( $options[ 'player_speed' ] !== 'on' ) {
			return '';
		}

        $speed_title = ! empty( $options[ 'player_speed_title' ] ) ?
            wp_sprintf(
                '<div class="mdp-speaker--speed-title">
                    <p>%s</p>
                </div>',
                esc_html__( $options[ 'player_speed_title' ], 'speaker' )
            ) : '';

        $speeds = [];
        foreach ( self::get_speeds( $options ) as $speed ) {

	        $title = $speed . 'x ' . esc_html__( 'speed', 'speaker' );

            $speeds[] = wp_sprintf(
                '<button class="mdp-speaker--speed-button" aria-label="%1$s" title="%1$s" data-speed="%2$s">%2$sx</button>',
	            esc_attr__( $title, 'speaker' ),
	            esc_attr( $speed )
            );

        }

        return wp_sprintf(
            '<div class="mdp-speaker--speed %1$s">
                %2$s
                <div class="mdp-speaker--speed-box %1$s">
                    %3$s
                </div>
            </div>',
	        esc_attr( $options['player_style'] ),
	        $speed_title,
            join( '', $speeds )
        );

    }

	/**
     * Data attributes for player
     *
	 * @param $options
	 * @param $id
	 * @param $post
	 * @param $audio_url
	 *
	 * @return mixed|null
	 */
    private static function data_attributes( $options, $id, $post, $audio_url ) {

        $player_type = $options[ 'player_style' ] ?? 'speaker-wavesurfer';
	    $attributes = '';

        switch ( $player_type ) {

            case 'speaker-wavesurfer':
                $attributes .= ' data-title="' . esc_attr( $post->post_title ) . '"';
                $attributes .= ' data-author="' . esc_attr( $post->post_author ) . '"';
                $attributes .= ' data-thumbnail="' . esc_attr( get_the_post_thumbnail_url( $id ) ) . '"';
                break;

            default:
                break;

        }

        $attributes .= ' data-type="' . esc_attr( $player_type ) . '"';
	    $attributes .= ' data-id="' . esc_attr( $id ) . '"';
        $attributes .= ' data-src="' . esc_url( $audio_url ) . '"';
        $attributes .= ' data-download="' . esc_attr( ! in_array( $options[ 'player_download_link' ], [ 'none', 'backend' ] ) ) . '"';

        // Speeds
        if ( $options[ 'player_speed' ] === 'on' ) {
	        $attributes .= ' data-speeds="' . esc_attr( join( ',', self::get_speeds( $options ) ) ) . '"';
        }

        $attributes .= ' data-share="' . esc_attr( self::get_socials( $options ) ) . '"';

        // Length formatted from backend
	    if ( $options[ 'player_preload' ] === 'backend' ) {
		    $attributes .= ' ' . self::preload_backend( $options, $id );
	    }

        return apply_filters( 'speaker_player_data_attributes', $attributes );

    }

	/**
     * Inline style for player
     *
	 * @param $options
	 * @param $id
	 *
	 * @return string
	 */
    private static function inline_style( $options, $id ): string {

        $style = '';
	    if ( in_array( $options[ 'player_thumbnail' ], [ 'background', 'above', 'below' ] ) ) {
		    $style = wp_sprintf(
			    ' style="background: url(%s) no-repeat center center; background-size: cover;"',
			    esc_attr( get_the_post_thumbnail_url( $id ) )
		    );
	    }

        return $style;

    }

	/**
     * Player classes
     *
	 * @param $options
	 *
	 * @return mixed|null
	 */
    private static function classes( $options ) {

        $classes = [
            'mdp-speaker-box',
            esc_attr( $options[ 'player_position' ] ),
            esc_attr( $options[ 'player_style' ] ),
        ];

        // Compatibility with old versions
        if ( in_array( $options[ 'player_style' ], [ 'speaker-round', 'speaker-rounded', 'speaker-squared' ] ) ) {

            $classes[] = 'speaker-mediaelement';

        }

        // Thumbnail position
        $thumbnail = $options[ 'player_thumbnail' ] ?? 'none';
        switch ( $thumbnail ) {

            case 'background':
                $classes[] = 'mdp-speaker-box--thumbnail-background';
                break;

            case 'above':
                $classes[] = 'mdp-speaker-box--thumbnail-above';
                break;

            case 'below':
                $classes[] = 'mdp-speaker-box--thumbnail-below';
                break;

            default:
                break;

        }

        return apply_filters( 'speaker_player_classname', join( ' ', $classes ) );

    }

    /**
     * Preload audio duration from backend
     * @param $options
     * @param $id
     * @return string
     */
    private static function preload_backend( $options, $id ): string {

        // Get audio duration from backend
        $length_formatted = $options[ 'player_preload' ] === 'backend' ?
            SpeakerCaster::get_instance()->get_audio_meta( $id, 'length_formatted' ) : 0;

        return $options[ 'player_preload' ] === 'backend' ?
            wp_sprintf( 'data-length-formatted="%s"', esc_attr( $length_formatted ) ) : '';

    }

	/**
	 * Download box markup
	 *
	 * @param $audio_url
	 * @param $id
	 * @param mixed $post
	 *
	 * @return string
	 */
    private static function download_box( $audio_url, $id, $post = [] ): string {

        if ( empty( $post ) ) { $post = get_post( $id ); }

		$options = Settings::get_instance()->options;

        /** @noinspection HtmlUnknownTarget */
        return wp_sprintf(
            '<div class="mdp-speaker-download-box">
                <p>%4$s: <a href="%1$s" download="" title="%4$s: %2$s">%3$s</a></p>
            </div>',
            esc_url( $audio_url ),
            esc_attr( $post->post_title ),
	        $post->post_title !== '' ? esc_html( $post->post_title ) : esc_html__( 'Audio file', 'speaker' ),
            esc_html( $options[ 'download_label' ] )
        );

    }

	/**
     * Share box markup
	 * @param $options
	 *
	 * @return string
	 */
    private static function share_box( $options ): string {

        $buttons = [];

	    $socials = isset( $options[ 'share_socials' ] ) && is_array( $options[ 'share_socials' ] ) ? $options[ 'share_socials' ] : array();
        foreach ( $socials as $social ) {

            $file_path = Plugin::get_path() . 'images/mdc-icons/speaker/' . esc_attr( $social ) . '.svg';

            if ( !file_exists( $file_path ) ) { return ''; }

            $buttons[] = wp_sprintf(
                '<button data-social="%1$s">%2$s</button>',
                esc_attr( $social ),
                file_get_contents( $file_path ) ?? ''
            );

        }

        return wp_sprintf(
            '<div class="mdp-speaker-share-box %3$s">
                <div class="mdp-speaker-share-box__title">
                    <p>%1$s</p>
                </div>
                <div class="mdp-speaker-share-box__buttons">
                    %2$s
                </div>
            </div>',
            esc_html( $options[ 'share_label' ] ),
            join( '', $buttons ),
            $options[ 'player_style' ]
        );

    }

	/**
	 * Get speeds from settings and validate them
	 *
	 * @param $options
	 *
	 * @return array
	 */
    private static function get_speeds( $options ): array {

	    // Get speeds
	    $speeds = preg_split("/[\s,]+/", $options[ 'player_speed_list' ] );
	    $speeds = array_filter( $speeds );
        $speeds = array_map( 'floatval', $speeds );
        $speeds = array_unique( $speeds );

	    // Validate speeds
        $valid_speeds = [];
	    foreach ( $speeds as $speed ) {
		    if ( $speed <= 0 || $speed > 16 ) { continue; } // Ignore nulls
		    $valid_speeds[] = $speed;
	    }

        return $valid_speeds;

    }

	/**
     * Get socials from settings
     *
	 * @param $options
	 *
	 * @return string
	 */
    private static function get_socials( $options ): string {

        $socials = $options[ 'share_socials' ] ?? array();
        return $options[ 'share_type' ] !== 'none' ? join( ',', $socials ) : '';

    }

}
