<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\TabAssignments;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

final class PlayerPositions {

    /**
     * @var PlayerPositions
     */
    private static $instance;

    /**
     * PlayerPositions constructor.
     */
    private function __construct() {

        $options = Settings::get_instance()->options;

        switch ( $options[ 'player_position' ] ) {

            case 'before-title':
            case 'after-title':
                add_filter( 'the_title', [ $this, 'add_player_to_title' ] );
                break;

            case 'before-content':
            case 'after-content':
            case 'top-fixed':
            case 'bottom-fixed':
                add_filter( 'the_content', [ $this, 'add_player_to_content' ] );
                break;

            case 'before-filter':
            case 'after-filter':
                if ( ! empty( $options[ 'custom_filter' ] ) ):
                    add_filter( $options[ 'custom_filter' ], [ $this, 'add_player_to_custom_filter' ] );
                endif;
                break;

            case 'action':
                if ( ! empty( $options[ 'custom_action' ] ) ):
                    add_action( $options[ 'custom_action' ], [ $this, 'add_player_to_action' ] );
                endif;
                break;

        }

    }

    /**
     * Add player before/after Title.
     *
     * @param $title  - Post/Page title.
     * @return string
     **/
    public function add_player_to_title( $title ): string {

        /** Get plugin settings. */
        $options = Settings::get_instance()->options;

        /** Do staff only for proper title instances */
        if ( ! $this->is_proper_title( $title, $options ) ) { return $title; }

        /** Add player before of after title */
        $position = $options['player_position'];
        $player = Player::the_player( get_the_ID() );

        switch ( $position ) {

            case 'before-title':
                $title = $player . $title;
                break;

            case 'after-title':
                $title = $title . $player;
                break;

            default:
                break;

        }

        return apply_filters( 'speaker-' . $position . '-title', $title );

    }

    /**
     * Add player before/after Content and Top/Bottom Fixed.
     *
     * @param $content - Post/Page content.
     * @access public
     * @return string
     */
    public function add_player_to_content( $content ): string {

        $options = Settings::get_instance()->options;

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return $content; }

        /** Check if we are in the loop and work only with selected post types. */
        if ( in_the_loop() && ! ( is_singular( $options['cpt_support'] ) ) ) { return $content; }

        /** Add player only Once. */
        if ( strpos( $content, 'class="mdp-speaker-wrapper"' ) !== false ) {
            return $content;
        }

	    /**  Add player only if WPBakery addon is not on the page. */
        if ( strpos( $content, 'vce_speaker' ) !== false ) {
			return $content;
	    }

        $player = Player::the_player( get_the_ID() );
        $position = $options['player_position'];

        switch ( $position ) {

            case 'before-content':
                $content = $player . $content;
                break;

            case 'after-content':
            case 'top-fixed':
            case 'bottom-fixed':
                $content = $content . $player;
                break;

            default:
                break;

        }

        return $content;

    }

    /**
     * Add player before/after Custom Filter.
     *
     * @param $content
     * @return mixed|string
     */
    public function add_player_to_custom_filter( $content ) {

        $options = Settings::get_instance()->options;

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return $content; }

        /** Add player only Once. */
        if ( strpos( $content, 'class="mdp-speaker-wrapper"' ) !== false ) {
            return $content;
        }

        $player = Player::the_player( get_the_ID() );
        $position = $options['player_position'];

        switch ( $position ) {

            case 'before-filter':
                $content = $player . $content;
                break;

            case 'after-filter':
                $content = $content . $player;
                break;

            default:
                break;

        }

        return $content;

    }

    /**
     * Add player to Custom Action.
     * @return false|void
     */
    public function add_player_to_action() {

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return false; }

        echo Player::the_player( get_the_ID() );

    }

    /**
     * Checks if title is proper for adding player
     *
     * @param $title
     * @param $options
     * @return bool
     */
    private function is_proper_title( $title, $options ): bool {

        /** Do staff only for proper title instances */
        $in_loop = $options[ 'in_loop' ] === 'on';
        if ( $in_loop ) {
            if ( ! in_the_loop() ) { return false; }
        }

        /** Checks if plugin should work on this page. */
        if ( is_singular( $options[ 'cpt_support' ] ?? array() ) ) {

            // Return original title if hook works on the wrong page title instance.
            $id = get_the_ID();
            $post_title = get_post( $id )->post_title;
            if ( $post_title !== $title ) { return false; }

        }

        if ( is_archive() ) { return false; }

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return false; }

        /** Add player only Once. */
        if ( strpos( $title, 'class="mdp-speaker-wrapper"' ) !== false ) {
            return false;
        }

        return true;

    }

    /**
     * Get the instance of the class.
     *
     * @return PlayerPositions
     */
    public static function get_instance(): PlayerPositions {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof PlayerPositions ) ) {

            self::$instance = new PlayerPositions();

        }

        return self::$instance;

    }

}
