<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class RSS {

    /**
     * @var RSS
     */
	private static $instance;

    /**
     * Add feeds on init
     */
    private function __construct() {

	    $options = Settings::get_instance()->options;
	    if ( $options[ 'podcasts_rss' ] !== 'on' ) { return; }

        $this->add_rss();

        add_filter( 'pre_get_posts', array( $this, 'pre_get_posts' ) );

        if ( $options[ 'podcast_root_file' ] === 'on' ) {
	        add_action( 'speaker_update_custom_rss_file', [ $this, 'update_custom_rss_file' ] );
        }

    }

	/**
     * Copy RSS feed to custom file
	 * @return void
	 */
    public function update_custom_rss_file() {

        // Get the RSS feed
        $rss = wp_remote_get(
            get_feed_link( 'speaker-podcast' ),
            [
                'sslverify' => false
            ]
        );
        if ( is_wp_error( $rss ) ) { return; }

        // Get content of the RSS feed
        $rss_feed = wp_remote_retrieve_body( $rss );

	    // Change the filename to your desired filename
	    $file_base_name = apply_filters( 'speaker_rss_file_base_name', 'podcast' );
	    $filename = str_replace( 'wp-content/plugins/speaker/src/Merkulove/Speaker', '', __DIR__ ) . '/' . esc_attr( $file_base_name ) .'.xml';

	    // Open the file for writing
	    $file = fopen( $filename, 'w' );

	    // Write the content to the file
	    fwrite( $file, $rss_feed );

	    // Close the file
	    fclose( $file );

    }

    /**
     * Add all feeds
     * @return void
     */
    public function add_rss() {

        $options = Settings::get_instance()->options;

        if ( $options[ 'podcasts_rss' ] === 'on' ) {

            add_feed( 'speaker-podcast', [ $this, 'podcasts_rss_feed' ] );

        }

    }

    /**
     * Remove pagination from query
     * @param $query
     * @return mixed
     */
    public function pre_get_posts( $query ) {

        if ( $query->is_main_query() && $query->is_feed( 'google-podcasts' ) ) {

            $query->set( 'nopaging', 1 );

        }

        return $query;

    }

    /**
     * Add Google Podcasts RSS feed
     *
     * IMPORTANT: DON'T TOUCH TABULATION and NEW LINES IN THIS METHOD!!!
     * @return void
     */
    public function podcasts_rss_feed() {

        header( 'Content-Type: ' . feed_content_type( 'rss' ) . '; charset=' . get_option( 'blog_charset' ) );
        status_header( 200 );

        $options = Settings::get_instance()->options;

echo '<?xml version="1.0" encoding="UTF-8"?>';
?>

<rss version="2.0" xmlns:itunes="http://www.itunes.com/dtds/podcast-1.0.dtd" xmlns:podcast="https://podcastindex.org/namespace/1.0" xmlns:atom="http://www.w3.org/2005/Atom" xmlns:content="http://purl.org/rss/1.0/modules/content/">
        <channel>
        <?php
        $header_by_line = preg_split("/\r\n|\n|\r/", RSS::get_instance()->get_podcast_header( $options ) );
        foreach ( $header_by_line as $line ) {
            if ( empty( $line ) ) { continue; }
        ?>
    <?php print $line;?>

        <?php } ?>
    <?php if ( have_posts() ) : while( have_posts()  ) : the_post(); ?>
<?php
$post_id = get_the_ID();

if ( SpeakerHelper::is_multipage( $post_id ) ) {

    for( $page_index = 1; $page_index <= SpeakerHelper::pages_counter( $post_id ); $page_index++ ) {
        if ( ! AudioFile::audio_exists( $post_id, $page_index ) ) { continue; }
        $item_by_line = preg_split("/\r\n|\n|\r/", RSS::get_instance()->get_podcast_item( $options, $post_id, $page_index ) );
        $this->item_xml( $item_by_line );
    }

} else {

    if ( ! AudioFile::audio_exists( $post_id) ) { continue; }
    $item_by_line = preg_split("/\r\n|\n|\r/", RSS::get_instance()->get_podcast_item( $options, $post_id ) );
    $this->item_xml( $item_by_line );

}
endwhile; endif;
?>
</channel>
</rss>
        <?php
        exit;

    }


	/**
     * XML for single item
     *
	 * @param array $item_by_line
	 *
	 * @return void
	 */
    private function item_xml( array $item_by_line = array() ) {

?>
<item>
        <?php foreach ( $item_by_line as $line ) {
        if ( empty( $line ) ) { continue; }
        ?>
            <?php print $line;?>

        <?php } ?>
    </item>
	    <?php

    }

    /**
     * Get podcast header
     * @param $options
     * @return mixed|null
     */
    private function get_podcast_header( $options ) {

        $podcast_header = apply_filters( 'speaker_podcast_header_raw', $options[ 'podcast_header' ] );

        $podcast_header = str_replace(
            [
                '[site-title]',
                '[site-tagline]',
                '[site-url]',
                '[site-lang]',
                '[site-logo]',
                '[site-email]'
            ],
            [
                get_bloginfo( 'name' ),
                get_bloginfo( 'description' ),
                get_bloginfo( 'url' ),
                get_bloginfo( 'language' ),
                $this->get_custom_logo_url(),
                get_bloginfo( 'admin_email' ),
            ],
            $podcast_header );

        return apply_filters( 'speaker_podcast_header', $podcast_header );

    }

	/**
     * Podcast item
     *
	 * @param $options
	 * @param $post_id
	 * @param int $page_index
	 *
	 * @return mixed|null
	 */
    private function get_podcast_item( $options, $post_id, int $page_index = 0 ) {

        $podcast_item = apply_filters( 'speaker_podcast_item_raw', $options[ 'podcast_item' ] );
        $page = $page_index > 0 ? ( ' - ' . esc_html__( 'Page', 'speaker' ) . ' ' . $page_index ) : '';

        $podcast_item = str_replace(
            [
                '[item_id]',
                '[item_title]',
                '[item_excerpt]',
                '[item_date]',
                '[item_url]',
                '[item_type]',
                '[item_length]',
                '[item_duration]',
                '[item_thumbnail]',
                '[item_guid]'
            ],
            [
                $post_id,
                htmlspecialchars( get_the_title( $post_id ) . esc_html( $page ), ENT_QUOTES, 'UTF-8'),
                htmlspecialchars( get_the_excerpt( $post_id ) ?? '', ENT_QUOTES, 'UTF-8'),
                get_post_time( 'D, d M Y H:i:s T', false, $post_id ), // previous format l, d F Y H:i:s T
                explode('?', SpeakerCaster::get_instance()->get_audio_url( $post_id, $page_index ) )[ 0 ],
                SpeakerCaster::get_instance()->get_audio_meta( $post_id, 'mime_type', $page_index ),
                SpeakerCaster::get_instance()->get_audio_meta( $post_id, 'filesize', $page_index ),
                SpeakerCaster::get_instance()->get_audio_meta( $post_id, 'length_formatted', $page_index ),
                get_the_post_thumbnail_url( $post_id ) ?? '',
                $this->get_guid( $post_id )
            ],
            $podcast_item );

        return apply_filters( 'speaker_podcast_item', $podcast_item );

    }

    /**
     * Get post guid
     * @return mixed|null
     */
    private function get_guid( $post_id ) {

        $site_uid = str_replace( '/', '', get_bloginfo( 'url' ) );
        $site_uid = str_replace( 'https:', '', $site_uid );
        $site_uid = str_replace( 'http:', '', $site_uid );
        $guid = 'speaker' . '-' . $site_uid . '-' . $post_id;

        return apply_filters( 'speaker_podcast_guid', $guid );

    }

    /**
     * Get custom logo url from customizer
     * @return mixed|string
     */
    private function get_custom_logo_url() {
        $custom_logo_id = get_theme_mod( 'custom_logo' );
        if ( ! $custom_logo_id ) { return ''; }

        $image = wp_get_attachment_image_src( $custom_logo_id , 'full' );
        return $image[0];
    }

	/**
	 * Main RSS Instance.
	 *
     * @return RSS
	 **/
	public static function get_instance(): RSS {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof RSS ) ) {

            self::$instance = new RSS;

		}

		return self::$instance;

	}

}
