<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\TabActivation;
use Merkulove\Speaker\Unity\TabAssignments;

/**
 * SINGLETON: Class used to implement shortcodes.
 *
 * @since 2.0.0
 *
 **/
final class Shortcodes {

	/**
	 * The one true Shortcodes.
	 *
	 * @var Shortcodes
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new Shortcodes instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Initializes shortcodes. */
		$this->shortcodes_init();

	}

	/**
	 * Initializes shortcodes.
	 *
	 * @since 2.0.0
	 * @access public
	 * @return void
	 **/
	public function shortcodes_init() {

		/** Add player by shortcode [speaker] or [speaker id=POST_ID] */
		add_shortcode( 'speaker', [ $this, 'speaker_shortcode' ] );

		/** Speaker Mute Shortcode. [speaker-mute]...[/speaker-mute] */
		add_shortcode( 'speaker-mute', [ $this, 'speaker_mute_shortcode' ] );

		/** Speaker Break Shortcode. [speaker-break time="2s"] */
		add_shortcode( 'speaker-break', [ $this, 'speaker_break_shortcode' ] );

		/** SSML shortcodes */
        if ( TabActivation::get_instance()->is_activated() ) {

			$this->ssml_shortcodes_init(); //TODO: SpeakerUtilities

		}

	}

	/**
	 * Initializes shortcodes.
	 *
	 * @since 3.0.0
	 * @access public
	 * @return void
	 **/
	public function ssml_shortcodes_init() {

		/** Speaker say‑as Shortcode. [speaker-say‑as interpret-as="cardinal"][/speaker-say‑as] */
		add_shortcode( 'speaker-say‑as', [ $this, 'speaker_say_as_shortcode' ] );

		/** Speaker sub Shortcode. [speaker-sub alias="World Wide Web Consortium"]W3C[/speaker-sub] */
		add_shortcode( 'speaker-sub', [ $this, 'speaker_sub_shortcode' ] );

		/** Speaker emphasis Shortcode. [speaker-emphasis level="moderate"]This is an important announcement.[/speaker-emphasis] */
		add_shortcode( 'speaker-emphasis', [ $this, 'speaker_emphasis_shortcode' ] );

		/** Speaker voice Shortcode. [speaker-voice name="en-GB-Wavenet-C"]I am not a real human.[/speaker-voice] */
		add_shortcode( 'speaker-voice', [ $this, 'speaker_voice_shortcode' ] );

		/** Link to audio file Shortcode [speaker-file]. */
		add_shortcode( 'speaker-file', [ $this, 'speaker_file' ] );

		/** Shortcode to say but not show content [speaker-say]Some Content.[/speaker-say]. */
		add_shortcode( 'speaker-say', [ $this, 'speaker_say' ] );

		/** Shortcode used to customize the pitch, speaking rate, and volume of text contained by the element. [speaker-prosody rate="" pitch="" volume=""]Some Content.[/speaker-prosody]. */
		add_shortcode( 'speaker-prosody', [ $this, 'speaker_prosody' ] );

		/** Speaker phoneme Shortcode [speaker-phoneme alphabet="ipa" ph="təˈmeɪ.ɾoʊ"]tomato[/speaker-phoneme] */
		add_shortcode( 'speaker-phoneme', [ $this, 'speaker_phoneme_shortcode' ] );

	}

	/**
	 * Add Speaker break by shortcode [speaker-break time="300ms"].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @return string
	 * @since 2.0.0
	 * @access public
	 **/
	public function speaker_break_shortcode( $atts ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'time' => '500ms',
			'strength' => 'medium'
		], $atts );

		/** Extra protection from the fools */
		$atts['time'] = trim( strip_tags( $atts['time'] ) );
		$atts['strength'] = trim( strip_tags( $atts['strength'] ) );

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return '<break time="' . esc_attr( $atts['time'] ) . '" strength="' . esc_attr( $atts['strength'] ) . '" />';

		}

        return '';

    }

	/**
	 * Add Speaker mute by shortcode [speaker-mute]...[/speaker-mute].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 * @param $content - Shortcode content when using the closing shortcode construct: [foo] shortcode text [/ foo].
	 *
	 * @return string
	 * @since 1.0.0
	 * @access public
	 **/
	public function speaker_mute_shortcode( $atts, $content ): string {

        /** White list of options with default values. */
        $atts = shortcode_atts( [
            'tag' => 'div',
        ], $atts );

        $tag = $atts['tag'];

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return '<' . $tag . ' speaker-mute="">' . do_shortcode( $content ) . '</' . $tag . '>';

		}

        return do_shortcode( $content );

    }

	/**
	 * Add player by shortcode [speaker].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @return bool|false|string
	 * @since 2.0.0
	 * @access public
	 **/
	public function speaker_shortcode( $atts ) {

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return ''; }

		/**
		 * If selected other settings, but we found shortcode.
		 * Show short code, but don't read it.
		 **/
		if ( 'shortcode' !== Settings::get_instance()->options['player_position'] && ! is_array( $atts ) ) {
			return '<span class="speaker-mute">[speaker]</span>';
		}

		$params = shortcode_atts( [ 'id' => '0' ], $atts );

		$id = (int) $params['id'];
		if ( $id === 0 ) :
			$id = get_the_ID();
		endif;

		return Player::the_player( $id, false );

	}

	/**
	 * Add Speaker say‑as by shortcode [speaker-say‑as interpret-as="ordinal"][/speaker-say‑as].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_say_as_shortcode( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'interpret-as' => '',
			'format' => '',
			'detail' => ''
		], $atts );

		/** If we don't have interpret-as, exit. */
		if ( ! $atts['interpret-as'] ) {
			return do_shortcode( $content );
		}

		/** Extra protection from the fools */
		$atts['interpret-as'] = trim( strip_tags( $atts['interpret-as'] ) );
		$atts['format'] = trim( strip_tags( $atts['format'] ) );
		$atts['detail'] = trim( strip_tags( $atts['detail'] ) );

		$res = '<say-as interpret-as="' . $atts['interpret-as'] . '" ';

		if ( $atts['format'] ) {
			$res .= 'format="' . $atts['format'] . '" ';
		}

		if ( $atts['detail'] ) {
			$res .= 'detail="' . $atts['detail'] . '" ';
		}

		$res .= '>' . do_shortcode( $content ) . '</say-as>';

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return $res;

		}

		return do_shortcode( $content );

	}

	/**
	 * Add Speaker sub by shortcode [speaker-sub alias="Second World War"]WW2[/speaker-sub].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content - Shortcode content when using the closing shortcode construct: [foo] shortcode text [/ foo].
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_sub_shortcode( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'alias' => ''
		], $atts );

		/** If we don't have alias, exit. */
		if ( ! $atts['alias'] ) {

			return do_shortcode( $content );

		}

		/** Extra protection from the fools */
		$atts['alias'] = trim( strip_tags( $atts['alias'] ) );

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			/** @noinspection HtmlUnknownAttribute */
			return '<sub alias="' . esc_attr( $atts['alias'] ) . '">' . do_shortcode( $content ) . '</sub>';

		}

		return do_shortcode( $content );

	}

	/**
	 * Add Speaker emphasis by shortcode [speaker-emphasis level="reduced"]Some information[/speaker-emphasis].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content - Shortcode content when using the closing shortcode construct: [foo] shortcode text [/ foo].
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_emphasis_shortcode( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'level' => 'none'
		], $atts );

		/** Extra protection from the fools */
		$atts['level'] = trim( strip_tags( $atts['level'] ) );

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return '<emphasis level="' . esc_attr( $atts['level'] ) . '">' . do_shortcode( $content ) . '</emphasis>';

		}

		return do_shortcode( $content );

	}

	/**
	 * Add Speaker voice by shortcode [speaker-voice name="en-GB-Wavenet-B"]My new voice.[/speaker-voice].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_voice_shortcode( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'name' => ''
		], $atts );

		/** If we don't have name, exit. */
		if ( ! $atts['name'] ) {
			return do_shortcode( $content );
		}

		/** Extra protection from the fools */
		$atts['name'] = trim( strip_tags( $atts['name'] ) );

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return '<voice name="' . esc_attr( $atts['name'] ) . '">' . do_shortcode( $content ) . '</voice>';

		}

		return do_shortcode( $content );

	}

	/**
	 * Link to audio file Shortcode [speaker-file] or [speaker-file id="123"].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @return bool|false|string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_file( $atts ) {

		$params = shortcode_atts( [ 'id' => '0' ], $atts );

		$id = (int) $params['id'];

		/** URL to post audio file. */
		return SpeakerCaster::get_instance()->get_audio_url( $id );

	}

	/**
	 * Shortcode to say but not show content [speaker-say]Some Content.[/speaker-say].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content - Shortcode content when using the closing shortcode construct: [foo] shortcode text [/ foo].
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function speaker_say( $atts, $content ): string {

		shortcode_atts( [], $atts ); // To hide unused param warning.

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			/** @noinspection HtmlUnknownAttribute */
			return '<span speaker-say="">' . do_shortcode( $content ) . '</span>';

		}

		return '';

	}

	/**
	 * Shortcode used to customize the pitch, speaking rate, and volume of text contained by the element.
	 * [speaker-prosody rate="" pitch="" volume=""]Some Content.[/speaker-prosody].
	 *
	 * @param $atts - An associative array of attributes specified in the shortcode.
	 *
	 * @param $content
	 *
	 * @since 3.0.0
	 * @access public
	 **@return string
	 */
	public function speaker_prosody( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'rate' => '',
			'pitch' => '',
			'volume' => ''
		], $atts );

		/** If we don't have any params then exit. */
		if ( ! $atts['rate']  && ! $atts['pitch'] && ! $atts['volume'] )   {
			return do_shortcode( $content );
		}

		/** Extra protection from the fools */
		$atts['rate'] = trim( strip_tags( $atts['rate'] ) );
		$atts['pitch'] = trim( strip_tags( $atts['pitch'] ) );
		$atts['volume'] = trim( strip_tags( $atts['volume'] ) );

		$res = '<prosody ';

		if ( $atts['rate'] ) {
			$res .= 'rate="' . $atts['rate'] . '" ';
		}

		if ( $atts['pitch'] ) {
			$res .= 'pitch="' . $atts['pitch'] . '" ';
		}

		if ( $atts['volume'] ) {
			$res .= 'volume="' . $atts['volume'] . '" ';
		}

		$res .= '>' . do_shortcode( $content ) . '</prosody>';

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return $res;

		}

		return do_shortcode( $content );

	}

	/**
	 * Speaker phoneme Shortcode [speaker-phoneme alphabet="ipa" ph="təˈmeɪ.ɾoʊ"]tomato[/speaker-phoneme].
	 *
	 * @param $atts
	 * @param $content
	 *
	 * @return string
	 */
	public function speaker_phoneme_shortcode( $atts, $content ): string {

		/** White list of options with default values. */
		$atts = shortcode_atts( [
			'alphabet' => '',
			'ph' => ''
		], $atts );

		/** If we don't have any params then exit. */
		if ( ! $atts['alphabet']  && ! $atts['ph'] )   {
			return do_shortcode( $content );
		}

		/** Extra protection from the fools */
		$atts['alphabet'] = trim( strip_tags( $atts['alphabet'] ) );
		$atts['ph'] = trim( strip_tags( $atts['ph'] ) );

		$res = '<phoneme ';

		if ( $atts['alphabet'] ) {
			$res .= 'alphabet="' . $atts['alphabet'] . '" ';
		}

		if ( $atts['ph'] ) {
			$res .= 'ph="' . $atts['ph'] . '" ';
		}

		$res .= '>' . do_shortcode( $content ) . '</phoneme>';

		/** Show shortcodes only for our parser. Hide on frontend. */
		if ( isset( $_GET['speaker-ssml'] ) && $_GET['speaker-ssml'] ) {

			return $res;

		}

		return do_shortcode( $content );

	}

	/**
	 * Main Shortcodes Instance.
	 *
	 * @return Shortcodes
	 **/
	public static function get_instance(): Shortcodes {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
