<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabStorage {

	/**
	 * Controls for Schema tab.
	 * @return void
	 */
	public static function controls() {

		$tabs = Plugin::get_tabs();
		$fields = array();

		$fields[ 'file_name' ] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Audio File Name', 'speaker' ),
			'description'       =>
				wp_sprintf(
					/* translators: %1$s - allowed characters, %2$s - not allowed characters */
					esc_html__( 'Supported characters for a file name are letters, numbers, spaces, and %1$s. Characters that are NOT supported include, but are not limited to: %2$s', 'speaker' ),
					'<span class="speaker-file-name-allowed">( ) _ -</span>',
					'<span class="speaker-file-name-not-allowed">@ $ % & \ / : * ? " < > | # { } + = ; ! ^ ~ .`</span>'
				) .
				'<br><span style="line-height: 1.5">' .
				wp_sprintf(
					/* translators: %1$s - [id], %2$s - [name], %3$s - [title], %4%s - [author], %5$s - [type] */
					esc_html__( 'Enter a pattern for the audio file name:
					%1$s - post ID;
					%2$s - post slug;
					%3$s - post title;
					%4$s - post author;
					%5$s - post type', 'speaker'),
					'<br><button class="speaker-file-name-peace">[id]</button>',
					'<button class="speaker-file-name-peace">[name]</button>',
					'<button class="speaker-file-name-peace">[title]</button>',
					'<button class="speaker-file-name-peace">[author]</button>',
					'<button class="speaker-file-name-peace">[type]</button>'
				) . '</span>',
			'default'           => 'post-[id]',
		];

		$fields[ 'post_meta' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Custom Fields', 'speaker' ),
			'description'       => esc_html__( 'Add audio meta-data to the post Custom Fields', 'speaker' ),
			'default'           => 'off',
		];

		$fields[ 'media_library' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Media Library', 'speaker' ),
			'description'       => esc_html__( 'Make the audio visible and available in the Media Library', 'speaker' ),
			'default'           => 'off',
		];

		$fields[ 'storage_select' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Storage', 'speaker' ),
			'description'       => esc_html__( 'Select a storage to store the generated audio files', 'speaker' ),
			'default'           => 'library',
			'options'           => [
				"library" => esc_html__( 'Media Library', 'speaker' ),
				"library+drive" => esc_html__( 'Library + Google Drive', 'speaker' )
			],
		];

		/** Display authorization link */
		$fields[ 'storage-api-key' ] = [
			'type'              => 'file_dnd',
			'label'             => esc_html__( 'Google Drive API key', 'speaker' ),
			'file_types'        => array( 'json' ),
			'default'           => '',
			'description'       => StorageGoogle::is_key_exists() ? StorageGoogle::get_instance()->gd_token_link() : '',
		];

		$fields[ 'legacy_divider' ] = [
			'type'              => 'divider',
			'default'           => '',
		];

		$fields[ 'legacy' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Legacy Mode', 'speaker' ),
			'placeholder'       => esc_html__( 'Legacy', 'speaker' ),
			'description'       => esc_html__( 'Use the old method of storing audio files. Ignore the Audio File Name pattern and get audio files with old names for Speaker versions lower 4.', 'speaker' ),
			'default'           => 'off',
		];

		$tabs[ 'storage' ][ 'fields' ] = $fields;
		Plugin::set_tabs( $tabs );

	}

}
