<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Google\ApiCore\ApiException;
use Google\Cloud\TextToSpeech\V1\TextToSpeechClient as TextToSpeechClient7;
use Google\Cloud\TextToSpeech\V1\Client\TextToSpeechClient as TextToSpeechClient8;
use Google\Cloud\TextToSpeech\V1\ListVoicesRequest as ListVoicesRequest8;
use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabVoice {

	/** Text-to-Speech client */
	private static $tts_client;

	/** List of voices */
	private static $voices;

    /**
     * @var ApiException
     */
    private static $api_error_message;

	/**
	 * Require TTS library and get list of voices.
	 * @return bool
	 * @throws ApiException
	 */
	private static function require_tts(): bool {

        if ( ! SpeakerHelper::is_key_exists() ) { return false; }

		/** Includes the autoloader for libraries installed with Composer. */
		require_once SpeakerHelper::get_vendor_path();

		/** Setting custom exception handler. */
		set_exception_handler( [ ErrorHandler::class, 'exception_handler' ] );

        /** Get list of voices and return voices availability status */
        return self::tts_list_voices() !== false;

	}

	/**
     * Get list of voices and return voices availability status
	 * @return mixed
     */
    public static function tts_list_voices(): bool {

        $vendor_version = SpeakerHelper::get_vendor_version();

        switch ( $vendor_version ) {

            case 7:

                /** Create TextToSpeechClient */
                self::$tts_client = new TextToSpeechClient7();

                /** Get a list of voices from transient */
                if ( self::tts_list_voices_transient() ) {
                    return true;
                }

                /** Perform list voices request. */
                try {
                    $response = self::$tts_client->listVoices();
                } catch ( ApiException $e ) {
                    self::$api_error_message = $e;
                    return false;
                } finally {
                    self::$tts_client->close();
                }

                break;

            case 8:
            default:

                /** Create TextToSpeechClient */

                self::$tts_client = new TextToSpeechClient8();

                /** Get a list of voices from transient */
                if ( self::tts_list_voices_transient() ) {
                    return true;
                }

                /** Perform list voices request. */
                try {
                    $response = self::$tts_client->listVoices( new ListVoicesRequest8() );
                } catch ( ApiException $e ) {
                    self::$api_error_message = $e;
                    return false;
                } finally {
                    self::$tts_client->close();
                }

                break;

        }

        /** Get a voice list */
	    $voices = $response->getVoices();

	    /** Show a warning if it was not possible to get a list of voices. */
	    if ( count( $voices ) === 0 ) {

		    self::voice_list_error_message();
		    return false;

	    }

        // Remove voices without - in name (invalid voices)
        $valid_voices = [];
        foreach ( $voices as $voice ) {
            if ( strpos( $voice->getName(), '-' ) !== false ) {
                $valid_voices[] = $voice;
            }
        }
        $voices = $valid_voices;

	    /** Set transient for 12 hours to reduce request count */
	    set_transient( 'speaker_list_voices', $voices, 12 * HOUR_IN_SECONDS );

	    self::$voices = $voices;
	    return true;

    }

    /**
     * Get a list of voices from transient.
     *
     * @return bool Returns true if the list of voices is retrieved from transient, false otherwise.
     */
    private static function tts_list_voices_transient(): bool {

        /** Get a list of voices from transient */
        $transient_list_voices = get_transient( 'speaker_list_voices' );
        if ( $transient_list_voices ) {
            self::$voices = $transient_list_voices;
            return true;
        }

        return false;

    }

	/**
	 * Controls for Voice tab.
	 * @return void
	 * @throws ApiException
	 */
	public static function controls( $options ) {

		$tabs = Plugin::get_tabs();
		$fields = array();

		// If API Key is not set - show only key control.
		if ( SpeakerHelper::is_key_exists() && self::require_tts() ) {

			$fields[ 'current_language' ] = [
	            'type'              => 'custom_type',
	            'render'            => [ TabVoice::class, 'current_language' ],
	            'label'             => esc_html__( 'Now used', 'speaker' ),
	            'default'           => 'en-US-Standard-A',
	        ];

			$language_code_options = self::language_options( self::$voices );
			$fields[ 'language-code' ] = [
				'type'              => 'chosen',
				'label'             => esc_html__( 'Language', 'speaker' ),
				'description'       => wp_sprintf(
					/* translators: 1: voice types, 2: pricing */
					esc_html__( 'The list includes all %1$s for each available languages. WaveNet and Neural2 voices are higher quality voices at a different %2$s', 'speaker' ),
					'<a href="https://cloud.google.com/text-to-speech/docs/voices" target="_blank" title="' . esc_attr__( 'voice types', 'speaker' ) . '">' . esc_html__( 'voice types', 'speaker' ) . '</a>',
					'<a href="https://cloud.google.com/text-to-speech/pricing" target="_blank" title="' . esc_attr__( 'pricing', 'speaker' ) . '">' . esc_html__( 'pricing', 'speaker' ) . '</a>'
				),
				'default'           => self::current_locale( $language_code_options ),
				'options'           => $language_code_options,
				'attr'              => [
					'placeholder' => esc_html__( 'Choose language', 'speaker' )
				]
			];

			$fields[ 'language' ] = [
				'type'              => 'language',
				'render'            => [ TabVoice::class, 'voices' ],
				'default'           => 'en-US-Standard-A',
			];

			$fields[ 'advanced_voice' ] = [
				'type'              => 'switcher',
				'label'             => esc_html__( 'Advanced voice settings', 'speaker' ),
				'description'       => esc_html__( 'Advanced voice options for experienced users', 'speaker' ),
				'default'           => 'off',
				'sanitize_callback' => 'sanitize_text_field',
			];

			$fields[ 'audio-format' ] = [
				'type'              => 'select',
				'label'             => esc_html__( 'Audio format', 'speaker' ),
				'description'       => esc_html__( 'Select the format in which the audio will be sent. All recordings in other formats will become unavailable.', 'speaker' ),
				'default'           => 'mp3',
				'options'           => [
					'mp3' => esc_html( 'MP3' ),
					'wav' => esc_html( 'WAV' ),
				],
			];

			$fields[ 'audio_encoding' ] = [
				'type'              => 'select',
				'label'             => esc_html__( 'Audio Encoding', 'speaker' ),
				'description'       => esc_html__( 'The format of the audio byte stream. MP3 is recommended for better page loading performance', 'speaker' ),
				'default'           => 'MP3',
				'options'           => [
					'MP3' => esc_html( 'MP3' ),
					'MULAW' => esc_html( 'MULAW' ),
				],
			];

			$fields[ 'audio-profile' ] = [
				'type'              => 'select',
				'label'             => esc_html__( 'Audio Profile', 'speaker' ),
				'description'       => esc_html__( 'Optimize the synthetic speech for playback on different types of hardware.', 'speaker' ),
				'default'           => 'handset-class-device',
				'options'           => [
					'wearable-class-device' => esc_html__( 'Smart watches and other wearables', 'speaker' ),
					'handset-class-device' => esc_html__( 'Smartphones', 'speaker' ),
					'headphone-class-device' => esc_html__( 'Earbuds or headphones', 'speaker' ),
					'small-bluetooth-speaker-class-device' => esc_html__( 'Small home speakers', 'speaker' ),
					'medium-bluetooth-speaker-class-device' => esc_html__( 'Smart home speakers', 'speaker' ),
					'large-home-entertainment-class-device' => esc_html__( 'Home entertainment systems', 'speaker' ),
					'large-automotive-class-device' => esc_html__( 'Car speakers', 'speaker' ),
					'telephony-class-application' => esc_html__( 'Interactive Voice Response', 'speaker' ),
				]
			];

			$value = $options[ 'speaking-rate' ] ?? 0;
			$fields[ 'speaking-rate' ] = [
				'type'              => 'slider',
				'label'             => esc_html__( 'Speaking Rate/Speed', 'speaker' ),
				'description'       => esc_html__( 'Speaking rate', 'speaker') . ': <strong>' . esc_attr( $value ) . '</strong>',
				'default'           => 1,
				'min'               => 0.25,
				'max'               => 4.0,
				'step'              => 0.1,
				'atts'              => [
					'class' => 'mdc-slider-width',
				],
				'discrete'         => false
			];

			$value = $options[ 'pitch' ] ?? 0;
			$fields[ 'pitch' ] = [
				'type'              => 'slider',
				'label'             => esc_html__( 'Pitch', 'speaker' ),
				'description'       => esc_html__( 'Current pitch', 'speaker') . ': <strong>' . esc_attr( $value ) . '</strong>',
				'default'           => 0,
				'min'               => -20,
				'max'               => 20,
				'step'              => 0.1,
				'atts'              => [
					'class' => 'mdc-slider-width',
				],
				'discrete'         => false
			];

			$value = $options[ 'volume' ] ?? 0;
			$fields[ 'volume' ] = [
				'type'              => 'slider',
				'label'             => esc_html__( 'Volume Gain', 'speaker' ),
				'description'       => esc_html__( 'Current volume gain', 'speaker') . ': <strong>' . esc_attr( $value ) . '</strong>',
				'default'           => 0,
				'min'               => -16,
				'max'               => 16,
				'step'              => 0.1,
				'atts'              => [
					'class' => 'mdc-slider-width',
				],
				'discrete'         => false
			];

			$value = $options[ 'sample_rate' ] ?? 0;
			$fields[ 'sample_rate' ] = [
				'type'              => 'slider',
				'label'             => esc_html__( 'Sample Rate', 'speaker' ),
				'description'       => esc_html__( 'The synthesis sample rate', 'speaker') . ': <strong>' . esc_attr( $value ) . ' Hz</strong>',
				'default'           => 24000,
				'min'               => 4000,
				'max'               => 48000,
				'step'              => 4000,
				'atts'              => [
					'class' => 'mdc-slider-width',
				],
				'discrete'         => false
			];

			// TODO: SpeakerUtilities
			$fields[ 'auto_generation' ] = [
				'type'              => 'switcher',
				'label'             => esc_html__( 'Synthesize audio on save', 'speaker' ),
				'description'       => esc_html__( 'This significantly increases your expenses in Google Cloud.', 'speaker' ),
				'default'           => 'off',
			];

		} else {

            if ( self::$api_error_message ) {
                $fields[ 'error_message' ] = [
                    'type'              => 'custom_type',
                    'render'            => [ TabVoice::class, 'api_error' ],
                    'label'             => '',
                ];
            }

        }

        $key_reset_message = ( isset( $_GET['message'] ) && $_GET['message'] ) ?
            esc_html__( 'The Google API key was reset due to an error', 'speaker' ) . ': ' . esc_html( $_GET['message'] ) : '';
        $fields[ 'dnd-api-key' ] = [
			'type'              => 'file_dnd',
			'label'             => esc_html__( 'Google API Key File', 'speaker' ),
			'file_types'        => array( 'json' ),
			'default'           => '',
            'description' => $key_reset_message ? '<br><b style="color: orangered">' . $key_reset_message .'</b><br>' : ''
		];

		$tabs[ 'general' ][ 'fields' ] = $fields;
		Plugin::set_tabs( $tabs );

	}

	/**
	 * Get language code with country code
	 *
	 * @return string
	 */
	private static function current_locale( $api_locales ) {

		$wp_locale = get_locale() ?? 'en-US';
		$wp_locale = str_replace( '_', '-', $wp_locale );

		if ( array_key_exists( $wp_locale, $api_locales ) ) {

			// Return the locale if it's in the list.
			return $wp_locale;

		} else {

			// Otherwise, try to find a match.
			foreach ( $api_locales as $key => $value ) {
				if ( strpos( $key, $wp_locale ) !== false ) {
					return $key;
				}
			}

		}

		return 'en-US';

	}

	/**
	 * Current language player
     * @param bool $autoplayUI
	 */
	public static function current_language( bool $autoplayUI = true ) {

		$options = Settings::get_instance()->options;

		echo wp_sprintf(
			'<div class="mdp-now-used">
				<div>
					<strong>%1$s</strong>
				</div>
				<div>
					<audio controls="">
						<source src="https://cloud.google.com/text-to-speech/docs/audio/%1$s.wav" type="audio/wav">
						<source src="https://cloud.google.com/text-to-speech/docs/audio/%1$s.mp3" type="audio/wav">
						%2$s
					</audio>
				</div>
			</div>',
			esc_attr( $options['language'] ),
			esc_html__( 'Your browser does not support the audio element.', 'speaker' )
		);

        // Voice preview autoplay UI
        if ( $autoplayUI ) {

            $switcher = '<div class="mdc-switch mdc-switch--checked">
                <div class="mdc-switch__track"></div>
                <div class="mdc-switch__thumb-underlay">
                    <div class="mdc-switch__thumb">
                        <input id="mdp-now-used--autoplay" class="mdc-switch__native-control" type="checkbox" role="switch" checked="">
                    </div>
                </div>
            </div>';

            echo wp_sprintf(
                '<div class="mdp-now-used--autoplay" data-on="%2$s" data-off="%3$s">%1$s %4$s <b>%2$s</b></div>',
                esc_html__('Voices previews autoplay is:', 'speaker'),
                esc_html__('On', 'speaker'),
                esc_html__('Off', 'speaker'),
                $switcher
            );

        }

	}

	/**
	 * @throws ApiException
	 */
	public static function voices() {

		$options = Settings::get_instance()->options;

		?>

		<table id="mdp-speaker-settings-language-tbl" class="display stripe hidden">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Voice', 'speaker' ); ?></th>
					<th><?php esc_html_e( 'Gender', 'speaker' ); ?></th>
				</tr>
			</thead>
			<tbody>
			<?php

			$rendered_voices = array();
			foreach ( self::$voices as $voice ) :

				/** Skip already rendered voices. */
				if ( in_array( $voice->getName(), $rendered_voices ) ) {
					continue;
				} else {
					$rendered_voices[] = $voice->getName();
				}

				?>

				<?php
				$lang_name = Language::get_lang_by_code( $voice->getLanguageCodes() );
				/** Skip missing language. */
				if ( false === $lang_name ) { continue; }
				?>

				<tr <?php if ( $voice->getName() === $options['language'] ) { echo 'class="selected"'; } ?>>

					<?php

					/** @noinspection CssUnknownTarget */
					/** @noinspection HtmlUnknownTarget */
					echo wp_sprintf(
						'<td>
							<div class="mdp-lang-name">
								<div class="mdp-lang-face" style="background-image: url(%3$s)">
									<img src="%1$s" alt="%2$s">
								</div>
								<div class="mdp-lang-slug">
									<span class="mdp-lang-code" title="%4$s">%4$s</span>-<span class="mdp-voice-type">%5$s</span>-<span class="mdp-voice-name" title="%2$s">%6$s</span>
								</div>
								<button class="mdp-voice-copy material-icons material-symbols-outlined" title="%7$s">
									filter_none
									<span class="mdp-voice-copy--title" data-copy="%8$s" data-copied="%9$s">%8$s</span>
								</button>
							</div>
						</td>',
						self::get_face_url( $voice ),
						esc_html( $voice->getName() ),
						self::get_flag_url( $voice ),
						esc_html( $voice->getLanguageCodes()[0] ),
						self::get_voice_type( $voice->getName() ),
                        self::get_voice_suffix( $voice->getName() ),
						esc_html__( 'Copy voice name', 'speaker' ),
						esc_html__( 'Copy', 'speaker' ),
						esc_html__( 'Copied', 'speaker')
					);

					?>

					<td>
						<?php
						$gender = self::get_voice_gender( $voice );
						/** @noinspection HtmlUnknownTarget */
						echo wp_sprintf(
							'<span title="%1$s"><img src="%2$s" alt="%1$s">%3$s</span>',
							$gender ,
							Plugin::get_url() . 'images/' . strtolower( $gender  ) . '.svg',
							esc_html__( $gender, 'speaker' )
						);
						?>
					</td>
				</tr>
			<?php
			endforeach;

			self::$tts_client->close();

			?>
			</tbody>

		</table>
		<input id="mdp_speaker_general_settings_language" type='hidden' name='mdp_speaker_general_settings[language]'
		       value='<?php echo esc_attr( $options[ 'language' ] ); ?>'>
		<?php

		/** Restore previous exception handler. */
		restore_exception_handler();

	}

	/**
	 * Error message for voice list.
	 * @return void
	 */
	private static function voice_list_error_message() {

        if ( ! is_admin() ) { return; }

		?><div class="mdp-alert-error"><?php

		esc_html_e( 'Failed to get the list of languages. 
            The request failed. It looks like a problem with your API Key File. 
            Make sure that you are using the correct key file, and that the quotas have not been exceeded. 
            If you set security restrictions on a key, make sure that the current domain is added to the exceptions.', 'speaker' );

		?></div><?php

	}

    /**
     * Google cloud error message.
     * @return void
     */
    public static function api_error() {

        // Get html content
        $html_content = self::$api_error_message->getBasicMessage();

        // Define the regular expression pattern
        $pattern = '/<!DOCTYPE html>.*?<title>.*?<\/title>/s';

        // Replace the matched pattern with an empty string
        $result = preg_replace( $pattern, '', $html_content );
        $result = str_replace( '*', '.mdp-speaker-api-error-container', $result );
        $result = str_replace( 'html{', '.mdp-speaker-api-error-html{', $result );
        $result = str_replace( 'body{', '.mdp-speaker-api-error-body{', $result );

        echo wp_sprintf(
            '<div class="mdp-speaker-api-error-container">
                <div class="mdp-speaker-api-error-body">
                    %s
                    <p>%s: <b>%s</b>. %s: <b>%s</b>.</p>
                </div>
                
            </div>',
            $result,
            esc_html__( 'Code', 'speaker' ),
            self::$api_error_message->getCode(),
            esc_html__( 'Status', 'speaker' ),
            self::$api_error_message->getStatus()
        );

    }

	/**
	 * Language list for select.
	 *
	 * @param $voices
	 *
	 * @return array
	 */
	private static function language_options( $voices ): array {

		/** Prepare Languages Options. */
		$options = [];
		$options[] = esc_html__( 'Select Language', 'speaker' );
		foreach ( $voices as $voice ) {

			$lang_code = $voice->getLanguageCodes()[0];
			$lang_name = Language::get_lang_by_code( $voice->getLanguageCodes() );
			if ( false === $lang_name ) { continue; } // Skip missing language

			$options[ $lang_code ] = $lang_name;

		}

		/** Remove duplicated lang codes */
		ksort( $options );

		return $options;

	}

	/**
	 * Return Voice Type.
	 *
	 * @param $lang_name - Google voice name.
	 *
	 * @return string
	 * @access private
	 *
	 * @noinspection HtmlUnknownTarget
	 */
	private static function get_voice_type( $lang_name): string
	{

        // Split the string by the hyphen character
        $parts = explode( '-', $lang_name );

        // Remove first 2 elements and last one
        array_shift( $parts );
        array_shift( $parts );
        array_pop( $parts );

        // Join the remaining elements with a hyphen
        $type = implode( '-', $parts );

        // Return with icon if exists
        $icons = ['wavenet', 'neural', 'news', 'studio', 'journey', 'casual', 'polyglot', 'standard', 'neural2', 'chirp3-hd', 'chirp-hd', 'chirp'];
        if ( in_array( strtolower($type), $icons ) ) {
            return wp_sprintf(
                '<img src="%s" alt="%s">%s',
                Plugin::get_url() . 'images/voices/' . strtolower($type) . '.svg',
                esc_attr( $type ),
                esc_html( $type )
            );
        }

        return esc_html( $type );

	}

    /**
     * Return Voice Suffix (the last part of voice name).
     *
     * @param $voice_name - Google voice name.
     *
     * @return string
     * @access private
     *
     * @noinspection HtmlUnknownTarget
     */
    private static function get_voice_suffix( $voice_name ): string {
        // Split the string by the hyphen character
        $parts = explode( '-', $voice_name );

        // If lang_name does not contain expected parts, return it as is
        if ( count( $parts ) < 1 ) {
            return esc_html( $voice_name );
        }

        // Get the last element
        $suffix = array_pop( $parts );

        return esc_html( $suffix );
    }

	/**
	 * Get face url for voice.
	 *
	 * @param $voice
	 *
	 * @return string
	 */
	private static function get_face_url( $voice ): string {

		$random = rand( 1, 16 );
		if ( $random < 10 ) { $random = '0' . $random; }

		$url = wp_sprintf(
			'images/faces/%1$s-%2$s.png',
			strtolower( self::get_voice_gender( $voice ) ),
			$random
		);

		return esc_url( Plugin::get_url() . $url );

	}

	/**
	 * Get flag url for voice.
	 *
	 * @param $voice
	 *
	 * @return string
	 */
	private static function get_flag_url( $voice ): string {

		$voiceSlug = explode( '-', $voice->getLanguageCodes()[0] );
		$locale = $voiceSlug[ 1 ] ?? 'none';

		// Replace country codes that do not have a flag
		if ( $locale === 'XA' ) { $locale = 'AE'; }

		$url = wp_sprintf(
			'https://flagcdn.com/h40/%1$s.png',
			strtolower( $locale )
		);

		return esc_url( $url );

	}

	/**
	 * Get voice gender
	 * @param $voice
	 *
	 * @return string|null
	 */
	public static function get_voice_gender( $voice ): ?string {

		$ssmlVoiceGender = [ 'SSML_VOICE_GENDER_UNSPECIFIED', 'Male', 'Female', 'Neutral' ];
		$gender = $ssmlVoiceGender[ $voice->getSsmlGender() ];

		return esc_attr( $gender );

	}

    /**
     * Voice select for Speech Template element.
     * @throws ApiException
     */
    public static function st_element_voice() {

        if ( ! self::require_tts() ) { return; }

        /** Prepare Languages Options. */
        $options = [];
        $options[] = esc_html__( 'Select Language', 'speaker' );
        foreach ( self::$voices as $voice ) {

            $lang = Language::get_lang_by_code( $voice->getLanguageCodes() );
            if ( false === $lang ) { continue; } // Skip missing language

            $options[$lang] = $lang;

        }
        ksort( $options );

        /** Render Language select. */
        UI::get_instance()->render_select(
            $options,
            '',
            esc_html__('Language', 'speaker' ),
            '',
            [
                'name' => 'mdp_speaker_language_filter',
                'id' => 'mdp-speaker-language-filter'
            ]
        );

        ?>

        <div class="mdc-text-field-helper-line mdp-speaker-helper-padding">
            <div class="mdc-text-field-helper-text mdc-text-field-helper-text--persistent"><?php echo wp_sprintf(
                    '%1$s <a href="https://cloud.google.com/text-to-speech/docs/wavenet" target="_blank" title="%2$s">%2$s</a> %3$s <a href="https://cloud.google.com/text-to-speech/pricing" target="_blank" title="%4$s">%4$s</a>',
                    esc_html__( 'The list includes Standard, WaveNet and Neural2 voice types', 'speaker' ),
                    esc_html__( 'voice types', 'speaker' ),
                    esc_html__( 'WaveNet and Neural2 voices are higher quality voices at a different', 'speaker' ),
                    esc_html__( 'pricing', 'speaker' )
                ); ?>
            </div>
        </div>

        <table id="mdp-speaker-settings-language-tbl" class="display stripe hidden">
            <thead>
            <tr>
                <th><?php esc_html_e( 'Language', 'speaker' ); ?></th>
                <th><?php esc_html_e( 'Voice', 'speaker' ); ?></th>
                <th><?php esc_html_e( 'Gender', 'speaker' ); ?></th>
            </tr>
            </thead>
            <tbody>
            <?php

            $rendered_voices = array();
            foreach ( self::$voices as $voice ) :

                /** Skip already rendered voices. */
                if ( in_array( $voice->getName(), $rendered_voices ) ) {
                    continue;
                } else {
                    $rendered_voices[] = $voice->getName();
                }

                ?>

                <?php
                $lang_name = Language::get_lang_by_code( $voice->getLanguageCodes() );
                /** Skip missing language. */
                if ( false === $lang_name ) { continue; }
                ?>

                <tr <?php if ( $voice->getName() === Settings::get_instance()->options['language'] ) { echo 'class="selected"'; } ?>>
                    <td class="mdp-lang-name">
                        <?php echo esc_html( $lang_name ); // Language. ?>
                    </td>
                    <td>
                        <?php
                        echo wp_sprintf(
                            '<span class="mdp-lang-code" title="%1$s">%1$s</span> - <span>%2$s</span> - <span class="mdp-voice-name" title="%3$s">%4$s</span>',
                            esc_html( $voice->getLanguageCodes()[0] ),
                            self::get_voice_type( $voice->getName() ),
                            esc_html( $voice->getName() ),
                            esc_html( substr( $voice->getName(), -1 ) )
                        );
                        ?>
                    </td>
                    <td>
                        <?php
                        $ssmlVoiceGender = [ 'SSML_VOICE_GENDER_UNSPECIFIED', 'Male', 'Female', 'Neutral' ];
                        echo wp_sprintf(
                            '<span title="%1$s"><img src="%2$s" alt="%1$s">%3$s</span>',
                            esc_attr( $ssmlVoiceGender[ $voice->getSsmlGender() ] ),
                            Plugin::get_url() . 'images/' . strtolower( $ssmlVoiceGender[ $voice->getSsmlGender() ] ) . '.svg',
                            esc_html__( $ssmlVoiceGender[ $voice->getSsmlGender() ], 'speaker' )
                        );
                        ?>
                    </td>
                </tr>
            <?php
            endforeach;

            self::$tts_client->close();

            ?>
            </tbody>

        </table>

        <input id="mdp-speaker-settings-language" type='hidden' name='mdp_speaker_settings[language]'
               value='<?php echo esc_attr( Settings::get_instance()->options['language'] ); ?>'>
        <input id="mdp-speaker-settings-language-code" type='hidden' name='mdp_speaker_settings[language-code]'
               value='<?php echo esc_attr( Settings::get_instance()->options['language-code'] ); ?>'>
        <?php

        /** Restore previous exception handler. */
        restore_exception_handler();

    }

}
