<?php

namespace Merkulove\Speaker\Unity;

class Task
{
    /**
     * Get the timestamp of the last finished cron job
     * @return int Timestamp of the last finished cron job
     */
    public static function interval(): int
    {
		$default_interval = 86400; // Default to 24 hours

        $schedules = wp_get_schedules();
		if ( ! $schedules ) {
			return $default_interval;
		}

        $recurrence = Plugin::const('cron_recurrence');
        if ( $recurrence && isset( $schedules[ $recurrence ] ) ) {
            return $schedules[ $recurrence ]['interval'] ?? $default_interval;
        } else {
            return $default_interval;
        }
    }

    /**
     * Check if the last finished cron job was within the interval
     * @return bool True if the last finished cron job was within the interval, false otherwise
     */
    public static function is_finished_recently(): bool
    {
        $interval = self::interval();
        $finished_at = self::get_finished_at();
        $now = time();

        return ($now - $finished_at) < $interval;
    }

    /**
     * Get the timestamp of the last finished cron job
     * @return int The timestamp of the last finished cron job
     */
    public static function get_finished_at(): int
    {
        $option_name = Plugin::const('cron_hook') . '_timestamp';
        return get_option($option_name);
    }

    /**
     * Set the timestamp of the last finished cron job to the current time
     * @return void
     */
    public static function set_finished_at(): void
    {
        $option_name = Plugin::const('cron_hook') . '_timestamp';
        update_option($option_name, time());
    }

	/**
	 * Check if cron jobs are enabled in the plugin settings
	 * @return bool True if cron jobs are enabled, false otherwise
	 */
	public static function is_use_cron(): bool
	{
		$options = Settings::get_instance()->options;
		return isset( $options['use_cron'] ) && $options['use_cron'] === 'on';
	}

	/**
	 * Check if the cron job is scheduled
	 * @return bool True if the cron job is scheduled, false otherwise
	 */
	public static function is_schedule(): bool {

		$cron_hook = Plugin::const('cron_hook');
		if (! $cron_hook) {
			return false;
		}

		$timestamp = wp_next_scheduled($cron_hook);
		return (bool) $timestamp;

	}

	/**
	 * Set up the cron job schedule
	 * @return void
	 */
	public static function set_schedule() {

		// Get cron props
		$cron_hook = Plugin::const('cron_hook');
		$cron_recurrence = Plugin::const('cron_recurrence');
		if (! $cron_hook || ! $cron_recurrence) {
			return;
		}

		// Unschedule if cron jobs are disabled
		if ( ! self::is_use_cron() ) {
			do_action( $cron_hook . '_on_demand' );
			self::unschedule();
			return;
		}

		// Schedule cron job if not already scheduled
		if (! wp_next_scheduled($cron_hook)) {
			wp_schedule_event(time(), $cron_recurrence, $cron_hook);
		}

	}

	/**
	 * Unschedule the cron job
	 * @return void
	 */
	public static function unschedule() {

		// Get cron hook
		$cron_hook = Plugin::const('cron_hook');
		if (! $cron_hook) {
			return;
		}

		// Unschedule cron job
		$timestamp = wp_next_scheduled($cron_hook);
		if ($timestamp) {
			wp_unschedule_event($timestamp, $cron_hook);
		}

	}
}
