<?php
/**
 * Delivery - admin class.
 *
 * @package SpringDevs\SubscriptionPro\Admin
 */

namespace SpringDevs\SubscriptionPro\Admin;

use WC_Countries;

/**
 * Delivery - admin class.
 */
class Delivery {

	/**
	 * Initialize the class.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ), 20 );
		add_filter( 'wp_subscription_admin_header_menu_items', array( $this, 'add_delivery_menu_item' ), 10, 2 );
		add_filter( 'manage_edit-subscrpt_delivery_columns', array( $this, 'manage_columns' ) );
		add_filter( 'manage_edit-subscrpt_delivery_sortable_columns', array( $this, 'manage_sortable_columns' ) );
		add_action( 'manage_subscrpt_delivery_posts_custom_column', array( $this, 'display_custom_columns' ), 10, 2 );
		add_filter( 'bulk_actions-edit-subscrpt_delivery', array( $this, 'remove_bulk_actions' ) );
		add_action( 'wp_ajax_subscrpt_update_delivery_status', [ $this, 'ajax_update_delivery_status' ] );
		add_action( 'admin_init', [ $this, 'maybe_render_printable_schedule' ] );
	}

	/**
	 * Register submenu under `subscriptions` menu.
	 *
	 * @return void
	 */
	public function register_menu() {
		$parent_slug = 'wp-subscription';
		add_submenu_page(
			$parent_slug,
			__( 'Delivery Schedules', 'wp_subscription_pro' ),
			__( 'Delivery Schedules', 'wp_subscription_pro' ),
			'manage_options',
			'wp-subscription-delivery',
			array( $this, 'render_delivery_page' ),
			40
		);
	}

	/**
	 * Render the Delivery Schedules page styled like the main subscriptions list.
	 */
	public function render_delivery_page() {
		// Get filter parameters from URL

		// ? Ignored nonce verification intentionally. Because this is not a form submission.
		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		$status      = isset( $_GET['subscrpt_status'] ) ? array_map( 'sanitize_text_field', (array) wp_unslash( $_GET['subscrpt_status'] ) ) : [];
		$date_filter = isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '';
		$search      = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
		$page        = isset( $_GET['paged'] ) ? max( 1, intval( $_GET['paged'] ) ) : 1;
		$per_page    = isset( $_GET['per_page'] ) ? intval( $_GET['per_page'] ) : 20;
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		// Define available statuses
		$available_statuses = array(
			'waiting'    => __( 'Waiting', 'wp_subscription_pro' ),
			'in_process' => __( 'In Process', 'wp_subscription_pro' ),
			'shipped'    => __( 'Shipped', 'wp_subscription_pro' ),
			'cancelled'  => __( 'Cancelled', 'wp_subscription_pro' ),
		);

		// Build query args
		$query_args = array(
			'post_type'      => 'subscrpt_delivery',
			'posts_per_page' => $per_page,
			'paged'          => $page,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		// Add status filter
		if ( ! empty( $status ) ) {
			// Filter only valid statuses
			$valid_statuses = array_intersect( $status, array_keys( $available_statuses ) );
			if ( ! empty( $valid_statuses ) ) {
				$query_args['post_status'] = $valid_statuses;
			} else {
				// Default to waiting and in_process if no valid status specified
				$query_args['post_status'] = array( 'waiting', 'in_process' );
			}
		} else {
			// Default to waiting and in_process if no status specified
			$query_args['post_status'] = array( 'waiting', 'in_process' );
		}

		// Add date filter
		if ( ! empty( $date_filter ) ) {
			$date_parts = explode( '-', $date_filter );
			if ( count( $date_parts ) === 2 ) {
				$year                     = intval( $date_parts[0] );
				$month                    = intval( $date_parts[1] );
				$query_args['date_query'] = array(
					array(
						'year'  => $year,
						'month' => $month,
					),
				);
			}
		}

		// Add search filter
		if ( ! empty( $search ) ) {
			$query_args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query, (Not always used)
				'relation' => 'OR',
				[
					'key'     => '_subscrpt_subscription_id',
					'value'   => $search,
					'compare' => 'LIKE',
				],
			];
		}

		// Get deliveries with pagination
		$deliveries_query = new \WP_Query( $query_args );
		$deliveries       = $deliveries_query->posts;
		$total_deliveries = $deliveries_query->found_posts;
		$total_pages      = $deliveries_query->max_num_pages;

		// Generate months for date filter
		$months = array();
		for ( $i = 0; $i < 12; $i++ ) {
			$month                             = strtotime( "-$i month" );
			$months[ gmdate( 'Y-m', $month ) ] = gmdate( 'F Y', $month );
		}

		// Load the template with additional data
		$template_data = [
			'deliveries'         => $deliveries,
			'available_statuses' => $available_statuses,
			'status'             => $status,
			'date_filter'        => $date_filter,
			'search'             => $search,
			'months'             => $months,
			'page'               => $page,
			'total_pages'        => $total_pages,
			'total_deliveries'   => $total_deliveries,
			'per_page'           => $per_page,
		];

		// Pass variables to template
		$deliveries         = $template_data['deliveries'];
		$available_statuses = $template_data['available_statuses'];
		$status             = $template_data['status'];
		$date_filter        = $template_data['date_filter'];
		$search             = $template_data['search'];
		$months             = $template_data['months'];
		$page               = $template_data['page'];
		$total_pages        = $template_data['total_pages'];
		$total_deliveries   = $template_data['total_deliveries'];
		$per_page           = $template_data['per_page'];

		// Enqueue admin JS and CSS.
		wp_enqueue_script( 'subscrpt_admin_delivery_js', SUBSCRIPT_PRO_URL . '/assets/js/admin/delivery.js', [ 'jquery' ], SUBSCRIPT_PRO_VERSION, true );
		wp_localize_script(
			'subscrpt_admin_delivery_js',
			'subscrpt_update_delivery_status',
			[
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'security' => wp_create_nonce( 'subscrpt_update_delivery_status' ),
			]
		);

		// Call the template.
		include SUBSCRIPT_PRO_PATH . '/templates/admin/delivery.php';
	}

	/**
	 * Add Delivery Schedule link to the WPSubscription admin header menu.
	 *
	 * @param array  $menu_items Menu items array.
	 * @param string $current   Current menu item.
	 */
	public function add_delivery_menu_item( $menu_items, $current ) {
		$menu_items[] = [
			'slug'  => 'wp-subscription-delivery',
			'label' => __( 'Delivery Schedules', 'wp_subscription_pro' ),
			'url'   => admin_url( 'admin.php?page=wp-subscription-delivery' ),
		];
		return $menu_items;
	}

	/**
	 * Columns for `subscrpt_delivery` post type.
	 *
	 * @param array $columns Columns.
	 *
	 * @return array
	 */
	public function manage_columns( array $columns ): array {
		unset( $columns['cb'] );
		unset( $columns['title'] );
		unset( $columns['date'] );
		$columns['product']       = __( 'Product', 'wp_subscription_pro' );
		$columns['subscription']  = __( 'Subscription', 'wp_subscription_pro' );
		$columns['status']        = __( 'Delivery Status', 'wp_subscription_pro' );
		$columns['shipping_on']   = __( 'Shipping on', 'wp_subscription_pro' );
		$columns['shipped_on']    = __( 'Shipped on', 'wp_subscription_pro' );
		$columns['delivery_info'] = __( 'Delivery Info', 'wp_subscription_pro' );

		return $columns;
	}

	/**
	 * Filter sortable columns.
	 *
	 * @param array $columns Columns.
	 *
	 * @return array
	 */
	public function manage_sortable_columns( array $columns ): array {
		unset( $columns['title'] );

		return $columns;
	}

	/**
	 * Display content for custom columns.
	 *
	 * @param string $column_name Column name.
	 * @param int    $post_id     Post ID.
	 *
	 * @return void
	 */
	public function display_custom_columns( $column_name, $post_id ) {
		wp_enqueue_script( 'subscrpt_admin_delivery_js' );
		$subscription_id = get_post_meta( $post_id, '_subscrpt_subscription_id', true );

		switch ( $column_name ) {
			case 'product':
				$order_item_id = get_post_meta( $subscription_id, '_subscrpt_order_item_id', true );
				$order_id      = wc_get_order_id_by_order_item_id( $order_item_id );
				$order         = wc_get_order( $order_id );
				if ( $order ) {
					$order_item = $order->get_item( $order_item_id );
					if ( $order_item && is_a( $order_item, 'WC_Order_Item_Product' ) ) {
						$variation_id = $order_item->get_variation_id();
						$product_id   = $order_item->get_product_id();
						$product_link = get_permalink( $variation_id !== 0 ? $variation_id : $product_id );
						echo '<a href="' . esc_url( $product_link ) . '" target="_blank">' . esc_html( $order_item->get_name() ) . '</a>';
					}
				}
				break;
			case 'subscription':
				echo '<a href="' . esc_url( get_edit_post_link( $subscription_id ) ) . '" target="_blank">' . esc_html( '# ' . $subscription_id ) . '</a>';
				break;

			case 'status':
				$current_status = get_post_status( $post_id );
				$statuses       = get_post_stati(
					array(
						'show_in_admin_status_list' => true,
						'post_type'                 => array( 'subscrpt_delivery' ),
					),
					'objects'
				);

				if ( ! isset( $statuses['cancelled'] ) ) {
					$statuses['cancelled'] = (object) array(
						'label' => __( 'Cancelled', 'wp_subscription_pro' ),
						'name'  => 'cancelled',
					);
				}

				echo '<select class="subscrpt_delivery_status" name="post_status" data-post-id="' . esc_attr( $post_id ) . '" id="post_status_' . esc_attr( $post_id ) . '">';
				foreach ( $statuses as $status => $status_object ) {
					echo '<option value="' . esc_attr( $status ) . '"' . selected( $current_status, $status, false ) . '>' . esc_html( $status_object->label ) . '</option>';
				}
				echo '</select>';
				break;

			case 'shipping_on':
				$shipping_date = get_post_meta( $post_id, '_subscrpt_shipping_on', true );
				echo esc_html( $shipping_date ? date_i18n( get_option( 'date_format' ), $shipping_date ) : '-' );
				break;

			case 'shipped_on':
				$shipped_date = get_post_meta( $post_id, '_subscrpt_shipped_date', true );
				echo esc_html( $shipped_date ? date_i18n( get_option( 'date_format' ), $shipped_date ) : '-' );
				break;

			case 'delivery_info':
				$order_id = get_post_meta( $subscription_id, '_subscrpt_order_id', true );
				$order    = wc_get_order( $order_id );
				echo wp_kses_post( $order->get_formatted_shipping_address() );
				break;
		}
	}

	/**
	 * Remove bulk actions for subscrpt_delivery post type.
	 *
	 * @param array $actions Bulk actions.
	 * @return array
	 */
	public function remove_bulk_actions( $actions ) {
		// Remove all bulk actions for delivery post type.
		return array();
	}

	/**
	 * AJAX handler for updating delivery status.
	 */
	public function ajax_update_delivery_status() {
		// Verify nonce
		if ( ! isset( $_POST['security'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['security'] ) ), 'subscrpt_update_delivery_status' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'wp_subscription_pro' ) ) );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'wp_subscription_pro' ) ) );
		}

		$delivery_id = intval( $_POST['delivery_id'] ?? 0 );
		$new_status  = sanitize_text_field( wp_unslash( $_POST['new_status'] ?? '' ) );

		if ( ! $delivery_id || ! in_array( $new_status, array( 'waiting', 'in_process', 'shipped', 'cancelled' ), true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid data.', 'wp_subscription_pro' ) ) );
		}

		$result = wp_update_post(
			[
				'ID'          => $delivery_id,
				'post_status' => $new_status,
			]
		);

		if ( $result && 'shipped' === $new_status ) {
			update_post_meta( $delivery_id, '_subscrpt_shipped_date', time() );
		}

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Status updated successfully.', 'wp_subscription_pro' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to update status.', 'wp_subscription_pro' ) ) );
		}
	}

	/**
	 * If requested, render a standalone printable HTML schedule before admin chrome loads.
	 */
	public function maybe_render_printable_schedule() {
		if ( ! is_admin() ) {
			return;
		}

		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		$page   = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
		$action = isset( $_GET['subscrpt_action'] ) ? sanitize_text_field( wp_unslash( $_GET['subscrpt_action'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		if ( 'wp-subscription-delivery' !== $page || 'print_schedule' !== $action ) {
			return;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to view this page.', 'wp_subscription_pro' ) );
		}

		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		$status       = isset( $_GET['subscrpt_status'] ) ? array_map( 'sanitize_text_field', (array) wp_unslash( $_GET['subscrpt_status'] ) ) : [];
		$date_filter  = isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '';
		$search       = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
		$scope        = isset( $_GET['scope'] ) ? sanitize_text_field( wp_unslash( $_GET['scope'] ) ) : '';
		$selected_ids = isset( $_GET['delivery_ids'] ) ? array_map( 'intval', (array) wp_unslash( $_GET['delivery_ids'] ) ) : [];
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		$available_statuses = array(
			'waiting'    => __( 'Waiting', 'wp_subscription_pro' ),
			'in_process' => __( 'In Process', 'wp_subscription_pro' ),
			'shipped'    => __( 'Shipped', 'wp_subscription_pro' ),
			'cancelled'  => __( 'Cancelled', 'wp_subscription_pro' ),
		);

		$query_args = array(
			'post_type'      => 'subscrpt_delivery',
			'posts_per_page' => -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		if ( ! empty( $status ) ) {
			$valid_statuses            = array_intersect( $status, array_keys( $available_statuses ) );
			$query_args['post_status'] = ! empty( $valid_statuses ) ? $valid_statuses : array( 'waiting', 'in_process' );
		} else {
			$query_args['post_status'] = array( 'waiting', 'in_process' );
		}

		if ( ! empty( $date_filter ) ) {
			$date_parts = explode( '-', $date_filter );
			if ( 2 === count( $date_parts ) ) {
				$query_args['date_query'] = array(
					array(
						'year'  => intval( $date_parts[0] ),
						'month' => intval( $date_parts[1] ),
					),
				);
			}
		}

		if ( ! empty( $search ) ) {
			$query_args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query, (Not always used)
				'relation' => 'OR',
				[
					'key'     => '_subscrpt_subscription_id',
					'value'   => $search,
					'compare' => 'LIKE',
				],
			];
		}

		if ( 'selected' === $scope && ! empty( $selected_ids ) ) {
			$query_args['post__in'] = $selected_ids;
		}

		// Helper for country names
		$wc_countries = new WC_Countries();

		// Get store information
		$store_address   = get_option( 'woocommerce_store_address', '' );
		$store_address_2 = get_option( 'woocommerce_store_address_2', '' );

		$store_location        = get_option( 'woocommerce_default_country', '' );
		$location_parts        = explode( ':', $store_location );
		$location_part_country = $location_parts[0] ?? '';
		$location_part_state   = $location_parts[1] ?? '';

		$store_country = $wc_countries->countries[ $location_part_country ] ?? $location_part_country;
		$store_state   = '';
		if ( $location_part_country && $location_part_state ) {
			$states      = $wc_countries->get_states( $location_part_country );
			$store_state = $states[ $location_part_state ] ?? $location_part_state;
		}

		$store_info = [
			'name'     => trim( get_bloginfo( 'name' ) ),
			'address'  => trim( $store_address . ( $store_address_2 ? ', ' . $store_address_2 : '' ) ),
			'city'     => trim( get_option( 'woocommerce_store_city', '' ) ),
			'postcode' => trim( get_option( 'woocommerce_store_postcode', '' ) ),
			'state'    => trim( $store_state ),
			'country'  => trim( $store_country ),
		];

		// Delivery addresses
		$print_query = new \WP_Query( $query_args );
		$deliveries  = $print_query->posts;

		// Send no-cache headers and set content type.
		nocache_headers();
		header( 'Content-Type: text/html; charset=' . get_option( 'blog_charset' ) );

		$generated_at = wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );

		$available_statuses_local = $available_statuses;

		// Call the template.
		include SUBSCRIPT_PRO_PATH . '/templates/admin/delivery-print.php';
		exit;
	}
}
