<?php

namespace SpringDevs\SubscriptionPro\Admin;

/**
 * Class Settings
 *
 * @package SpringDevs\SubscriptionPro\Admin
 */
class Settings {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'wp_subscription_setting_fields', array( $this, 'setting_fields' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting( 'wp_subscription_settings', 'subscrpt_renewal_price' );
		register_setting( 'wp_subscription_settings', 'subscrpt_early_renew' );
		register_setting( 'wpsubscription_api_key', 'wpsubscription_api_key' );
		register_setting( 'wpsubscription_api_enabled', 'wpsubscription_api_enabled' );

		// Payment Failure Handling Settings
		register_setting( 'wp_subscription_settings', 'subscrpt_default_max_payment_retries' );
		register_setting( 'wp_subscription_settings', 'subscrpt_default_payment_grace_period' );
		register_setting( 'wp_subscription_settings', 'subscrpt_enable_payment_failure_emails' );
		register_setting( 'wp_subscription_settings', 'subscrpt_payment_failure_email_delay' );
		register_setting( 'wp_subscription_settings', 'subscrpt_enable_grace_period_notifications' );
		register_setting( 'wp_subscription_settings', 'subscrpt_grace_period_warning_days' );
	}

	/**
	 * Enqueue admin scripts
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( 'subscrpt_admin_api_js' );
		wp_localize_script(
			'subscrpt_admin_api_js',
			'subscrptPro',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'subscrpt_pro_ajax_nonce' ),
				'i18n'    => array(
					'confirmRegenerate' => __( 'Are you sure you want to regenerate the API key? This will invalidate the existing key.', 'wp_subscription_pro' ),
					'apiKeyRegenerated' => __( 'API key has been regenerated successfully.', 'wp_subscription_pro' ),
					'error'             => __( 'An error occurred. Please try again.', 'wp_subscription_pro' ),
				),
			)
		);
	}

	/**
	 * Settings Fields HTML
	 */
	public function setting_fields() {
		?>
		<tr>
				<th scope="row">
					<label for="subscrpt_renewal_price">
		<?php esc_html_e( 'Renewal Price', 'wp_subscription_pro' ); ?>
					</label>
				</th>
				<td>
					<select name="subscrpt_renewal_price" id="subscrpt_renewal_price">
						<option value="subscribed"><?php esc_html_e( 'Subscribed Price', 'wp_subscription_pro' ); ?></option>
						<option value="updated" <?php echo get_option( 'subscrpt_renewal_price', 'subscribed' ) === 'updated' ? 'selected' : null; ?>><?php esc_html_e( 'New/Updated Price', 'wp_subscription_pro' ); ?></option>
					</select>
					<p class="description"><?php esc_html_e( 'Choose a price that will be used for subscription renewal.', 'wp_subscription_pro' ); ?></p>
				</td>
			</tr>
		<tr valign="top">
			<tr valign="top">
				<th scope="row" class="titledesc"><?php echo esc_html_e( 'Early Renewal', 'wp_subscription_pro' ); ?></th>
				<td class="forminp forminp-checkbox">
					<fieldset>
						<legend class="screen-reader-text"><span><?php echo esc_html_e( 'Early Renewal', 'wp_subscription_pro' ); ?></span></legend>
						<label for="subscrpt_early_renew">
							<input class="wp-subscription-toggle" name="subscrpt_early_renew" id="subscrpt_early_renew" type="checkbox" value="1" <?php checked( get_option( 'subscrpt_early_renew', '1' ), '1' ); ?>>
							<span class="wp-subscription-toggle-ui" aria-hidden="true"></span>
							
							<?php echo esc_html_e( 'Accept Early Renewal Payments', 'wp_subscription_pro' ); ?>
						</label>
						<p class="description"><?php echo esc_html_e( 'With early renewals enabled, customers can renew their subscriptions before the next payment date.', 'wp_subscription_pro' ); ?></p>
					</fieldset>
				</td>
			</tr>
		</tr>

		<!-- Payment Failure Handling Settings -->
		<tr>
			<th scope="row" colspan="2">
				<h2><?php esc_html_e( 'Payment Failure Handling', 'wp_subscription_pro' ); ?></h2>
			</th>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Default Max Payment Retries', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-number">
				<input type="number" id="subscrpt_default_max_payment_retries" name="subscrpt_default_max_payment_retries" value="<?php echo esc_attr( get_option( 'subscrpt_default_max_payment_retries', '3' ) ); ?>" min="0" max="10" class="small-text" />
				<p class="description"><?php echo esc_html_e( 'Default number of automatic retry attempts for failed payments. (0 = no retries)', 'wp_subscription_pro' ); ?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Enable Payment Failure Emails', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-checkbox">
				<fieldset>
					<legend class="screen-reader-text"><span><?php echo esc_html_e( 'Enable Payment Failure Emails', 'wp_subscription_pro' ); ?></span></legend>
					<label for="subscrpt_enable_payment_failure_emails">
						<input class="wp-subscription-toggle" name="subscrpt_enable_payment_failure_emails" id="subscrpt_enable_payment_failure_emails" type="checkbox" value="1" <?php checked( get_option( 'subscrpt_enable_payment_failure_emails', '1' ), '1' ); ?>>
						<span class="wp-subscription-toggle-ui" aria-hidden="true"></span>

						<?php echo esc_html_e( 'Send email notifications when payments fail', 'wp_subscription_pro' ); ?>
					</label>
					<p class="description"><?php echo esc_html_e( 'Customers will receive emails about failed payments and retry attempts.', 'wp_subscription_pro' ); ?></p>
				</fieldset>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Payment Failure Email Delay (Hours)', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-number">
				<input type="number" id="subscrpt_payment_failure_email_delay" name="subscrpt_payment_failure_email_delay" value="<?php echo esc_attr( get_option( 'subscrpt_payment_failure_email_delay', '24' ) ); ?>" min="1" max="168" class="small-text" />
				<p class="description"><?php echo esc_html_e( 'Delay before sending payment failure emails to avoid spam during temporary issues. (1-168 hours)', 'wp_subscription_pro' ); ?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Grace Period (Days)', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-number">
				<input type="number" id="subscrpt_default_payment_grace_period" name="subscrpt_default_payment_grace_period" value="<?php echo esc_attr( get_option( 'subscrpt_default_payment_grace_period', '7' ) ); ?>" min="0" max="30" class="small-text" />
				<p class="description"><?php echo esc_html_e( 'Days to maintain access after subscriptions expires. (0 = No grace period. Max 30 days)', 'wp_subscription_pro' ); ?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Enable Grace Period Notifications', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-checkbox">
				<fieldset>
					<legend class="screen-reader-text"><span><?php echo esc_html_e( 'Enable Grace Period Notifications', 'wp_subscription_pro' ); ?></span></legend>
					<label for="subscrpt_enable_grace_period_notifications">
						<input class="wp-subscription-toggle" name="subscrpt_enable_grace_period_notifications" id="subscrpt_enable_grace_period_notifications" type="checkbox" value="1" <?php checked( get_option( 'subscrpt_enable_grace_period_notifications', '1' ), '1' ); ?>>
						<span class="wp-subscription-toggle-ui" aria-hidden="true"></span>
						
						<?php echo esc_html_e( 'Send notifications during grace period', 'wp_subscription_pro' ); ?>
					</label>
					<p class="description"><?php echo esc_html_e( 'Customers will receive warnings before their access is suspended.', 'wp_subscription_pro' ); ?></p>
				</fieldset>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Grace Period Warning (Days Before)', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-number">
				<input type="number" id="subscrpt_grace_period_warning_days" name="subscrpt_grace_period_warning_days" value="<?php echo esc_attr( get_option( 'subscrpt_grace_period_warning_days', '2' ) ); ?>" min="1" max="7" class="small-text" />
				<p class="description"><?php echo esc_html_e( 'Send warning emails this many days before grace period expires. (1-7 days)', 'wp_subscription_pro' ); ?></p>
			</td>
		</tr>

		<!-- API Settings -->
		<tr>
			<th scope="row" colspan="2">
				<h2><?php esc_html_e( 'API Settings', 'wp_subscription_pro' ); ?></h2>
			</th>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'API Key', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-text">
				<input type="text" id="wpsubscription_api_key" name="wpsubscription_api_key" value="<?php echo esc_attr( get_option( 'wpsubscription_api_key', '' ) ); ?>" readonly class="regular-text" />
				<button type="button" class="button" id="regenerate_api_key"><?php esc_html_e( 'Regenerate API Key', 'wp_subscription_pro' ); ?></button>
				<p class="description"><?php esc_html_e( 'API key for external integrations. Keep this secure.', 'wp_subscription_pro' ); ?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" class="titledesc"><?php echo esc_html_e( 'Enable API', 'wp_subscription_pro' ); ?></th>
			<td class="forminp forminp-checkbox">
				<fieldset>
					<legend class="screen-reader-text"><span><?php echo esc_html_e( 'Enable REST API', 'wp_subscription_pro' ); ?></span></legend>
					<label for="wpsubscription_api_enabled">
						<input class="wp-subscription-toggle" name="wpsubscription_api_enabled" id="wpsubscription_api_enabled" type="checkbox" value="on" <?php checked( get_option( 'wpsubscription_api_enabled', 'on' ), 'on' ); ?>>
						<span class="wp-subscription-toggle-ui" aria-hidden="true"></span>
						
						<?php echo esc_html_e( 'Enable REST API endpoints for subscription actions', 'wp_subscription_pro' ); ?>
					</label>
				</fieldset>
				<?php if ( get_option( 'wpsubscription_api_enabled', 'on' ) === 'on' ) : ?>
					<div style="margin-top:8px;">
						<strong><?php esc_html_e( 'REST API Endpoint URL:', 'wp_subscription_pro' ); ?></strong>
						<input type="text" readonly value="<?php echo esc_url( home_url( '/wp-json/wpsubscription/v1/action' ) ); ?>" style="width: 420px;" onclick="this.select();document.execCommand('copy');" title="Click to copy" />
						<p class="description" style="margin:4px 0 0 0;">Copy and use this endpoint for all API actions.</p>
					</div>
				<?php endif; ?>
			</td>
		</tr>
		<?php
	}

	public function get_settings_sections() {
		return array(
			array(
				'id'    => 'wpsubscription_general',
				'title' => __( 'General Settings', 'wp_subscription_pro' ),
			),
			array(
				'id'    => 'wpsubscription_payment_failure',
				'title' => __( 'Payment Failure Handling', 'wp_subscription_pro' ),
			),
			array(
				'id'    => 'wpsubscription_api',
				'title' => __( 'API Settings', 'wp_subscription_pro' ),
			),
		);
	}

	/**
	 * Generate a new API key
	 *
	 * @return string
	 */
	private function generate_api_key() {
		$api_key = get_option( 'wpsubscription_api_key' );
		if ( empty( $api_key ) ) {
			$api_key = wp_generate_password( 32, false );
			update_option( 'wpsubscription_api_key', $api_key );
		}
		return $api_key;
	}
}
