<?php

namespace SpringDevs\SubscriptionPro\Api;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Subscription Action API Handler
 *
 * @package SpringDevs\SubscriptionPro\Api
 */
class SubscriptionAction {

	/**
	 * Allowed actions
	 *
	 * @var array
	 */
	private $allowed_actions = array(
		'cancel_subscription',
		'pause_subscription',
		'resume_subscription',
		'change_status',
		'reactivate_subscription',
		'verify_api',
		'trial_end',
		'expire_subscription',
		'payment_failed',
	);

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		register_rest_route(
			'wpsubscription/v1',
			'/action',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'handle_action' ),
					'permission_callback' => array( $this, 'check_permission' ),
					'args'                => array(
						'action'          => array(
							'required'          => true,
							'type'              => 'string',
							'enum'              => $this->allowed_actions,
							'sanitize_callback' => 'sanitize_text_field',
						),
						'subscription_id' => array(
							'required'          => false,
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
						),
						'api_key'         => array(
							'required'          => true,
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_text_field',
						),
						'data'            => array(
							'required' => false,
							'type'     => 'object',
							'default'  => array(),
						),
					),
				),
			)
		);
	}

	/**
	 * Check if the request has permission to access the endpoint
	 *
	 * @param WP_REST_Request $request
	 * @return bool|WP_Error
	 */
	public function check_permission( $request ) {
		$api_key = $request->get_param( 'api_key' );

		// Check if API key is valid
		if ( ! $this->validate_api_key( $api_key ) ) {
			return new WP_Error(
				'rest_forbidden',
				__( 'Invalid API key', 'wp_subscription_pro' ),
				array( 'status' => 403 )
			);
		}

		return true;
	}

	/**
	 * Handle the subscription action
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_action( $request ) {
		$action          = $request->get_param( 'action' );
		$subscription_id = $request->get_param( 'subscription_id' );
		$data            = $request->get_param( 'data' );

		// Special case for verify_api: just check API key and return
		if ( $action === 'verify_api' ) {
			$api_key = $request->get_param( 'api_key' );
			if ( $this->validate_api_key( $api_key ) ) {
				return new WP_REST_Response(
					array(
						'status'  => 'success',
						'message' => 'API key is valid.',
					),
					200
				);
			} else {
				return new WP_Error(
					'rest_forbidden',
					__( 'Invalid API key', 'wp_subscription_pro' ),
					array( 'status' => 403 )
				);
			}
		}

		// Validate subscription exists (skip for verify_api)
		if ( ! $this->validate_subscription( $subscription_id ) ) {
			return new WP_Error(
				'not_found',
				__( 'Subscription not found', 'wp_subscription_pro' ),
				array( 'status' => 404 )
			);
		}

		// Validate action-specific data
		$validation_result = $this->validate_action_data( $action, $data );
		if ( is_wp_error( $validation_result ) ) {
			return $validation_result;
		}

		try {
			// Log the incoming request
			$this->log_request( $action, $subscription_id, $data );

			// Initialize subscription status handler
			$status_handler = new \SpringDevs\SubscriptionPro\Illuminate\SubscriptionStatusHandler();

			// Map API actions to subscription status handlers
			switch ( $action ) {
				case 'cancel_subscription':
					$status_handler->handle_cancel( $subscription_id );
					do_action( 'subscrpt_subscription_cancelled', $subscription_id );
					break;

				case 'pause_subscription':
					$status_handler->handle_pause( $subscription_id );
					do_action( 'subscrpt_subscription_pending', $subscription_id );
					break;

				case 'resume_subscription':
					$status_handler->handle_resume( $subscription_id );
					do_action( 'subscrpt_subscription_activated', $subscription_id );
					break;

				case 'change_status':
					if ( isset( $data['status'] ) ) {
						$status_handler->handle_status_change( $subscription_id, $data['status'] );
						do_action( 'subscrpt_subscription_status_changed', $subscription_id, get_post_status( $subscription_id ), $data['status'] );
					}
					break;

				case 'reactivate_subscription':
					$status_handler->handle_reactivation( $subscription_id );
					do_action( 'subscrpt_subscription_activated', $subscription_id );
					break;

				case 'trial_end':
					do_action( 'subscrpt_subscription_trial_ended', $subscription_id );
					break;

				case 'expire_subscription':
					do_action( 'subscrpt_subscription_expired', $subscription_id );
					break;

				case 'payment_failed':
					do_action( 'subscrpt_subscription_payment_failed', $subscription_id );
					do_action( 'subscrpt_split_payment_failed', $subscription_id );
					break;
			}

			// Log the successful action
			$this->log_success( $action, $subscription_id );

			return new WP_REST_Response(
				array(
					'status'          => 'success',
					'message'         => sprintf( 'Subscription action "%s" processed successfully', $action ),
					'subscription_id' => $subscription_id,
				),
				200
			);
		} catch ( \Exception $e ) {
			// Log the error
			$this->log_error( $action, $subscription_id, $e->getMessage() );

			return new WP_Error(
				'action_failed',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}

	/**
	 * Validate API key
	 *
	 * @param string $api_key
	 * @return bool
	 */
	private function validate_api_key( $api_key ) {
		$stored_api_key = get_option( 'wpsubscription_api_key' );
		return ! empty( $stored_api_key ) && $api_key === $stored_api_key;
	}

	/**
	 * Validate subscription exists
	 *
	 * @param int $subscription_id
	 * @return bool
	 */
	private function validate_subscription( $subscription_id ) {
		return get_post_type( $subscription_id ) === 'subscrpt_order';
	}

	/**
	 * Validate action-specific data
	 *
	 * @param string $action
	 * @param array  $data
	 * @return true|WP_Error
	 */
	private function validate_action_data( $action, $data ) {
		switch ( $action ) {
			case 'change_status':
				if ( ! isset( $data['status'] ) || ! in_array( $data['status'], array( 'active', 'cancelled', 'pe_cancelled', 'expired', 'pending' ) ) ) {
					return new WP_Error(
						'invalid_status',
						__( 'Invalid status value', 'wp_subscription_pro' ),
						array( 'status' => 400 )
					);
				}
				break;

			// Add more action-specific validations as needed
		}

		return true;
	}

	/**
	 * Log incoming request
	 *
	 * @param string $action
	 * @param int    $subscription_id
	 * @param array  $data
	 */
	private function log_request( $action, $subscription_id, $data ) {
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->info(
				sprintf(
					'Incoming subscription action request - Action: %s, Subscription ID: %d, Data: %s',
					$action,
					$subscription_id,
					json_encode( $data )
				),
				array( 'source' => 'subscription-pro-api' )
			);
		}
	}

	/**
	 * Log successful action
	 *
	 * @param string $action
	 * @param int    $subscription_id
	 */
	private function log_success( $action, $subscription_id ) {
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->info(
				sprintf(
					'Successfully processed subscription action - Action: %s, Subscription ID: %d',
					$action,
					$subscription_id
				),
				array( 'source' => 'subscription-pro-api' )
			);
		}
	}

	/**
	 * Log error
	 *
	 * @param string $action
	 * @param int    $subscription_id
	 * @param string $error_message
	 */
	private function log_error( $action, $subscription_id, $error_message ) {
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->error(
				sprintf(
					'Failed to process subscription action - Action: %s, Subscription ID: %d, Error: %s',
					$action,
					$subscription_id,
					$error_message
				),
				array( 'source' => 'subscription-pro-api' )
			);
		}
	}
}
