<?php

namespace SpringDevs\SubscriptionPro\Frontend;

use SpringDevs\Subscription\Illuminate\Helper;
use SpringDevs\SubscriptionPro\Illuminate\Helper as ProHelper;
use SpringDevs\SubscriptionPro\Illuminate\SubscriptionSwitch;

/**
 * Class Cart
 *
 * @package SpringDevs\SubscriptionPro\Frontend
 */
class Cart {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_filter( 'woocommerce_add_cart_item_data', array( $this, 'add_to_cart_item_data' ), 10, 3 );
		add_filter( 'woocommerce_get_item_data', array( $this, 'set_line_item_meta' ), 10, 2 );
		// add_filter( 'woocommerce_cart_item_price', array( $this, 'change_price_cart_html' ), 10, 2 );
		// add_filter( 'woocommerce_cart_item_subtotal', array( $this, 'change_price_cart_html' ), 10, 2 );
		add_action( 'woocommerce_check_cart_items', array( $this, 'check_cart_items' ) );
		// add_action( 'woocommerce_cart_totals_after_order_total', array( $this, 'add_rows_order_total' ) );
		// add_action( 'woocommerce_review_order_after_order_total', array( $this, 'add_rows_order_total' ) );
		add_filter( 'subscrpt_block_simple_cart_item_data', array( $this, 'add_pro_fields_data' ), 10, 2 );
		add_action( 'woocommerce_cart_calculate_fees', array( $this, 'add_cart_fee' ) );
		add_action( 'woocommerce_before_calculate_totals', array( $this, 'add_calculation_price_filter' ) );
		add_action( 'woocommerce_calculate_totals', array( $this, 'remove_calculation_price_filter' ) );
		add_action( 'woocommerce_after_calculate_totals', array( $this, 'remove_calculation_price_filter' ) );
	}

	/**
	 * Add product meta on cart item -- variable product.
	 *
	 * @param array $cart_item_data cart_item_data.
	 * @param int   $product_id Product ID.
	 * @param int   $variation_id Variation ID.
	 *
	 * @return array
	 */
	public function add_to_cart_item_data( $cart_item_data, $product_id, $variation_id ) {
		$product = wc_get_product( $product_id );
		if ( ! $product->is_type( 'variable' ) ) {
			return $cart_item_data;
		}
		$variation = sdevs_get_subscription_product( $variation_id );
		if ( $variation->is_enabled() ) :
			$timing_per                 = $variation->get_timing_per();
			$timing_option              = $variation->get_timing_option();
			$type                       = Helper::get_typos( $timing_per, $timing_option );
			$subscription_data          = array();
			$subscription_data['time']  = $timing_per;
			$subscription_data['type']  = $type;
			$trial_timing_per           = $variation->get_trial_timing_per();
			$subscription_data['trial'] = $variation->has_trial() ? $trial_timing_per . ' ' . Helper::get_typos( $trial_timing_per, $variation->get_trial_timing_option() ) : null;

			$subscription_data['signup_fee'] = ( 0 === $variation->get_signup_fee() ) ? null : $variation->get_signup_fee();
			$subscription_data['per_cost']   = $variation->get_price();
			$cart_item_data['subscription']  = $subscription_data;
		endif;

		return $cart_item_data;
	}

	/**
	 * Remove filter after calculate calculation.
	 *
	 * @return void
	 */
	public function remove_calculation_price_filter() {
		remove_filter( 'woocommerce_product_variation_get_price', array( $this, 'set_prices_for_calculation' ), 100 );
	}

	/**
	 * Add filter before cart calculation.
	 *
	 * @return void
	 */
	public function add_calculation_price_filter() {
		add_filter( 'woocommerce_product_variation_get_price', array( $this, 'set_prices_for_calculation' ), 100, 2 );
	}

	/**
	 * Return 0 if product has trial.
	 *
	 * @param float       $price Price.
	 * @param \WC_Product $product Product object.
	 *
	 * @return float
	 */
	public function set_prices_for_calculation( $price, $product ) {
		$product = sdevs_get_subscription_product( $product );
		if ( $product->is_enabled() && $product->is_type( 'variation' ) ) {
			if ( $product->has_trial() ) {
				return 0;
			}
		}

		return $price;
	}

	/**
	 * Add signup fee if available
	 *
	 * @param mixed $cart Cart.
	 *
	 * @return void
	 */
	public function add_cart_fee( $cart ) {
		$cart_items = WC()->cart->cart_contents;
		$signup_fee = 0;
		foreach ( $cart_items as $cart_item ) {
			if ( isset( $cart_item['subscription'] ) && $cart_item['subscription']['signup_fee'] ) {
				$signup_fee += (float) $cart_item['subscription']['signup_fee'];
			}
		}
		if ( $signup_fee > 0 ) {
			$cart->add_fee( 'Signup Fee', $signup_fee );
		}
	}

	/**
	 * Insert pro fields data on cart item data.
	 *
	 * @param array       $subscription_data Subscription array on cart item data.
	 * @param \WC_Product $product Product Object.
	 *
	 * @return array
	 */
	public function add_pro_fields_data( $subscription_data, $product ) {
		$subscription_data['time'] = $product->get_timing_per();
		$subscription_data['type'] = Helper::get_typos( $subscription_data['time'], $product->get_timing_option() );

		$signup_fee = $product->get_signup_fee();
		if ( $signup_fee > 0 ) {
			$subscription_data['signup_fee'] = $signup_fee;
		}

		return $subscription_data;
	}

	/**
	 * Display "Recurring totals" on cart
	 *
	 * @return void
	 */
	public function add_rows_order_total() {
		$cart_items = WC()->cart->cart_contents;
		$recurrs    = array();
		foreach ( $cart_items as $cart_item ) {
			$product = sdevs_get_subscription_product( $cart_item['data'] );
			if ( $product->is_enabled() && isset( $cart_item['subscription'] ) ) :
				$meta_time = $product->get_timing_per();
				$time      = 1 === $meta_time ? null : $meta_time . '-';
				$type      = Helper::get_typos( $meta_time, $product->get_timing_option() );

				$price_html = wc_price( $cart_item['subscription']['per_cost'] * $cart_item['quantity'] ) . '/ ' . $time . $type;

				$trial      = null;
				$start_date = null;
				if ( $product->has_trial() ) {
					$meta_trial_per = $product->get_trial_timing_per();
					$trial          = $meta_trial_per . ' ' . Helper::get_typos( $meta_trial_per, $product->get_trial_timing_option() );
					$start_date     = Helper::start_date( $trial );
				}

				$trial_status = null !== $trial;
				$next_date    = Helper::next_date(
					$meta_time . ' ' . $type,
					$trial
				);
				$recurrs[]    = array(
					'trial'      => $trial_status,
					'price_html' => $price_html,
					'start_date' => $start_date,
					'next_date'  => $next_date,
				);
			endif;
		}

		if ( 0 === count( $recurrs ) ) {
			return;
		}

		?>
		<tr class="recurring-total">
			<th><?php esc_html_e( 'Recurring totals', 'wp_subscription_pro' ); ?></th>
			<td data-title="<?php esc_attr_e( 'Recurring totals', 'wp_subscription_pro' ); ?>">
				<?php foreach ( $recurrs as $recurr ) : ?>
					<?php if ( $recurr['trial'] ) : ?>
						<p>
							<span><?php echo wp_kses_post( $recurr['price_html'] ); ?></span><br/>
							<small><?php esc_html_e( 'First billing on', 'wp_subscription_pro' ); ?>
								: <?php echo esc_html( $recurr['start_date'] ?? wp_date( get_option( 'date_format' ) ) ); ?></small>
						</p>
					<?php else : ?>
						<p>
							<span><?php echo wp_kses_post( $recurr['price_html'] ); ?></span><br/>
							<small><?php esc_html_e( 'Next billing on', 'wp_subscription_pro' ); ?>
								: <?php echo wp_kses_post( $recurr['next_date'] ); ?></small>
						</p>
					<?php endif; ?>
				<?php endforeach; ?>
			</td>
		</tr>
		<?php
	}

	/**
	 * Check cart items if it's valid or not?
	 *
	 * @return void
	 */
	public function check_cart_items() {
		$cart_items = WC()->cart->cart_contents;
		if ( is_array( $cart_items ) ) {
			foreach ( $cart_items as $key => $value ) {
				$product = sdevs_get_subscription_product( $value['data'] );
				if ( isset( $value['subscription'] ) ) {
					// $condition  = $value['subscription']['time'] !== $product->get_timing_per();
					// $condition1 = Helper::get_typos( $product->get_timing_per(), $product->get_timing_option() ) !== $value['subscription']['type'];
					// $condition2 = $product->get_trial() !== $value['subscription']['trial'];
					// $condition3 = ( 0 === $product->get_signup_fee() ? null : $product->get_signup_fee() ) !== $value['subscription']['signup_fee'];

					if ( $value['subscription']['time'] !== $product->get_timing_per() || Helper::get_typos( $product->get_timing_per(), $product->get_timing_option() ) !== $value['subscription']['type'] || $product->get_trial() !== $value['subscription']['trial'] || ( 0 === $product->get_signup_fee() ? null : $product->get_signup_fee() ) !== $value['subscription']['signup_fee'] ) {
						// remove the item.
						wc_add_notice( __( 'An item which is no longer available was removed from your cart.', 'wp_subscription_pro' ), 'error' );
						WC()->cart->remove_cart_item( $key );
					}
				} elseif ( $product->get_meta( '_subscrpt_enabled' ) ) {
					// remove the item.
					wc_add_notice( __( 'An item which is no longer available was removed from your cart.', 'wp_subscription_pro' ), 'error' );
					WC()->cart->remove_cart_item( $key );
				}
			}
		}
	}

	/**
	 * Display formatted price on cart.
	 *
	 * @param string $price price.
	 * @param array  $cart_item cart item.
	 *
	 * @return string
	 */
	public function change_price_cart_html( $price, $cart_item ) {
		$product = wc_get_product( $cart_item['product_id'] );
		if ( $product->is_type( 'variable' ) ) {
			return $price;
		}

		$enabled = $product->get_meta( '_subscrpt_enabled' );
		if ( $enabled ) :
			$meta_time       = $product->get_meta( '_subscrpt_timing_per' );
			$meta_trial_time = $product->get_meta( '_subscrpt_trial_timing_per' );
			$meta_signup_fee = $product->get_meta( '_subscrpt_signup_fee' );

			$time            = 1 === intval( $meta_time ) ? null : $meta_time . '-';
			$type            = Helper::get_typos( $meta_time, $product->get_meta( '_subscrpt_timing_option' ) );
			$has_trial       = Helper::check_trial( $product->get_id() );
			$trial           = null;
			$signup_fee_html = null;
			if ( ! empty( $meta_trial_time ) && $meta_trial_time > 0 && $has_trial ) {
				$trial = '<br/> + Get ' . $meta_trial_time . ' ' . Helper::get_typos( $meta_trial_time, $product->get_meta( '_subscrpt_trial_timing_option' ) ) . ' free trial!';
				if ( '' !== $meta_signup_fee ) {
					$signup_fee_html = '<br/> + Signup fee of ' . wc_price( $meta_signup_fee );
				}
			}
			$price_html = $price . '/ ' . $time . $type . $signup_fee_html . $trial;

			return $price_html;
		else :
			return $price;
		endif;
	}

	/**
	 * Set line item for display meta details.
	 *
	 * @param array $cart_item_data Cart Item Data.
	 * @param array $cart_item Cart Item.
	 *
	 * @return array
	 */
	public function set_line_item_meta( $cart_item_data, $cart_item ) {
		if ( isset( $cart_item['subscription'] ) ) {
			if ( $cart_item['subscription']['signup_fee'] ) {
				$cart_item_data[] = array(
					'key'    => __( 'Signup Fee', 'wp_subscription_pro' ),
					'value'  => wc_price( $cart_item['subscription']['signup_fee'] ),
					'hidden' => true,
					'__experimental_woocommerce_blocks_hidden' => false,
				);
			}
		}

		return $cart_item_data;
	}
}
