<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

/**
 * Class GatewayLimitEnforcer
 * Enforces renewal limits for Stripe/PayPal renewals and adds notifications.
 */
class GatewayLimitEnforcer {

	public function __construct() {
		// Hook after renewal order is created (Stripe/PayPal/manual)
		add_action( 'subscrpt_after_create_renew_order', array( $this, 'enforce_gateway_renewal_limit' ), 5, 4 );
		// Show user notification on My Account page
		add_action( 'woocommerce_before_my_account', array( $this, 'show_user_renewal_limit_notice' ) );
	}

	/**
	 * Enforce renewal limit for Stripe/PayPal/manual renewals.
	 *
	 * @param \WC_Order $new_order
	 * @param \WC_Order $old_order
	 * @param int       $subscription_id
	 * @param bool      $is_early
	 */
	public function enforce_gateway_renewal_limit( $new_order, $old_order, $subscription_id, $is_early = false ) {
		if ( ! class_exists( 'SpringDevs\\SubscriptionPro\\Illuminate\\LimitChecker' ) ) {
			return;
		}
		$limit_check = LimitChecker::check_renewal_limit( $subscription_id );
		if ( $limit_check !== true ) {
			// Expire the subscription and add order note
			wp_update_post(
				[
					'ID'          => $subscription_id,
					'post_status' => 'expired',
				]
			);
			update_post_meta( $subscription_id, '_subscrpt_expired_reason', 'renewal_limit_reached' );
			$new_order->update_status( 'cancelled', __( 'Renewal blocked: subscription renewal limit reached.', 'wp_subscription_pro' ) );
			// Add WooCommerce notice for user
			if ( function_exists( 'wc_add_notice' ) ) {
				wc_add_notice( $limit_check, 'error' );
			}
			// Add admin order note
			$new_order->add_order_note( __( 'Renewal blocked: subscription renewal limit reached.', 'wp_subscription_pro' ) );
		} else {
			// If approaching limit, show a warning
			$remaining = LimitChecker::get_remaining_renewals( $subscription_id );
			if ( $remaining !== 'unlimited' && $remaining <= 1 ) {
				if ( function_exists( 'wc_add_notice' ) ) {
					wc_add_notice(
						// translators: %d: number of remaining renewals
						sprintf( __( 'Warning: Only %d renewal(s) remaining for this subscription.', 'wp_subscription_pro' ), $remaining ),
						'notice'
					);
				}
			}
		}

		// Cancel Stripe subscription if payment method is Stripe
		if ( $old_order->get_payment_method() === 'stripe' ) {
			$stripe_subscription_id = $old_order->get_meta( '_stripe_subscription_id' );
			if ( $stripe_subscription_id ) {
				try {
					if ( class_exists( 'WC_Stripe_API' ) ) {
						\WC_Stripe_API::request(
							array( 'cancel_at_period_end' => true ),
							"subscriptions/{$stripe_subscription_id}",
							'POST'
						);
					}
				} catch ( \Exception $e ) {
					error_log( 'Failed to cancel Stripe subscription: ' . $e->getMessage() );
				}
			}
		}
	}

	/**
	 * Show user notification on My Account page if any active subscription is approaching or at limit.
	 */
	public function show_user_renewal_limit_notice() {
		if ( ! is_user_logged_in() ) {
			return;
		}
		$user_id = get_current_user_id();
		$args    = array(
			'post_type'      => 'subscrpt_order',
			'post_status'    => 'active',
			'author'         => $user_id,
			'fields'         => 'ids',
			'posts_per_page' => 10,
		);
		$subs    = get_posts( $args );
		foreach ( $subs as $sub_id ) {
			$remaining = LimitChecker::get_remaining_renewals( $sub_id );

			// Get payment type using consistent helper function
			$payment_type = function_exists( 'subscrpt_get_payment_type' ) ? subscrpt_get_payment_type( $sub_id ) : 'recurring';

			// Check completion status for split payments
			$is_split_payment = ( $payment_type === 'split_payment' );
			$is_completed     = false;

			if ( $is_split_payment ) {
				// Use simple completion check for warnings
				$max_payments  = subscrpt_get_max_payments( $sub_id );
				$payments_made = subscrpt_count_payments_made( $sub_id );
				$is_completed  = ( $max_payments && $payments_made >= intval( $max_payments ) );

				// Debug logging (can be removed in production)
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					error_log(
						sprintf(
							'[GatewayLimitEnforcer] Subscription #%d: remaining=%s, max_payments=%s, payments_made=%d, is_completed=%s',
							$sub_id,
							$remaining,
							$max_payments,
							$payments_made,
							$is_completed ? 'true' : 'false'
						)
					);
				}
			}

			// Skip warnings for truly completed split payments
			if ( $is_split_payment && $is_completed ) {
				continue;
			}

			if ( $remaining !== 'unlimited' && $remaining <= 1 && $remaining > 0 ) {
				$message_type = $is_split_payment ? 'payment' : 'renewal';
				wc_print_notice(
					// translators: 1: remaining count, 2: type label (payment/renewal), 3: subscription ID
					sprintf( __( 'Warning: Only %1$d %2$s(s) remaining for your subscription #%3$d.', 'wp_subscription_pro' ), $remaining, $message_type, $sub_id ),
					'notice'
				);
			}
			if ( $remaining === 0 && ! $is_completed ) {
				$message_type = $is_split_payment ? 'payment plan has been completed' : 'has reached its renewal limit and will not renew further';
				wc_print_notice(
					// translators: 1: subscription ID, 2: subscription status message
					sprintf( __( 'Your subscription #%1$d %2$s.', 'wp_subscription_pro' ), $sub_id, $message_type ),
					$is_split_payment ? 'success' : 'error'
				);
			}
		}
	}
}
