<?php
/**
 * LicenseManagerWoo integration class file.
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Integrations
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Integrations;

use SpringDevs\Subscription\Illuminate\Helper;

/**
 * Class LicenseManagerWoo
 */
class LicenseManagerWoo {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'init' ] );

		add_filter( 'subscrpt_before_saving_renewal_order', [ $this, 'disable_license_creation_for_renewal_order' ], 10, 3 );
	}

	/**
	 * Initialize the license manager.
	 */
	public function init() {
		add_action( 'subscrpt_subscription_activated', [ $this, 'maybe_activate_licenses' ], 10, 1 );

		add_action( 'subscrpt_subscription_cancelled', [ $this, 'maybe_deactivate_licenses' ], 10, 1 );
		add_action( 'subscrpt_subscription_expired', [ $this, 'maybe_deactivate_licenses' ], 10, 1 );

		// Add license meta box to subscription details page.
		add_action( 'add_meta_boxes', array( $this, 'create_meta_boxes' ) );
		add_action( 'subscrpt_after_subscription_totals', [ $this, 'render_frontend_license_info' ] );
	}

	/**
	 * Check if License Manager plugin is active.
	 */
	public function check_plugin() {
		$is_plugin_active = is_plugin_active( 'license-manager-for-woocommerce/license-manager-for-woocommerce.php' );
		$is_class_exists  = class_exists( 'LicenseManagerForWooCommerce\Models\Resources\License' );

		return $is_plugin_active && $is_class_exists;
	}

	/**
	 * Create meta boxes for subscription edit page.
	 */
	public function create_meta_boxes() {
		if ( ! $this->check_plugin() ) {
			return;
		}

		if ( ! empty( get_post_meta( get_the_ID(), '_subscrpt_lmfwc_license_order', true ) ) ) {
			add_meta_box(
				'subscription_lmfwc_license',
				__( 'License Keys', 'wp_subscription_pro' ),
				[ $this, 'render_license_meta_box' ],
				'subscrpt_order',
				'normal',
				'default'
			);
		}
	}

	/**
	 * Get parent order from subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function get_parent_order( int $subscription_id ) {
		$related_orders = Helper::get_related_orders( $subscription_id );
		$last_order     = end( $related_orders );

		if ( ! $last_order || strtolower( $last_order->type ?? '' ) !== 'new' ) {
			foreach ( $related_orders as $order ) {
				if ( strtolower( $order->type ?? '' ) === 'new' ) {
					$last_order = $order;
					break;
				}
			}
		}

		$parent_order_id = $last_order->order_id ?? 0;
		if ( $parent_order_id ) {
			$parent_order = wc_get_order( $parent_order_id );
		}
		return $parent_order_id ? $parent_order : null;
	}

	/**
	 * Get license order from subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function get_license_order_from_subscription( int $subscription_id ) {
		// Check if parent order have license.
		$parent_order = $this->get_parent_order( $subscription_id );

		$license_order = $parent_order;
		$license       = apply_filters( 'lmfwc_get_customer_license_keys', $parent_order );
		$license       = reset( $license )['keys'] ?? null;

		// If parent order doesn't have license, check other related orders.
		if ( ! $license ) {
			$related_orders = Helper::get_related_orders( $subscription_id );
			$related_orders = array_reverse( $related_orders );

			foreach ( $related_orders as $related_order ) {
				$order_id = $related_order->order_id ?? 0;
				$order    = wc_get_order( $order_id );
				$license  = apply_filters( 'lmfwc_get_customer_license_keys', $order );
				$license  = reset( $license )['keys'] ?? null;

				if ( $license ) {
					$license_order = $order;
					break;
				}
			}
		}

		return ! empty( $license_order ) ? $license_order : null;
	}

	/**
	 * Disable license creation for renewal order.
	 *
	 * @param \WC_Order $new_order New order object.
	 * @param \WC_Order $old_order Old order object.
	 * @param int       $subscription_id Subscription ID.
	 */
	public function disable_license_creation_for_renewal_order( $new_order, $old_order, $subscription_id ) {
		// Check if License Manager plugin is active.
		if ( ! $this->check_plugin() ) {
			return $new_order;
		}

		// Check for existing license.
		$existing_license_order_id = get_post_meta( $subscription_id, '_subscrpt_lmfwc_license_order', true );
		if ( empty( $existing_license_order_id ) ) {
			$license_order = $this->get_license_order_from_subscription( $subscription_id );

			if ( $license_order ) {
				$existing_license_order_id = $license_order->get_id();
				update_post_meta( $subscription_id, '_subscrpt_lmfwc_license_order', $existing_license_order_id );
			}
		}

		// Disable license creation for renewal order if license already exists.
		if ( ! empty( $existing_license_order_id ) ) {
			$new_order->update_meta_data( 'lmfwc_order_complete', 1 );

			wp_subscrpt_write_log( 'License already exists for Subscription ID: ' . $subscription_id . ' Skipping license creation for renewal Order ID: ' . $new_order->get_id() );
		}

		return $new_order;
	}

	/**
	 * Activate licenses if subscription is active.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function maybe_activate_licenses( int $subscription_id ) {
		// Check if License Manager plugin is active.
		if ( ! $this->check_plugin() ) {
			return;
		}

		// Get parent order.
		$parent_order = $this->get_parent_order( $subscription_id );
		if ( ! $parent_order ) {
			wp_subscrpt_write_log( 'No parent order found for subscription ID: ' . $subscription_id . ' Skipping license operations.' );
			return;
		}

		// Get license.
		$license_data = apply_filters( 'lmfwc_get_customer_license_keys', $parent_order );
		if ( empty( $license_data ) ) {
			wp_subscrpt_write_log( 'No license data found for order ID: ' . $parent_order->get_id() . ' Skipping license operations.' );
			return;
		}

		// Get the licenses.
		$license_keys = [];
		foreach ( $license_data as $row ) {
			foreach ( $row['keys'] as $license ) {
				$license_keys[] = $license->getDecryptedLicenseKey();
			}
		}

		// Update the license status and expiration date.
		// Get the next payment date from subscription.
		$expires_at = get_post_meta( $subscription_id, '_subscrpt_next_date', true );
		$expires_at = $expires_at ? gmdate( 'Y-m-d H:i:s', (int) $expires_at ) : null;

		foreach ( $license_keys as $license_key ) {
			$args = [
				'status'     => 3, // Possible values are: 1 for SOLD, 2 for DELIVERED, 3 for ACTIVE, and 4 for INACTIVE.
				'expires_at' => $expires_at,
				'valid_for'  => null,
			];

			lmfwc_update_license( $license_key, $args );
		}
	}

	/**
	 * Activate licenses if subscription is active.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function maybe_deactivate_licenses( int $subscription_id ) {
		// Check if License Manager plugin is active.
		if ( ! $this->check_plugin() ) {
			return;
		}

		// Get parent order.
		$parent_order = $this->get_parent_order( $subscription_id );
		if ( ! $parent_order ) {
			wp_subscrpt_write_log( 'No parent order found for subscription ID: ' . $subscription_id . ' Skipping license operations.' );
			return;
		}

		// Get license.
		$license_data = apply_filters( 'lmfwc_get_customer_license_keys', $parent_order );
		if ( empty( $license_data ) ) {
			wp_subscrpt_write_log( 'No license data found for order ID: ' . $parent_order->get_id() . ' Skipping license operations.' );
			return;
		}

		// Get the licenses.
		$license_keys = [];
		foreach ( $license_data as $row ) {
			foreach ( $row['keys'] as $license ) {
				$license_keys[] = $license->getDecryptedLicenseKey();
			}
		}

		// Update the license status.
		foreach ( $license_keys as $license_key ) {
			$args = [
				'status' => 4, // Possible values are: 1 for SOLD, 2 for DELIVERED, 3 for ACTIVE, and 4 for INACTIVE.
			];

			lmfwc_update_license( $license_key, $args );
		}
	}

	/**
	 * Render license meta box content.
	 *
	 * @param WP_Post $post The post object.
	 */
	public function render_license_meta_box( $post ) {
		$order_id     = get_post_meta( $post->ID, '_subscrpt_lmfwc_license_order', true );
		$order        = wc_get_order( $order_id );
		$license_data = ! empty( $order ) ? apply_filters( 'lmfwc_get_customer_license_keys', $order ) : [];
		$license_data = reset( $license_data );

		if ( empty( $license_data ) ) {
			echo '<p>' . esc_html_e( 'No license found.', 'wp_subscription_pro' ) . '</p>';
			return;
		}

		?>
		<style>
			.subscrpt_lmfwc_license_table td{
				vertical-align: middle;
			}

			#subscription_lmfwc_license .inside table {
				border: none;
			}
		</style>

		<table class="wp-list-table widefat fixed striped table-view-list licensekeys subscrpt_lmfwc_license_table">
			<thead>
				<tr>
					<th><?php esc_html_e( 'License Key', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Order', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Product', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Status', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Activation', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Expires At', 'wp_subscription_pro' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $license_data['keys'] ?? [] as $license ) : ?>
					<?php
						$license_key = $license->getDecryptedLicenseKey();
						$order_id    = $license->getOrderId();

						$product_id   = $license->getProductId();
						$product      = $product_id ? wc_get_product( $product_id ) : null;
						$product_name = $product ? $product->get_name() : '-';

						$status_key = $license->getStatus();
						$status_map = [
							1 => 'Sold',
							2 => 'Delivered',
							3 => 'Active',
							4 => 'Inactive',
						];
						$status     = $status_map[ $status_key ] ?? 'sold';

						$activated_count = $license->getTimesActivated() ?? 0;
						$max_count       = $license->getTimesActivatedMax() ?? '&infin;';
						$activation      = $activated_count . ' / ' . $max_count;

						$expires_at = $license->getExpiresAt();
						$expires_at = $expires_at ? gmdate( 'F j, Y - g:i A', strtotime( $expires_at ) ) : '-';

						$status_class      = 'lmfwc-status ' . strtolower( $status );
						$status_icon_class = 'dashicons dashicons-marker';
						if ( 'Sold' === $status ) {
							$status_icon_class = 'dashicons dashicons-yes';
						} elseif ( 'Delivered' === $status ) {
							$status_icon_class = 'lmfwc-icons delivered';
						}

						?>

					<tr>
						<td>
							<code class="lmfwc-placeholder"><?php echo esc_html( $license_key ); ?></code>
						</td>
						<td>
							<a href="<?php echo esc_url( get_edit_post_link( $order_id ) ); ?>" target="_blank">
								#<?php echo esc_html( $order_id ); ?>
							</a>
						</td>
						<td>
							<?php echo esc_html( $product_name ); ?>
						</td>
						<td>
							<div class="<?php echo esc_attr( $status_class ); ?>" style="width: 100px;">
								<span class="<?php echo esc_attr( $status_icon_class ); ?>"></span>
								<?php echo esc_html( $status ); ?>
							</div>
						</td>
						<td>
							<?php echo esc_html( $activation ); ?>
						</td>
						<td>
							<?php echo esc_html( $expires_at ); ?>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php
	}


	/**
	 * Render license info on subscription details page.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function render_frontend_license_info( int $subscription_id ) {
		$order_id     = get_post_meta( $subscription_id, '_subscrpt_lmfwc_license_order', true );
		$order        = wc_get_order( $order_id );
		$license_data = ! empty( $order ) ? apply_filters( 'lmfwc_get_customer_license_keys', $order ) : [];

		if ( empty( $license_data ) ) {
			return;
		}

		?>
		<style>
			.lmfwc-myaccount-license-key {
				font-family: Consolas, monospace;
				font-size: 13px;
			}
		</style>

		<h2><?php echo esc_html_e( 'Related Licenses', 'wp_subscription_pro' ); ?></h2>

		<?php foreach ( $license_data as $license ) : ?>
			<table class="woocommerce-table woocommerce-table--order-details shop_table order_details">
				<thead>
					<tr>
						<th><?php echo esc_html( $license['name'] ?? 'License Key' ); ?></th>
						<th></th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $license['keys'] ?? [] as $key ) : ?>
						<tr>
							<td>
								<span class="lmfwc-myaccount-license-key">
									<?php echo esc_html( $key->getDecryptedLicenseKey() ); ?>
								</span>
							</td>
							
							<td>
								<?php if ( $key->getExpiresAt() ) : ?>
									<span class="lmfwc-myaccount-license-key">
										Valid until
										<br/>
										<b>
											<?php echo esc_html( gmdate( 'F j, Y  g:i A', strtotime( $key->getExpiresAt() ) ) ); ?>
										</b>
									</span>
								<?php endif; ?>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		<?php endforeach; ?>

		<?php
	}
}
