<?php
/**
 * TutorLMS integration class file.
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Integrations
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Integrations;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

define( 'TUTOR_WPSCRIPTION_FILE', __FILE__ );
define( 'TUTOR_WPSCRIPTION_VERSION', '1.0.0' );

/**
 * Class TutorLMS
 */
class TutorLMS {
	//phpcs:disable
	public $version = TUTOR_WPSCRIPTION_VERSION;
	public $path;
	public $url;
	public $basename;
	private $paid_memberships_pro;
	//phpcs:enable

	/**
	 * Initialize the class
	 */
	public function __construct() {
		// Activate enrollment if subscription is active
		add_action( 'subscrpt_subscription_activated', [ $this, 'activate_enrollment' ] );

		// Cancel enrollment if subscription is inactive
		add_action( 'subscrpt_subscription_expired', [ $this, 'cancel_enrollment' ] );
		add_action( 'subscrpt_subscription_cancelled', [ $this, 'cancel_enrollment' ] );
		add_action( 'subscrpt_grace_period_ended', [ $this, 'cancel_enrollment' ] );

		// Remove 'Start Learning' button if subscription is inactive
		add_filter( 'tutor_is_enrolled', array( $this, 'tutor_is_enrolled' ), 10, 3 );

		// Modify add to cart price
		add_filter( 'tutor_course_details_wc_add_to_cart_price', array( $this, 'modify_cart_price_text' ), 10, 2 );

		// Disable course gifting if subscription product
		add_filter( 'tutor_can_gift_course', array( $this, 'disable_gift_course' ), 10, 2 );
	}

	/**
	 * Check if License Manager plugin is active.
	 */
	public function check_plugin() {
		$is_plugin_active = is_plugin_active( 'tutor/tutor.php' );
		$is_class_exists  = class_exists( 'TUTOR\Tutor' );

		return $is_plugin_active && $is_class_exists;
	}

	/**
	 * Modify cart price text
	 *
	 * @param string     $price Price text.
	 * @param WC_Product $product WooCommerce product object.
	 */
	public function modify_cart_price_text( $price, $product ) {
		$product = sdevs_get_subscription_product( $product );

		if ( ! $product || ! $product->is_enabled() ) {
			return $price;
		}

		ob_start();
		?>
			<div>
				<span class="tutor-fs-4 tutor-fw-bold tutor-color-black">
					<?php echo $product->get_price_html(); //phpcs:ignore ?>
				</span>
			</div>
		</div>
		<div class="tutor-color-muted"></div>
		<?php
		$price = ob_get_clean();

		return $price;
	}

	/**
	 * Filter tutor_is_enrolled
	 *
	 * @param mixed $get_enrolled_info  enrollment info.
	 * @param int   $course_id  tutor course id.
	 * @param int   $user_id  user id.
	 */
	public function tutor_is_enrolled( $get_enrolled_info, $course_id, $user_id ) {
		$product_id = tutor_utils()->get_course_product_id( $course_id );

		if ( $product_id ) {
			$subscription_args = array(
				'status'     => [ 'active', 'pe_cancelled' ],
				'user_id'    => $user_id,
				'product_id' => $product_id,
			);
			$subscriptions     = FreeHelper::get_subscriptions( $subscription_args );

			return ! empty( $subscriptions ) ? $get_enrolled_info : false;
		}

		return $get_enrolled_info;
	}

	/**
	 * Activate enrollment
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function activate_enrollment( $subscription_id ) {
		if ( ! $this->check_plugin() ) {
			return;
		}

		$parent_order = FreeHelper::get_parent_order( $subscription_id );

		$order_id       = $parent_order ? $parent_order->get_id() : 0;
		$is_tutor_order = tutor_utils()->is_tutor_order( $order_id );

		if ( $is_tutor_order ) {
			$enrolled_courses = tutor_utils()->get_course_enrolled_ids_by_order_id( $order_id );
			$enrolled_ids     = tutor_utils()->count( $enrolled_courses ) ? array_column( $enrolled_courses, 'enrolled_id' ) : [];

			if ( tutor_utils()->count( $enrolled_ids ) ) {
				tutor_utils()->update_enrollments( 'complete', $enrolled_ids );
			}
		}
	}

	/**
	 * Cancel enrollment
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function cancel_enrollment( $subscription_id ) {
		if ( ! $this->check_plugin() ) {
			return;
		}

		$subscription_data = FreeHelper::get_subscription_data( $subscription_id );

		// Do not cancel if in grace period.
		if ( isset( $subscription_data['grace_period'] ) && $subscription_data['grace_period']['remaining_days'] ?? 0 > 0 ) {
			return;
		}

		$parent_order = FreeHelper::get_parent_order( $subscription_id );

		$order_id       = $parent_order ? $parent_order->get_id() : 0;
		$is_tutor_order = tutor_utils()->is_tutor_order( $order_id );

		if ( $is_tutor_order ) {
			$enrolled_courses = tutor_utils()->get_course_enrolled_ids_by_order_id( $order_id );
			$enrolled_ids     = tutor_utils()->count( $enrolled_courses ) ? array_column( $enrolled_courses, 'enrolled_id' ) : [];

			if ( tutor_utils()->count( $enrolled_ids ) ) {
				tutor_utils()->update_enrollments( 'cancel', $enrolled_ids );
			}
		}
	}

	/**
	 * Disable gift course
	 *
	 * @param bool $can_gift can gift.
	 * @param int  $course_id course id.
	 */
	public function disable_gift_course( $can_gift, $course_id ) {
		$product_id = tutor_utils()->get_course_product_id( $course_id );

		$product = wc_get_product( $product_id );
		$product = $product ? sdevs_get_subscription_product( $product ) : $product;

		if ( $product && $product->is_enabled() ) {
			$can_gift = false;
		}

		return $can_gift;
	}
}
