<?php
/**
 * WPSoftwareLicense integration class file.
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Integrations
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Integrations;

use SpringDevs\Subscription\Illuminate\Helper;

/**
 * Class WPSoftwareLicense
 */
class WPSoftwareLicense {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'init' ] );
	}

	/**
	 * Initialize the license manager.
	 */
	public function init() {
		// Add license meta box to subscription details page.
		add_action( 'add_meta_boxes', array( $this, 'create_meta_boxes' ) );
		add_action( 'subscrpt_after_subscription_totals', [ $this, 'render_frontend_license_info' ] );
	}

	/**
	 * Check if License Manager plugin is active.
	 */
	public function check_plugin() {
		$is_plugin_active = is_plugin_active( 'software-license/software-license.php' );
		$is_class_exists  = class_exists( 'WOO_SL' );

		return $is_plugin_active && $is_class_exists;
	}

	/**
	 * Create meta boxes for subscription edit page.
	 */
	public function create_meta_boxes() {
		if ( ! $this->check_plugin() ) {
			return;
		}

		$licenses = $this->get_licenses_from_subscription( get_the_ID() );
		if ( ! empty( $licenses ) ) {
			add_meta_box(
				'subscription_wpsfl_license',
				__( 'License Keys', 'wp_subscription_pro' ),
				[ $this, 'render_license_meta_box' ],
				'subscrpt_order',
				'normal',
				'default'
			);
		}
	}

	/**
	 * Get parent order from subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function get_parent_order( int $subscription_id ) {
		$related_orders = Helper::get_related_orders( $subscription_id );
		$last_order     = end( $related_orders );

		if ( ! $last_order || strtolower( $last_order->type ?? '' ) !== 'new() ' ) {
			foreach ( $related_orders as $order ) {
				if ( strtolower( $order->type ?? '' ) === 'new() ' ) {
					$last_order = $order;
					break;
				}
			}
		}

		$parent_order_id = $last_order->order_id ?? 0;
		if ( $parent_order_id ) {
			$parent_order = wc_get_order( $parent_order_id );
		}
		return $parent_order_id ? $parent_order : null;
	}

	/**
	 * Get license order from subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function get_license_order_from_subscription( int $subscription_id ) {
		// Check if parent order have license.
		$parent_order = $this->get_parent_order( $subscription_id );

		$license_order    = $parent_order;
		$license_order_id = $license_order ? $license_order->get_id() : 0;

		$licenses = \WOO_SL_functions::get_order_licence_details( $license_order_id );

		// If parent order doesn't have license, check other related orders .
		if ( empty( $licenses ) ) {
			$related_orders = Helper::get_related_orders( $subscription_id );
			$related_orders = array_reverse( $related_orders );

			foreach ( $related_orders as $related_order ) {
				$order_id = $related_order->order_id ?? 0;
				$order    = wc_get_order( $order_id );

				$licenses = \WOO_SL_functions::get_order_licence_details( $order_id );

				if ( $licenses ) {
					$license_order = $order;
					break;
				}
			}
		}

		return ! empty( $license_order ) ? $license_order : null;
	}

	/**
	 * Get licenses from subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function get_licenses_from_subscription( int $subscription_id ) {
		$license_order = $this->get_license_order_from_subscription( $subscription_id );
		$order_id      = $license_order ? $license_order->get_id() : 0;

		$all_licenses = \WOO_SL_functions::get_order_licence_details( $order_id );

		if ( empty( $all_licenses ) ) {
			return [];
		}

		$processed_licenses = [];
		foreach ( $all_licenses as $item_licenses ) {
			foreach ( $item_licenses as $license ) {
				$order_item_id = $license->order_item_id ?? 0;
				$group_id      = $license->group_id ?? 0;
				$group_title   = $license->license_data['group_title'] ?? '';

				$license_keys = (array) \WOO_SL_functions::get_order_product_generated_keys( $order_id, $order_item_id, $group_id );

				$processed_licenses[ $group_title ] = $license_keys;
			}
		}

		return $processed_licenses;
	}

	/**
	 * Render license meta box content.
	 *
	 * @param WP_Post $post The post object.
	 */
	public function render_license_meta_box( $post ) {
		$all_licenses = $this->get_licenses_from_subscription( $post->ID );

		$licenses = [];
		foreach ( $all_licenses as $group_title => $group_licenses ) {
			foreach ( $group_licenses as $license ) {
				$license->group_title = $group_title;
				$licenses[]           = $license;
			}
		}

		?>
		<style>
			.subscrpt_wpsfl_license_table td{
				vertical-align: middle;
			}

			#subscription_wpsfl_license .inside table {
				border: none;
			}
		</style>

		<table class="wp-list-table widefat fixed striped table-view-list licensekeys subscrpt_wpsfl_license_table">
			<thead>
				<tr>
					<th><?php esc_html_e( 'License Key', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'License Group', 'wp_subscription_pro' ); ?></th>
					<th style="width: 100px;"><?php esc_html_e( 'Order', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Product', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Created At', 'wp_subscription_pro' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $licenses as $license ) : ?>
					<?php
					$group_title = $license->group_title;

					$license_key = $license->licence;

					$order_id      = $license->order_id;
					$order_item_id = $license->order_item_id;

					$order        = wc_get_order( $order_id );
					$order_item   = $order ? $order->get_item( $order_item_id ) : null;
					$product_name = $order_item ? $order_item->get_name() : '-';

					$created_at = $license->created;
					$created_at = $created_at ? gmdate( 'F j, Y - g:i A', strtotime( $created_at ) ) : '-';
					?>

					<tr>
						<td>
							<code><?php echo esc_html( $license_key ); ?></code>
						</td>
						<td>
							<?php echo esc_html( $group_title ); ?>
						</td>
						<td>
							<a href="<?php echo esc_url( get_edit_post_link( $order_id ) ); ?>" target="_blank">
								#<?php echo esc_html( $order_id ); ?>
							</a>
						</td>
						<td>
							<?php echo esc_html( $product_name ); ?>
						</td>
						<td>
							<?php echo esc_html( $created_at ); ?>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php
	}


	/**
	 * Render license info on subscription details page.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function render_frontend_license_info( int $subscription_id ) {
		if ( ! $this->check_plugin() ) {
			return;
		}

		$all_licenses = $this->get_licenses_from_subscription( $subscription_id );

		?>
		<style>
			.wpsfl-myaccount-license-key {
				font-family: Consolas, monospace;
				font-size: 13px;
			}
		</style>

		<h2><?php echo esc_html_e( 'Related Licenses', 'wp_subscription_pro' ); ?></h2>

		<?php foreach ( $all_licenses as $group_title => $licenses ) : ?>
			<table class="woocommerce-table woocommerce-table--order-details shop_table order_details">
				<thead>
					<tr>
						<th><?php echo esc_html( $group_title ); ?></th>
						<th></th>
					</tr>
				</thead>

				<tbody>
					<?php foreach ( $licenses as $license ) : ?>
						<tr>
							<td>
								<span class="wpsfl-myaccount-license-key">
									<?php echo wp_kses_post( $license->licence ); ?>
								</span>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		<?php endforeach; ?>

		<?php
	}
}
