<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Action;

/**
 * Class Order
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class Order {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'subscrpt_order_status_changed', array( $this, 'handle_order_status_changed' ), 10, 2 );
	}

	/**
	 * Handle Order status changed.
	 *
	 * @param \WC_Order $order   Order.
	 * @param mixed     $history Order relations.
	 */
	public function handle_order_status_changed( $order, $history ) {
		if ( 'early-renew' === $history->type ) {
			$early_meta = get_post_meta( $history->subscription_id, '_order_waiting_early_renew_subscrpt_meta', true );
			if ( $early_meta && 'completed' === $order->get_status() ) {
				update_post_meta( $history->subscription_id, '_subscrpt_order_id', $early_meta['order_id'] );
				update_post_meta( $history->subscription_id, '_subscrpt_order_item_id', $early_meta['order_item_id'] );
				wp_update_post(
					array(
						'ID'          => $history->subscription_id,
						'post_status' => 'active',
					)
				);

				// Increment renewal count for completed early renewal orders
				if ( class_exists( '\\SpringDevs\\SubscriptionPro\\Illuminate\\LimitChecker' ) ) {
					LimitChecker::increment_renewal_count( $history->subscription_id );
				}

				$order_item = $order->get_item( $early_meta['order_item_id'] );
				$product    = $order_item->get_product();
				if ( $product && $product->is_type( 'variation' ) ) {
					update_post_meta( $history->subscription_id, '_subscrpt_variation_id', $product->get_id() );
				}

				Action::write_comment( 'active', $history->subscription_id );
			}
			if ( in_array( $order->get_status(), array( 'cancelled', 'completed' ), true ) ) {
				delete_post_meta( $history->subscription_id, '_order_waiting_early_renew_subscrpt_meta' );
				if ( 'cancelled' === $order->get_status() ) {
					$comment_id = wp_insert_comment(
						array(
							'comment_author'  => 'Subscription for WooCommerce',
							'comment_content' => "Early renewal order {$order->get_id()} is cancelled.",
							'comment_post_ID' => $history->subscription_id,
							'comment_type'    => 'order_note',
						)
					);
					update_comment_meta( $comment_id, '_subscrpt_activity', 'Early Renewal Cancelled' );
					update_comment_meta( $comment_id, '_subscrpt_activity_type', 'early_renewal_cancelled' );
				}
			}
		}

		if ( in_array( $history->type, [ 'upgrade', 'downgrade' ], true ) ) {
			$order_status = $order->get_status();

			if ( in_array( $order_status, [ 'completed' ], true ) ) {
				$order_item     = $order->get_item( $history->order_item_id );
				$switch_context = $order_item->get_meta( '_wp_subs_switch_context' );

				// Update subscription after switch order is completed
				( new SubscriptionSwitch() )->subs_switch_update_subscription( $switch_context, $history, $order );

				$switch_type = $switch_context['switch_type'] ?? 'upgrade';

				unset( $switch_context['nonce'] );
				unset( $switch_context['redirect_back_url'] );

				/**
				 * Action fired after subscription switch completed and order is completed.
				 *
				 * @param string    $switch_type    Switch type (upgrade/downgrade).
				 * @param \WC_Order $order          Order object.
				 * @param \WC_Order_Item_Product $order_item Order Item object.
				 * @param array     $switch_context Switch context data. [ 'switch_type', 'subscription_id', 'order_id', 'product_id', 'old_variation_id', 'new_variation_id' ]
				 */
				do_action( 'subscrpt_switch_completed', $switch_type, $order, $order_item, $switch_context );
			}
		}
	}
}
