<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

/**
 * Handle payment failures in split payment subscriptions
 */
class PaymentFailureHandler {

	/**
	 * Initialize hooks
	 */
	public static function init() {
		// Handle both split payment and recurring payment failures
		add_action( 'subscrpt_split_payment_failed', array( __CLASS__, 'handle_payment_failure' ), 10, 1 );
		add_action( 'subscrpt_subscription_payment_failed', array( __CLASS__, 'handle_payment_failure' ), 10, 1 );
		add_action( 'woocommerce_order_status_failed', array( __CLASS__, 'handle_order_failed' ), 10, 1 );
		add_action( 'wp_loaded', array( __CLASS__, 'schedule_retry_checks' ) );
		add_action( 'subscrpt_check_payment_retries', array( __CLASS__, 'process_payment_retries' ) );
		add_action( 'subscrpt_check_grace_period_expiry', array( __CLASS__, 'process_grace_period_expiry' ) );
		add_action( 'subscrpt_send_delayed_failure_notification', array( __CLASS__, 'send_delayed_failure_notification' ) );
		add_action( 'subscrpt_check_grace_period_warnings', array( __CLASS__, 'send_grace_period_warnings' ) );
	}

	/**
	 * Handle payment failure for both recurring and split payments
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public static function handle_payment_failure( $subscription_id ) {
		$failure_count     = self::increment_failure_count( $subscription_id );
		$max_retries       = self::get_max_retries( $subscription_id );
		$grace_period_days = self::get_grace_period_days( $subscription_id );

		// Log the failure
		self::log_payment_failure( $subscription_id, $failure_count );

		// Check if we should suspend access
		if ( $failure_count >= $max_retries ) {
			if ( $grace_period_days > 0 ) {
				self::start_grace_period( $subscription_id, $grace_period_days );
			} else {
				self::suspend_access( $subscription_id );
			}
		} else {
			// Schedule retry
			self::schedule_payment_retry( $subscription_id, $failure_count );
		}

		// Send notification
		self::send_failure_notification( $subscription_id, $failure_count, $max_retries );
	}

	/**
	 * Handle WooCommerce order failure
	 *
	 * @param int $order_id Order ID.
	 */
	public static function handle_order_failed( $order_id ) {
		$subscription_id = self::get_subscription_from_order( $order_id );
		if ( $subscription_id ) {
			// Handle payment failure for both recurring and split payments
			self::handle_payment_failure( $subscription_id );
		}
	}

	/**
	 * Increment failure count for subscription
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int New failure count.
	 */
	private static function increment_failure_count( $subscription_id ) {
		$current_count = get_post_meta( $subscription_id, '_subscrpt_payment_failure_count', true ) ?: 0;
		$new_count     = intval( $current_count ) + 1;
		update_post_meta( $subscription_id, '_subscrpt_payment_failure_count', $new_count );
		update_post_meta( $subscription_id, '_subscrpt_last_payment_failure', current_time( 'timestamp' ) );
		return $new_count;
	}

	/**
	 * Get maximum retry attempts
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int Maximum retries.
	 */
	private static function get_max_retries( $subscription_id ) {
		// Use global setting as default
		$default_retries = get_option( 'subscrpt_default_max_payment_retries', 3 );
		return apply_filters( 'subscrpt_default_max_payment_retries', $default_retries );
	}

	/**
	 * Get grace period days
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int Grace period in days.
	 */
	private static function get_grace_period_days( $subscription_id ) {
		// Use global setting as default
		$default_grace_period = get_option( 'subscrpt_default_payment_grace_period', 7 );
		return apply_filters( 'subscrpt_default_grace_period_days', $default_grace_period );
	}

	/**
	 * Schedule payment retry
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $failure_count Current failure count.
	 */
	private static function schedule_payment_retry( $subscription_id, $failure_count ) {
		// Calculate retry delay (exponential backoff)
		$base_delay  = apply_filters( 'subscrpt_retry_base_delay_hours', 24 ); // 24 hours
		$retry_delay = $base_delay * pow( 2, $failure_count - 1 ); // 24h, 48h, 96h, etc.
		$retry_time  = current_time( 'timestamp' ) + ( $retry_delay * HOUR_IN_SECONDS );

		update_post_meta( $subscription_id, '_subscrpt_next_retry_time', $retry_time );
		update_post_meta( $subscription_id, '_subscrpt_retry_scheduled', 1 );

		// Log retry schedule
		$subscription_post = get_post( $subscription_id );
		if ( $subscription_post ) {
			$note = sprintf(
				/* translators: %1$d: failure count, %2$s: retry date */
				__( 'Payment failed (attempt %1$d). Next retry scheduled for %2$s', 'wp_subscription_pro' ),
				$failure_count,
				gmdate( 'Y-m-d H:i:s', $retry_time )
			);
			wp_insert_comment(
				array(
					'comment_post_ID'  => $subscription_id,
					'comment_content'  => $note,
					'comment_type'     => 'subscrpt_note',
					'comment_approved' => 1,
					'user_id'          => 0,
				)
			);
		}
	}

	/**
	 * Start grace period
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $grace_days Grace period in days.
	 */
	private static function start_grace_period( $subscription_id, $grace_days ) {
		$grace_end_time = current_time( 'timestamp' ) + ( $grace_days * DAY_IN_SECONDS );

		update_post_meta( $subscription_id, '_subscrpt_grace_period_end', $grace_end_time );
		update_post_meta( $subscription_id, '_subscrpt_access_suspended', 0 ); // Keep access during grace

		// Update subscription status to indicate grace period
		wp_update_post(
			array(
				'ID'          => $subscription_id,
				'post_status' => 'subscrpt-grace-period',
			)
		);

		// Log grace period start
		$note = sprintf(
			/* translators: %1$d: grace days, %2$s: grace end date */
			__( 'Payment retries exhausted. Grace period started (%1$d days until %2$s)', 'wp_subscription_pro' ),
			$grace_days,
			gmdate( 'Y-m-d H:i:s', $grace_end_time )
		);
		wp_insert_comment(
			array(
				'comment_post_ID'  => $subscription_id,
				'comment_content'  => $note,
				'comment_type'     => 'subscrpt_note',
				'comment_approved' => 1,
				'user_id'          => 0,
			)
		);
	}

	/**
	 * Suspend access for subscription
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	private static function suspend_access( $subscription_id ) {
		update_post_meta( $subscription_id, '_subscrpt_access_suspended', 1 );
		update_post_meta( $subscription_id, '_subscrpt_access_suspended_date', current_time( 'timestamp' ) );

		// Update subscription status
		wp_update_post(
			array(
				'ID'          => $subscription_id,
				'post_status' => 'subscrpt-suspended',
			)
		);

		// Log access suspension
		$note = __( 'Access suspended due to payment failures', 'wp_subscription_pro' );
		wp_insert_comment(
			array(
				'comment_post_ID'  => $subscription_id,
				'comment_content'  => $note,
				'comment_type'     => 'subscrpt_note',
				'comment_approved' => 1,
				'user_id'          => 0,
			)
		);

		do_action( 'subscrpt_access_suspended', $subscription_id );
	}

	/**
	 * Send failure notification
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $failure_count Current failure count.
	 * @param int $max_retries Maximum retries allowed.
	 */
	private static function send_failure_notification( $subscription_id, $failure_count, $max_retries ) {
		// Check if payment failure emails are enabled globally
		if ( ! get_option( 'subscrpt_enable_payment_failure_emails', '1' ) ) {
			return;
		}

		$customer_id = get_post_meta( $subscription_id, '_subscrpt_user_id', true );
		if ( ! $customer_id ) {
			return;
		}

		$customer = get_user_by( 'id', $customer_id );
		if ( ! $customer ) {
			return;
		}

		// Check if we should delay the email based on global settings
		$email_delay_hours = get_option( 'subscrpt_payment_failure_email_delay', 24 );
		$last_failure_time = get_post_meta( $subscription_id, '_subscrpt_last_payment_failure', true );

		if ( $last_failure_time && $email_delay_hours > 0 ) {
			$time_since_failure = current_time( 'timestamp' ) - $last_failure_time;
			$delay_seconds      = $email_delay_hours * HOUR_IN_SECONDS;

			if ( $time_since_failure < $delay_seconds ) {
				// Schedule delayed email
				wp_schedule_single_event(
					$last_failure_time + $delay_seconds,
					'subscrpt_send_delayed_failure_notification',
					array( $subscription_id, $failure_count, $max_retries )
				);
				return;
			}
		}

		// Use the new email system instead of wp_mail
		do_action( 'subscrpt_payment_failure_email_notification', $subscription_id, $failure_count, $max_retries, false );

		do_action( 'subscrpt_payment_failure_notification_sent', $subscription_id, $customer_id, $failure_count );
	}

	/**
	 * Schedule retry and grace period checks
	 */
	public static function schedule_retry_checks() {
		if ( ! wp_next_scheduled( 'subscrpt_check_payment_retries' ) ) {
			wp_schedule_event( time(), 'hourly', 'subscrpt_check_payment_retries' );
		}

		if ( ! wp_next_scheduled( 'subscrpt_check_grace_period_expiry' ) ) {
			wp_schedule_event( time(), 'twicedaily', 'subscrpt_check_grace_period_expiry' );
		}

		if ( ! wp_next_scheduled( 'subscrpt_check_grace_period_warnings' ) ) {
			wp_schedule_event( time(), 'daily', 'subscrpt_check_grace_period_warnings' );
		}
	}

	/**
	 * Process payment retries
	 */
	public static function process_payment_retries() {
		global $wpdb;

		$current_time = current_time( 'timestamp' );

		// Find subscriptions ready for retry
		$subscriptions = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT p.ID 
			FROM {$wpdb->posts} p
			INNER JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_subscrpt_retry_scheduled'
			INNER JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_subscrpt_next_retry_time'
			WHERE p.post_type = 'subscrpt_order'
			AND pm1.meta_value = '1'
			AND pm2.meta_value <= %d",
				$current_time
			)
		);

		foreach ( $subscriptions as $subscription ) {
			self::attempt_payment_retry( $subscription->ID );
		}
	}

	/**
	 * Attempt payment retry
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	private static function attempt_payment_retry( $subscription_id ) {
		// Clear retry flag
		delete_post_meta( $subscription_id, '_subscrpt_retry_scheduled' );
		delete_post_meta( $subscription_id, '_subscrpt_next_retry_time' );

		// Create renewal order for retry
		$renewal_order_id = self::create_retry_renewal_order( $subscription_id );
		if ( $renewal_order_id ) {
			// Log retry attempt
			$note = sprintf(
				/* translators: %d: order ID */
				__( 'Payment retry attempted. Renewal order #%d created.', 'wp_subscription_pro' ),
				$renewal_order_id
			);
			wp_insert_comment(
				array(
					'comment_post_ID'  => $subscription_id,
					'comment_content'  => $note,
					'comment_type'     => 'subscrpt_note',
					'comment_approved' => 1,
					'user_id'          => 0,
				)
			);

			do_action( 'subscrpt_payment_retry_attempted', $subscription_id, $renewal_order_id );
		}
	}

	/**
	 * Process grace period expiry
	 */
	public static function process_grace_period_expiry() {
		global $wpdb;

		$current_time = current_time( 'timestamp' );

		// Find subscriptions with expired grace periods
		$subscriptions = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT p.ID 
			FROM {$wpdb->posts} p
			INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_subscrpt_grace_period_end'
			WHERE p.post_type = 'subscrpt_order'
			AND p.post_status = 'subscrpt-grace-period'
			AND pm.meta_value <= %d",
				$current_time
			)
		);

		foreach ( $subscriptions as $subscription ) {
			self::expire_grace_period( $subscription->ID );
		}
	}

	/**
	 * Expire grace period and suspend access
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	private static function expire_grace_period( $subscription_id ) {
		delete_post_meta( $subscription_id, '_subscrpt_grace_period_end' );
		self::suspend_access( $subscription_id );

		// Send final notification
		$customer_id = get_post_meta( $subscription_id, '_subscrpt_user_id', true );
		if ( $customer_id ) {
			$customer = get_user_by( 'id', $customer_id );
			if ( $customer ) {
				$subject = __( 'Subscription Access Suspended', 'wp_subscription_pro' );
				$message = sprintf(
					/* translators: %s: customer name */
					__( 'Hello %s,\n\nYour grace period has expired and your subscription access has been suspended due to payment failures.\n\nPlease contact support to restore your access.', 'wp_subscription_pro' ),
					$customer->display_name
				);
				wp_mail( $customer->user_email, $subject, $message );
			}
		}

		do_action( 'subscrpt_grace_period_expired', $subscription_id );
	}

	/**
	 * Helper functions
	 */

	/**
	 * Get subscription ID from order
	 *
	 * @param int $order_id Order ID.
	 * @return int|null Subscription ID.
	 */
	public static function get_subscription_from_order( $order_id ) {
		global $wpdb;

		$result = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT subscription_id FROM {$wpdb->prefix}subscrpt_order_relation WHERE order_id = %d",
				$order_id
			)
		);

		return $result ? intval( $result ) : null;
	}

	/**
	 * Check if subscription supports payment retry (both recurring and split payment)
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return bool True if supports retry.
	 */
	public static function supports_payment_retry( $subscription_id ) {
		// All subscription types support payment retry
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		return (bool) $product_id;
	}

	/**
	 * Create retry renewal order
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int|null Renewal order ID.
	 */
	private static function create_retry_renewal_order( $subscription_id ) {
		// This would integrate with existing renewal order creation logic
		// For now, return null - this needs to be implemented based on existing renewal system
		return apply_filters( 'subscrpt_create_retry_renewal_order', null, $subscription_id );
	}

	/**
	 * Log payment failure
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $failure_count Failure count.
	 */
	private static function log_payment_failure( $subscription_id, $failure_count ) {
		$note = sprintf(
			/* translators: %d: failure count */
			__( 'Split payment failed (attempt %d)', 'wp_subscription_pro' ),
			$failure_count
		);

		wp_insert_comment(
			array(
				'comment_post_ID'  => $subscription_id,
				'comment_content'  => $note,
				'comment_type'     => 'subscrpt_note',
				'comment_approved' => 1,
				'user_id'          => 0,
			)
		);

		do_action( 'subscrpt_payment_failure_logged', $subscription_id, $failure_count );
	}

	/**
	 * Reset failure count (called when payment succeeds)
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public static function reset_failure_count( $subscription_id ) {
		delete_post_meta( $subscription_id, '_subscrpt_payment_failure_count' );
		delete_post_meta( $subscription_id, '_subscrpt_last_payment_failure' );
		delete_post_meta( $subscription_id, '_subscrpt_retry_scheduled' );
		delete_post_meta( $subscription_id, '_subscrpt_next_retry_time' );
		delete_post_meta( $subscription_id, '_subscrpt_grace_period_end' );
		delete_post_meta( $subscription_id, '_subscrpt_access_suspended' );
		delete_post_meta( $subscription_id, '_subscrpt_access_suspended_date' );
		delete_post_meta( $subscription_id, '_subscrpt_grace_period_warning_sent' );

		// Restore subscription to active if it was suspended
		$current_status = get_post_status( $subscription_id );
		if ( in_array( $current_status, array( 'subscrpt-suspended', 'subscrpt-grace-period' ) ) ) {
			wp_update_post(
				array(
					'ID'          => $subscription_id,
					'post_status' => 'subscrpt-active',
				)
			);
		}

		do_action( 'subscrpt_payment_failure_reset', $subscription_id );
	}

	/**
	 * Check if subscription access is suspended
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return bool True if suspended.
	 */
	public static function is_access_suspended( $subscription_id ) {
		return (bool) get_post_meta( $subscription_id, '_subscrpt_access_suspended', true );
	}

	/**
	 * Send delayed failure notification
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $failure_count Failure count.
	 * @param int $max_retries Maximum retries.
	 */
	public static function send_delayed_failure_notification( $subscription_id, $failure_count, $max_retries ) {
		// Check if emails are still enabled
		if ( ! get_option( 'subscrpt_enable_payment_failure_emails', '1' ) ) {
			return;
		}

		$customer_id = get_post_meta( $subscription_id, '_subscrpt_user_id', true );
		if ( ! $customer_id ) {
			return;
		}

		$customer = get_user_by( 'id', $customer_id );
		if ( ! $customer_id ) {
			return;
		}

		// Use the new email system instead of wp_mail
		do_action( 'subscrpt_payment_failure_email_notification', $subscription_id, $failure_count, $max_retries, true );

		do_action( 'subscrpt_delayed_payment_failure_notification_sent', $subscription_id, $customer_id, $failure_count );
	}

	/**
	 * Send grace period warnings
	 */
	public static function send_grace_period_warnings() {
		// Check if grace period notifications are enabled
		if ( ! get_option( 'subscrpt_enable_grace_period_notifications', '1' ) ) {
			return;
		}

		$warning_days = get_option( 'subscrpt_grace_period_warning_days', 2 );
		if ( $warning_days <= 0 ) {
			return;
		}

		global $wpdb;

		$current_time = current_time( 'timestamp' );
		$warning_time = $current_time + ( $warning_days * DAY_IN_SECONDS );

		// Find subscriptions that need grace period warnings
		$subscriptions = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT p.ID 
			FROM {$wpdb->posts} p
			INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_subscrpt_grace_period_end'
			WHERE p.post_type = 'subscrpt_order'
			AND p.post_status = 'subscrpt-grace-period'
			AND pm.meta_value <= %d
			AND pm.meta_value > %d",
				$warning_time,
				$current_time
			)
		);

		foreach ( $subscriptions as $subscription ) {
			$subscription_id = $subscription->ID;
			$grace_end_time  = get_post_meta( $subscription_id, '_subscrpt_grace_period_end', true );
			$days_remaining  = ceil( ( $grace_end_time - $current_time ) / DAY_IN_SECONDS );

			$customer_id = get_post_meta( $subscription_id, '_subscrpt_user_id', true );
			if ( ! $customer_id ) {
				continue;
			}

			$customer = get_user_by( 'id', $customer_id );
			if ( ! $customer ) {
				continue;
			}

			// Check if we've already sent a warning for this subscription
			$warning_sent = get_post_meta( $subscription_id, '_subscrpt_grace_period_warning_sent', true );
			if ( $warning_sent ) {
				continue;
			}

			$subject = __( 'Grace Period Ending Soon', 'wp_subscription_pro' );
			$message = sprintf(
				/* translators: %1$s: customer name, %2$d: days remaining */
				__( 'Hello %1$s,\n\nYour subscription grace period will end in %2$d days. After this period, your access will be suspended.\n\nPlease update your payment method immediately to avoid service interruption.\n\nIf you need assistance, please contact our support team.', 'wp_subscription_pro' ),
				$customer->display_name,
				$days_remaining
			);

			wp_mail( $customer->user_email, $subject, $message );

			// Mark warning as sent
			update_post_meta( $subscription_id, '_subscrpt_grace_period_warning_sent', 1 );

			do_action( 'subscrpt_grace_period_warning_sent', $subscription_id, $customer_id, $days_remaining );
		}
	}
}
