<?php
/**
 * Role Management Pro File
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\RoleManagement as FreeRoleManagement;

/**
 * RoleManagement [ helper class ]
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class RoleManagement extends FreeRoleManagement {
	/**
	 * Initialize the class
	 */
	public function __construct() {
		// Run parent constructor first
		parent::__construct();

		// Simple product role forms
		add_action( 'subscrpt_simple_pro_after_subs_fields', [ $this, 'add_subscription_role_forms' ] );
		add_action( 'save_post_product', [ $this, 'save_subscription_role_form_data' ] );

		// Variable product role forms
		add_action( 'woocommerce_product_after_variable_attributes', [ $this, 'add_variable_subscription_role_forms' ], 10, 3 );
		add_action( 'woocommerce_save_product_variation', [ $this, 'save_variable_subscription_role_form_data' ], 10, 2 );

		// Product visibility filter
		add_filter( 'woocommerce_product_query_meta_query', [ $this, 'control_product_visibility_based_on_roles' ], 10, 2 );
		add_filter( 'woocommerce_get_children', [ $this, 'control_variations_visibility_based_on_roles' ], 10, 2 );
	}

	/**
	 * Add subscription role forms
	 *
	 * @param int $product_id Product ID.
	 */
	public function add_subscription_role_forms( $product_id ) {
		global $wp_roles;
		$roles = wp_list_pluck( $wp_roles->roles, 'name' );

		$selected_roles = get_post_meta( $product_id, '_subscrpt_visible_to_roles', true );
		$selected_roles = empty( $selected_roles ) ? [] : $selected_roles;

		?>
		<br/>
		<strong style="margin: 10px;">
			<?php esc_html_e( 'Role Based Settings', 'wp_subscription_pro' ); ?>
		</strong>
		
		<?php
		wp_subs_multiselect_field(
			[
				'id'          => 'subscrpt_visible_to_roles',
				'label'       => __( 'Visible to', 'wp_subscription_pro' ),
				'options'     => $roles,
				'selected'    => $selected_roles,
				'description' => __( 'Select one or more user roles to limit access to this product. Keep empty for no restrictions.', 'wp_subscription_pro' ),
				'desc_tip'    => true,
				'style'       => 'width: 50%;',
			]
		);
		?>
		<?php
	}

	/**
	 * Add variable subscription role forms
	 *
	 * @param int    $loop Loop.
	 * @param array  $variation_data Variation Data.
	 * @param object $variation Variation.
	 */
	public function add_variable_subscription_role_forms( $loop, $variation_data, $variation ) {
		global $wp_roles;
		$roles        = wp_list_pluck( $wp_roles->roles, 'name' );
		$variation_id = $variation->ID;

		$selected_roles = get_post_meta( $variation_id, '_subscrpt_visible_to_roles', true );
		$selected_roles = empty( $selected_roles ) ? [] : $selected_roles;

		?>
		<div class="subscrpt_role_fields show_if_subscription_<?php echo esc_attr( $loop ); ?>">
			<br/>
			<strong><?php esc_html_e( 'Role Based Settings', 'wp_subscription_pro' ); ?></strong>
			<p class="description">Settings dependent on user roles.</p>

			<?php
			wp_subs_multiselect_field(
				[
					'id'            => 'subscrpt_visible_to_roles[' . $variation_id . ']',
					'label'         => __( 'Visible to', 'wp_subscription_pro' ),
					'options'       => $roles,
					'selected'      => $selected_roles,
					'description'   => __( 'Select one or more user roles to limit access to this variation. Keep empty for no restrictions.', 'wp_subscription_pro' ),
					'desc_tip'      => true,
					'style'         => 'width: 100%;',
					'wrapper_class' => 'form-row',
				]
			);
			?>
		</div>
		<?php
	}

	/**
	 * Save subscription role form data
	 *
	 * @param int $product_id Product ID.
	 */
	public function save_subscription_role_form_data( $product_id ) {
		// ? escaped intentionally.
		// phpcs:disable WordPress.Security.NonceVerification
		$selected_roles = isset( $_POST['subscrpt_visible_to_roles'] )
							? array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['subscrpt_visible_to_roles'] ) )
							: [];
		// phpcs:enable WordPress.Security.NonceVerification

		// Save the selected roles as post meta.
		update_post_meta( $product_id, '_subscrpt_visible_to_roles', $selected_roles );
	}

	/**
	 * Save variable subscription role form data
	 *
	 * @param int $variation_id Variation ID.
	 * @param int $i Index.
	 */
	public function save_variable_subscription_role_form_data( $variation_id, $i ) {
		// ? escaped intentionally.
		// phpcs:disable WordPress.Security.NonceVerification
		$selected_roles = isset( $_POST['subscrpt_visible_to_roles'][ $variation_id ] )
							? array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['subscrpt_visible_to_roles'][ $variation_id ] ) )
							: [];
		// phpcs:enable WordPress.Security.NonceVerification

		// Save the selected roles as post meta.
		update_post_meta( $variation_id, '_subscrpt_visible_to_roles', $selected_roles );
	}

	/**
	 * Control product visibility based on roles
	 *
	 * @param array  $meta_query Meta query.
	 * @param object $query Query object.
	 */
	public function control_product_visibility_based_on_roles( $meta_query, $query ) {
		$user_roles = [];
		if ( is_user_logged_in() ) {
			$user       = wp_get_current_user();
			$user_roles = (array) $user->roles;
		}

		$role_queries = [];
		foreach ( $user_roles as $role ) {
			$role_queries[] = [
				'key'     => '_subscrpt_visible_to_roles',
				'value'   => $role,
				'compare' => 'LIKE',
			];
		}

		$meta_query[] = [
			'relation' => 'OR',
			[
				'key'     => '_subscrpt_visible_to_roles',
				'compare' => 'NOT EXISTS', // No restriction
			],
			[
				'key'     => '_subscrpt_visible_to_roles',
				'value'   => [
					'',
					'[]',
					'a:0:{}', // Serialized empty array
				],
				'compare' => 'IN',
			],
			[
				'relation' => 'OR',
				...$role_queries,
			],
		];

		return $meta_query;
	}

	/**
	 * Control variations visibility based on roles
	 *
	 * @param array       $children Array of children variation IDs.
	 * @param \WC_Product $product Product object.
	 */
	public function control_variations_visibility_based_on_roles( $children, $product ) {
		foreach ( $children as $key => $variation_id ) {
			$allowed_roles = get_post_meta( $variation_id, '_subscrpt_visible_to_roles', true );
			$allowed_roles = empty( $allowed_roles ) ? [] : $allowed_roles;

			// Do not filter if no roles are set
			if ( empty( $allowed_roles ) ) {
				continue;
			}

			$chk = true;

			// role set but user not logged in
			if ( ! empty( $allowed_roles ) && ! is_user_logged_in() ) {
				$chk = false;
			}

			// role set and user logged in
			$user_roles = wp_get_current_user()->roles;
			if ( empty( array_intersect( $allowed_roles, $user_roles ) ) ) {
				$chk = false;
			}

			if ( ! $chk ) {
				unset( $children[ $key ] );
			}
		}

		return $children;
	}
}
