<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

/**
 * Class SplitPaymentHandler
 *
 * Handles business logic for Payment Type and Access Ends Timing features
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class SplitPaymentHandler {

	/**
	 * Calculate access end date for a split payment subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int|null Access end timestamp or null if no end date.
	 */
	public static function calculate_access_end_date( $subscription_id ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		if ( ! $product_id ) {
			return null;
		}

		// Check if subscription has variation
		$variation_id   = get_post_meta( $subscription_id, '_subscrpt_variation_id', true );
		$product_to_use = $variation_id ? $variation_id : $product_id;

		$product = wc_get_product( $product_to_use );
		if ( ! $product ) {
			return null;
		}

		$payment_type = $product->get_meta( '_subscrpt_payment_type' ) ?: 'recurring';

		// Only calculate for split payments
		if ( 'split_payment' !== $payment_type ) {
			return null;
		}

		$access_ends_timing = $product->get_meta( '_subscrpt_access_ends_timing' ) ?: 'after_full_duration';
		$start_date         = get_post_meta( $subscription_id, '_subscrpt_start_date', true );

		if ( ! $start_date ) {
			return null;
		}

		switch ( $access_ends_timing ) {
			case 'lifetime':
				// Access never ends - lifetime access
				return null;

			case 'after_full_duration':
				// Access ends based on payment interval × number of payments
				return self::calculate_full_duration_end_date( $subscription_id, $product );

			case 'custom_duration':
				// Access ends based on custom duration after first payment
				return self::calculate_custom_duration_end_date( $subscription_id, $product );

			default:
				return null;
		}
	}

	/**
	 * Get the date when final payment is expected.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int|null Final payment timestamp.
	 */
	public static function get_final_payment_date( $subscription_id ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		if ( ! $product_id ) {
			return null;
		}

		// Use helper function that handles variations properly
		$max_payments = subscrpt_get_max_payments( $subscription_id );

		if ( ! $max_payments || intval( $max_payments ) <= 0 ) {
			return null;
		}

		$start_date    = get_post_meta( $subscription_id, '_subscrpt_start_date', true );
		$timing_per    = get_post_meta( $subscription_id, '_subscrpt_timing_per', true ) ?: 1;
		$timing_option = get_post_meta( $subscription_id, '_subscrpt_timing_option', true ) ?: 'months';

		if ( ! $start_date ) {
			return null;
		}

		// Calculate final payment date: start + (payments - 1) × interval
		$interval_string    = $timing_per . ' ' . $timing_option;
		$final_payment_date = $start_date;

		for ( $i = 1; $i < $max_payments; $i++ ) {
			$final_payment_date = sdevs_wp_strtotime( $interval_string, $final_payment_date );
		}

		return $final_payment_date;
	}

	/**
	 * Get the first payment date for a subscription.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int|null First payment timestamp.
	 */
	public static function get_first_payment_date( $subscription_id ) {
		// The first payment date is simply the start date
		$start_date = get_post_meta( $subscription_id, '_subscrpt_start_date', true );

		if ( ! $start_date ) {
			return null;
		}

		return $start_date;
	}

	/**
	 * Calculate access end date based on full duration (default behavior).
	 *
	 * @param int         $subscription_id Subscription ID.
	 * @param \WC_Product $product Product object.
	 * @return int|null Access end timestamp.
	 */
	public static function calculate_full_duration_end_date( $subscription_id, $product ) {
		$max_payments = $product->get_meta( '_subscrpt_max_no_payment' );
		if ( ! $max_payments || $max_payments <= 0 ) {
			return null;
		}

		$start_date    = get_post_meta( $subscription_id, '_subscrpt_start_date', true );
		$timing_per    = get_post_meta( $subscription_id, '_subscrpt_timing_per', true ) ?: 1;
		$timing_option = get_post_meta( $subscription_id, '_subscrpt_timing_option', true ) ?: 'months';

		if ( ! $start_date ) {
			return null;
		}

		// Calculate full duration: start + payments × interval
		$interval_string = $timing_per . ' ' . $timing_option;
		$end_date        = $start_date;

		for ( $i = 0; $i < $max_payments; $i++ ) {
			$end_date = sdevs_wp_strtotime( $interval_string, $end_date );
		}

		return $end_date;
	}

	/**
	 * Calculate access end date based on custom duration after first payment.
	 *
	 * @param int         $subscription_id Subscription ID.
	 * @param \WC_Product $product Product object.
	 * @return int|null Access end timestamp.
	 */
	public static function calculate_custom_duration_end_date( $subscription_id, $product ) {
		$first_payment_date = self::get_first_payment_date( $subscription_id );
		if ( ! $first_payment_date ) {
			return null;
		}

		$custom_duration_time = $product->get_meta( '_subscrpt_custom_access_duration_time' ) ?: 1;
		$custom_duration_type = $product->get_meta( '_subscrpt_custom_access_duration_type' ) ?: 'months';

		// Add custom duration to first payment date
		$custom_interval = $custom_duration_time . ' ' . $custom_duration_type;
		return sdevs_wp_strtotime( $custom_interval, $first_payment_date );
	}

	/**
	 * Check if subscription access should be expired based on access timing rules.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return bool True if access should be expired.
	 */
	public static function should_expire_access( $subscription_id ) {
		$access_end_date = self::calculate_access_end_date( $subscription_id );

		if ( ! $access_end_date ) {
			return false;
		}

		return time() >= $access_end_date;
	}

	/**
	 * Get human-readable access end date string.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return string|null Human-readable access end date.
	 */
	public static function get_access_end_date_string( $subscription_id ) {
		$access_end_date = self::calculate_access_end_date( $subscription_id );

		if ( ! $access_end_date ) {
			return null;
		}

		return date_i18n( wc_date_format(), $access_end_date );
	}

	/**
	 * Apply access timing rules when split payment is completed.
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $payments_made Number of payments made.
	 * @param int $max_payments Maximum number of payments.
	 */
	public static function handle_split_payment_completion( $subscription_id, $payments_made, $max_payments ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		if ( ! $product_id ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product || 'split_payment' !== ( $product->get_meta( '_subscrpt_payment_type' ) ?: 'recurring' ) ) {
			return;
		}

		$access_ends_timing = $product->get_meta( '_subscrpt_access_ends_timing' ) ?: 'after_full_duration';

		// Add detailed activity logging for split payment completion
		self::add_split_payment_completion_activity( $subscription_id, $payments_made, $max_payments, $access_ends_timing );

		switch ( $access_ends_timing ) {
			case 'lifetime':
				// Keep subscription active indefinitely - lifetime access
				wp_update_post(
					array(
						'ID'          => $subscription_id,
						'post_status' => 'active',
					)
				);

				// Mark as lifetime access
				update_post_meta( $subscription_id, '_subscrpt_lifetime_access', 'yes' );
				delete_post_meta( $subscription_id, '_subscrpt_access_end_date' );
				break;

			case 'after_full_duration':
			case 'custom_duration':
				// Keep subscription active but schedule expiration
				wp_update_post(
					array(
						'ID'          => $subscription_id,
						'post_status' => 'active',
					)
				);

				// Store access end date for future reference
				$access_end_date = self::calculate_access_end_date( $subscription_id );
				if ( $access_end_date ) {
					update_post_meta( $subscription_id, '_subscrpt_access_end_date', $access_end_date );
				}
				break;
		}
	}

	/**
	 * Add detailed activity logging for split payment completion.
	 *
	 * @param int    $subscription_id    Subscription ID.
	 * @param int    $payments_made      Number of payments made.
	 * @param int    $max_payments       Maximum number of payments.
	 * @param string $access_ends_timing Access ends timing setting.
	 */
	private static function add_split_payment_completion_activity( $subscription_id, $payments_made, $max_payments, $access_ends_timing ) {
		$access_end_date   = self::calculate_access_end_date( $subscription_id );
		$access_end_string = $access_end_date ? self::get_access_end_date_string( $subscription_id ) : '';

		// Create completion note based on access timing
		$completion_note = '';
		$activity_type   = '';

		switch ( $access_ends_timing ) {
			case 'lifetime':
				$completion_note = sprintf(
					/* translators: %1$d: payments made, %2$d: total payments */
					__( 'Split payment plan completed: %1$d of %2$d payments received. Lifetime access granted!', 'wp_subscription_pro' ),
					$payments_made,
					$max_payments
				);
				$activity_type = __( 'Split Payment - Plan Complete (Lifetime Access)', 'wp_subscription_pro' );
				break;

			case 'after_full_duration':
				$completion_note = sprintf(
					/* translators: %1$d: payments made, %2$d: total payments, %3$s: access end date */
					__( 'Split payment plan completed: %1$d of %2$d payments received. Access continues until full duration: %3$s.', 'wp_subscription_pro' ),
					$payments_made,
					$max_payments,
					$access_end_string
				);
				$activity_type = __( 'Split Payment - Plan Complete (Full Duration)', 'wp_subscription_pro' );
				break;

			case 'custom_duration':
				$completion_note = sprintf(
					/* translators: %1$d: payments made, %2$d: total payments, %3$s: access end date */
					__( 'Split payment plan completed: %1$d of %2$d payments received. Access continues with custom duration until: %3$s.', 'wp_subscription_pro' ),
					$payments_made,
					$max_payments,
					$access_end_string
				);
				$activity_type = __( 'Split Payment - Plan Complete (Custom Duration)', 'wp_subscription_pro' );
				break;
		}

		// Add the completion activity note
		if ( $completion_note ) {
			$comment_id = wp_insert_comment(
				array(
					'comment_author'  => 'Subscription for WooCommerce',
					'comment_content' => $completion_note,
					'comment_post_ID' => $subscription_id,
					'comment_type'    => 'order_note',
				)
			);
			update_comment_meta( $comment_id, '_subscrpt_activity', $activity_type );
			update_comment_meta( $comment_id, '_subscrpt_activity_type', 'split_payment' );
		}

		// Add access timing decision note
		self::add_access_timing_decision_note( $subscription_id, $access_ends_timing, $access_end_string );
	}

	/**
	 * Add access timing decision note explaining why access continues or expires.
	 *
	 * @param int    $subscription_id    Subscription ID.
	 * @param string $access_ends_timing Access ends timing setting.
	 * @param string $access_end_string  Human-readable access end date.
	 */
	private static function add_access_timing_decision_note( $subscription_id, $access_ends_timing, $access_end_string ) {
		$decision_note = '';
		$activity_type = '';

		switch ( $access_ends_timing ) {
			case 'lifetime':
				$decision_note = __( 'Lifetime access granted after completing split payment plan as per product settings (lifetime).', 'wp_subscription_pro' );
				$activity_type = __( 'Access Control - Lifetime Access', 'wp_subscription_pro' );
				break;

			case 'after_full_duration':
				$decision_note = sprintf(
					/* translators: %s: access end date */
					__( 'Access continues until full duration as per product settings (after_full_duration). Access ends: %s.', 'wp_subscription_pro' ),
					$access_end_string
				);
				$activity_type = __( 'Access Control - Full Duration', 'wp_subscription_pro' );
				break;

			case 'custom_duration':
				$decision_note = sprintf(
					/* translators: %s: access end date */
					__( 'Access continues with custom duration as per product settings (custom_duration). Access ends: %s.', 'wp_subscription_pro' ),
					$access_end_string
				);
				$activity_type = __( 'Access Control - Custom Duration', 'wp_subscription_pro' );
				break;
		}

		// Add the decision note
		if ( $decision_note ) {
			$comment_id = wp_insert_comment(
				array(
					'comment_author'  => 'Subscription for WooCommerce',
					'comment_content' => $decision_note,
					'comment_post_ID' => $subscription_id,
					'comment_type'    => 'order_note',
				)
			);
			update_comment_meta( $comment_id, '_subscrpt_activity', $activity_type );
		}
	}
}
