<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Action;

/**
 * Subscription Status Handler
 *
 * Handles subscription status changes and related operations.
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class SubscriptionStatusHandler {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		// Core subscription actions
		add_action( 'subscrpt_cancel_subscription', array( $this, 'handle_cancel' ) );
		add_action( 'subscrpt_pause_subscription', array( $this, 'handle_pause' ) );
		add_action( 'subscrpt_resume_subscription', array( $this, 'handle_resume' ) );
		add_action( 'subscrpt_change_status', array( $this, 'handle_status_change' ) );

		// Additional subscription actions
		add_action( 'subscrpt_reactivate_subscription', array( $this, 'handle_reactivation' ) );
	}

	/**
	 * Handle subscription cancellation
	 *
	 * @param int $subscription_id Subscription ID
	 * @return void
	 */
	public function handle_cancel( $subscription_id ) {
		try {
			if ( ! $this->is_valid_subscription( $subscription_id ) ) {
				$this->log_error( 'Invalid subscription ID for cancellation: ' . $subscription_id );
				return;
			}
			Action::status( 'cancelled', $subscription_id );
			$this->log_success( 'Subscription cancelled successfully: ' . $subscription_id );
		} catch ( \Exception $e ) {
			$this->log_error( 'Error cancelling subscription: ' . $e->getMessage() );
		}
	}

	/**
	 * Handle subscription pause
	 *
	 * @param int $subscription_id Subscription ID
	 * @return void
	 */
	public function handle_pause( $subscription_id ) {
		try {
			if ( ! $this->is_valid_subscription( $subscription_id ) ) {
				$this->log_error( 'Invalid subscription ID for pause: ' . $subscription_id );
				return;
			}
			Action::status( 'paused', $subscription_id );
			$this->log_success( 'Subscription paused successfully: ' . $subscription_id );
		} catch ( \Exception $e ) {
			$this->log_error( 'Error pausing subscription: ' . $e->getMessage() );
		}
	}

	/**
	 * Handle subscription resume
	 *
	 * @param int $subscription_id Subscription ID
	 * @return void
	 */
	public function handle_resume( $subscription_id ) {
		try {
			if ( ! $this->is_valid_subscription( $subscription_id ) ) {
				$this->log_error( 'Invalid subscription ID for resume: ' . $subscription_id );
				return;
			}
			Action::status( 'active', $subscription_id );
			$this->log_success( 'Subscription resumed successfully: ' . $subscription_id );
		} catch ( \Exception $e ) {
			$this->log_error( 'Error resuming subscription: ' . $e->getMessage() );
		}
	}

	/**
	 * Handle subscription status change
	 *
	 * @param int    $subscription_id Subscription ID
	 * @param string $new_status New status to set
	 * @return void
	 */
	public function handle_status_change( $subscription_id, $new_status ) {
		try {
			if ( ! $this->is_valid_subscription( $subscription_id ) ) {
				$this->log_error( 'Invalid subscription ID for status change: ' . $subscription_id );
				return;
			}
			if ( ! $this->is_valid_status( $new_status ) ) {
				$this->log_error( 'Invalid status for subscription ' . $subscription_id . ': ' . $new_status );
				return;
			}
			Action::status( $new_status, $subscription_id );
			$this->log_success( 'Subscription status changed successfully: ' . $subscription_id . ' to ' . $new_status );
		} catch ( \Exception $e ) {
			$this->log_error( 'Error changing subscription status: ' . $e->getMessage() );
		}
	}

	/**
	 * Handle subscription reactivation
	 *
	 * @param int $subscription_id Subscription ID
	 * @return void
	 */
	public function handle_reactivation( $subscription_id ) {
		try {
			if ( ! $this->is_valid_subscription( $subscription_id ) ) {
				$this->log_error( 'Invalid subscription ID for reactivation: ' . $subscription_id );
				return;
			}
			Action::status( 'active', $subscription_id );
			$this->log_success( 'Subscription reactivated successfully: ' . $subscription_id );
		} catch ( \Exception $e ) {
			$this->log_error( 'Error reactivating subscription: ' . $e->getMessage() );
		}
	}

	/**
	 * Validate if the given ID is a valid subscription
	 *
	 * @param int $subscription_id Subscription ID to validate
	 * @return bool
	 */
	private function is_valid_subscription( $subscription_id ) {
		return get_post_type( $subscription_id ) === 'subscrpt_order';
	}

	/**
	 * Validate if the given status is valid
	 *
	 * @param string $status Status to validate
	 * @return bool
	 */
	private function is_valid_status( $status ) {
		$valid_statuses = array( 'active', 'cancelled', 'expired', 'paused', 'pending' );
		return in_array( $status, $valid_statuses );
	}

	/**
	 * Log error message
	 *
	 * @param string $message Error message to log
	 * @return void
	 */
	private function log_error( $message ) {
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->error( $message, array( 'source' => 'subscription-status-handler' ) );
		}
	}

	/**
	 * Log success message
	 *
	 * @param string $message Success message to log
	 * @return void
	 */
	private function log_success( $message ) {
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->info( $message, array( 'source' => 'subscription-status-handler' ) );
		}
	}
}
