<?php
/**
 * Enhanced Stats page template - European-style precious UI
 *
 * @var array $metrics Comprehensive subscription metrics
 * @var array $popular_subscriptions Popular subscriptions data
 * @var array $trial_conversion_data Trial conversion metrics
 * @var array $revenue_metrics Revenue metrics
 * @var array $monthly_trends Monthly trends data
 */

// Helper function for price formatting
function format_price( $amount ) {
	return function_exists( 'wc_price' ) ? wc_price( $amount ) : '$' . number_format( $amount, 2 );
}
?>

<div class="wp-subscription-dashboard">
	
	<!-- Header -->
	<div class="wp-subscription-header">
		<div class="wp-subscription-header-content">
			<div>
				<h1 class="wp-subscription-title"><?php esc_html_e( 'Subscription Statistics', 'wp_subscription_pro' ); ?></h1>
				<p class="wp-subscription-subtitle"><?php esc_html_e( 'Data-driven insights for your subscription business', 'wp_subscription_pro' ); ?></p>
			</div>
			<div class="wp-subscription-actions">
				<a href="<?php echo esc_url( add_query_arg( 'refresh', 'true', remove_query_arg( 'refresh' ) ) ); ?>" class="wp-subscription-btn">
					<?php esc_html_e( '↻ Refresh Data', 'wp_subscription_pro' ); ?>
				</a>
			</div>
		</div>
	</div>

	<!-- Executive Summary -->
	<div class="wp-subscription-section">
		<h2 class="wp-subscription-section-title"><?php esc_html_e( 'Executive Summary', 'wp_subscription_pro' ); ?></h2>
		<div class="wp-subscription-metrics-grid">
			<div class="wp-subscription-metric-card">
				<div class="wp-subscription-metric-value"><?php echo number_format( $metrics['active_subscriptions'] ); ?></div>
				<div class="wp-subscription-metric-label"><?php esc_html_e( 'Active Subscriptions', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-metric-description"><?php esc_html_e( 'Your recurring revenue base', 'wp_subscription_pro' ); ?></div>
			</div>
			<div class="wp-subscription-metric-card">
				<div class="wp-subscription-metric-value"><?php echo wp_kses_post( format_price( $revenue_metrics['mrr'] ) ); ?></div>
				<div class="wp-subscription-metric-label"><?php esc_html_e( 'Monthly Recurring Revenue', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-metric-description"><?php esc_html_e( 'Predictable monthly income', 'wp_subscription_pro' ); ?></div>
			</div>
			<div class="wp-subscription-metric-card">
				<div class="wp-subscription-metric-value"><?php echo number_format( $metrics['trial_conversion_rate'], 1 ); ?>%</div>
				<div class="wp-subscription-metric-label"><?php esc_html_e( 'Trial Conversion Rate', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-metric-description"><?php esc_html_e( 'Trial to paid success', 'wp_subscription_pro' ); ?></div>
			</div>
			<div class="wp-subscription-metric-card">
				<div class="wp-subscription-metric-value"><?php echo number_format( $metrics['churn_rate'], 1 ); ?>%</div>
				<div class="wp-subscription-metric-label"><?php esc_html_e( 'Churn Rate', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-metric-description"><?php esc_html_e( 'Customer retention', 'wp_subscription_pro' ); ?></div>
			</div>
		</div>
	</div>

	<!-- Charts & Insights -->
	<div class="wp-subscription-charts-grid">
		<div class="wp-subscription-chart-container">
			<h3 class="wp-subscription-chart-title"><?php esc_html_e( 'Revenue & Subscription Trends', 'wp_subscription_pro' ); ?></h3>
			<canvas id="mainTrendsChart" style="width:100%;height:300px;"></canvas>
		</div>

		<div class="wp-subscription-insights">
			<h3 class="wp-subscription-chart-title"><?php esc_html_e( 'Quick Insights', 'wp_subscription_pro' ); ?></h3>
			
			<?php if ( $metrics['trial_conversion_rate'] < 20 ) : ?>
			<div class="wp-subscription-insight warning">
				<div class="wp-subscription-insight-title"><?php esc_html_e( 'Low Trial Conversion', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-insight-text"><?php esc_html_e( 'Your conversion rate is below industry average. Consider improving your onboarding process.', 'wp_subscription_pro' ); ?></div>
			</div>
			<?php endif; ?>

			<?php if ( $metrics['churn_rate'] > 10 ) : ?>
			<div class="wp-subscription-insight danger">
				<div class="wp-subscription-insight-title"><?php esc_html_e( 'High Churn Rate', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-insight-text"><?php esc_html_e( 'Your churn rate is concerning. Focus on customer retention strategies.', 'wp_subscription_pro' ); ?></div>
			</div>
			<?php endif; ?>

			<?php if ( $revenue_metrics['mrr'] > 0 ) : ?>
			<div class="wp-subscription-insight success">
				<div class="wp-subscription-insight-title"><?php esc_html_e( 'Revenue Growth', 'wp_subscription_pro' ); ?></div>
				<?php // translators: %s: formatted monthly recurring revenue (MRR) ?>
				<div class="wp-subscription-insight-text"><?php printf( esc_html__( 'Your MRR is %s. Great foundation for scaling!', 'wp_subscription_pro' ), wp_kses_post( format_price( $revenue_metrics['mrr'] ) ) ); ?></div>
			</div>
			<?php endif; ?>

			<div class="wp-subscription-insight">
				<div class="wp-subscription-insight-title"><?php esc_html_e( 'Pro Tip', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-insight-text"><?php esc_html_e( 'Focus on your top-performing products and optimize trial conversions for maximum growth.', 'wp_subscription_pro' ); ?></div>
			</div>
		</div>
	</div>

	<!-- Performance Metrics -->
	<div class="wp-subscription-performance-grid">
		<div class="wp-subscription-performance-card">
			<h3 class="wp-subscription-performance-title"><?php esc_html_e( 'Revenue Performance', 'wp_subscription_pro' ); ?></h3>
			<div class="wp-subscription-performance-grid-inner">
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo wp_kses_post( format_price( $revenue_metrics['total_revenue'] ) ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Total Revenue', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo wp_kses_post( format_price( $revenue_metrics['arr'] ) ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Annual Recurring', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo wp_kses_post( format_price( $revenue_metrics['net_revenue'] ) ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Net Revenue', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo wp_kses_post( format_price( $metrics['average_subscription_value'] ) ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Avg. Value', 'wp_subscription_pro' ); ?></div>
				</div>
			</div>
		</div>
		
		<div class="wp-subscription-performance-card">
			<h3 class="wp-subscription-performance-title"><?php esc_html_e( 'Trial Performance', 'wp_subscription_pro' ); ?></h3>
			<div class="wp-subscription-performance-grid-inner">
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $trial_conversion_data['total_trials'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Total Trials', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $trial_conversion_data['converted_trials'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Converted', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $trial_conversion_data['active_trials'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Active Trials', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo wp_kses_post( format_price( $trial_conversion_data['trial_revenue'] ) ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Trial Revenue', 'wp_subscription_pro' ); ?></div>
				</div>
			</div>
		</div>
		
		<div class="wp-subscription-performance-card">
			<h3 class="wp-subscription-performance-title"><?php esc_html_e( 'Subscription Health', 'wp_subscription_pro' ); ?></h3>
			<div class="wp-subscription-performance-grid-inner">
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $metrics['active_subscriptions'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Active', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $metrics['cancelled_subscriptions'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Cancelled', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $metrics['trial_subscriptions'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Trials', 'wp_subscription_pro' ); ?></div>
				</div>
				<div class="wp-subscription-performance-metric">
					<div class="wp-subscription-performance-value"><?php echo number_format( $metrics['total_products'] ); ?></div>
					<div class="wp-subscription-performance-label"><?php esc_html_e( 'Products', 'wp_subscription_pro' ); ?></div>
				</div>
			</div>
		</div>
	</div>

	<!-- Products & Actions -->
	<div class="wp-subscription-products-grid">
		<div class="wp-subscription-performance-card">
			<h3 class="wp-subscription-performance-title"><?php esc_html_e( 'Top Performing Products', 'wp_subscription_pro' ); ?></h3>
			<?php if ( ! empty( $popular_subscriptions ) ) : ?>
			<div class="wp-subscription-products-list">
				<?php foreach ( array_slice( $popular_subscriptions, 0, 5 ) as $index => $subscription ) : ?>
				<div class="wp-subscription-product-item">
					<div class="wp-subscription-product-info">
						<div class="wp-subscription-product-rank <?php echo $index < 3 ? 'top' : ''; ?>">
							<?php echo wp_kses_post( $index + 1 ); ?>
						</div>
						<div>
							<div class="wp-subscription-product-name"><?php echo esc_html( $subscription['product_name'] ); ?></div>
							<?php // translators: %s: number of subscriptions ?>
							<div class="wp-subscription-product-count"><?php printf( esc_html__( '%s subscriptions', 'wp_subscription_pro' ), number_format( $subscription['subscription_count'] ) ); ?></div>
						</div>
					</div>
					<div class="wp-subscription-product-revenue">
						<div class="wp-subscription-product-total"><?php echo wp_kses_post( format_price( $subscription['total_revenue'] ) ); ?></div>
						<?php // translators: %s: average revenue amount ?>
						<div class="wp-subscription-product-avg"><?php printf( esc_html__( '%s avg', 'wp_subscription_pro' ), wp_kses_post( format_price( $subscription['average_revenue'] ) ) ); ?></div>
					</div>
				</div>
				<?php endforeach; ?>
			</div>
			<?php else : ?>
			<div class="wp-subscription-empty-state">
				<div class="wp-subscription-empty-icon">📊</div>
				<div><?php esc_html_e( 'No subscription data available yet', 'wp_subscription_pro' ); ?></div>
			</div>
			<?php endif; ?>
		</div>

		<div class="wp-subscription-performance-card">
			<h3 class="wp-subscription-performance-title"><?php esc_html_e( 'Action Items', 'wp_subscription_pro' ); ?></h3>
			
			<?php if ( $metrics['trial_conversion_rate'] < 20 ) : ?>
			<div class="wp-subscription-action-item warning">
				<div class="wp-subscription-action-title"><?php esc_html_e( 'Improve Trial Conversion', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-action-text"><?php esc_html_e( 'Your conversion rate is below average. Consider:', 'wp_subscription_pro' ); ?></div>
				<ul class="wp-subscription-action-list">
					<li><?php esc_html_e( 'Enhancing onboarding experience', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Adding value during trial period', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Improving product messaging', 'wp_subscription_pro' ); ?></li>
				</ul>
			</div>
			<?php endif; ?>

			<?php if ( $metrics['churn_rate'] > 10 ) : ?>
			<div class="wp-subscription-action-item danger">
				<div class="wp-subscription-action-title"><?php esc_html_e( 'Reduce Churn Rate', 'wp_subscription_pro' ); ?></div>
				<div class="wp-subscription-action-text"><?php esc_html_e( 'High churn indicates issues. Focus on:', 'wp_subscription_pro' ); ?></div>
				<ul class="wp-subscription-action-list">
					<li><?php esc_html_e( 'Customer success programs', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Regular check-ins', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Product improvements', 'wp_subscription_pro' ); ?></li>
				</ul>
			</div>
			<?php endif; ?>

			<?php if ( ! empty( $popular_subscriptions ) ) : ?>
			<div class="wp-subscription-action-item success">
				<div class="wp-subscription-action-title"><?php esc_html_e( 'Your best performers', 'wp_subscription_pro' ); ?></div>
				<ul class="wp-subscription-action-list">
					<li><?php printf( '%s - %s', esc_html( $popular_subscriptions[0]['product_name'] ), wp_kses_post( format_price( $popular_subscriptions[0]['total_revenue'] ) ) ); ?></li>
					<?php if ( isset( $popular_subscriptions[1] ) ) : ?>
					<li><?php printf( '%s - %s', esc_html( $popular_subscriptions[1]['product_name'] ), wp_kses_post( format_price( $popular_subscriptions[1]['total_revenue'] ) ) ); ?></li>
					<?php endif; ?>
				</ul>
			</div>
			<?php endif; ?>

			<div class="wp-subscription-action-item">
				<div class="wp-subscription-action-title"><?php esc_html_e( 'Recommended next steps', 'wp_subscription_pro' ); ?></div>
				<ul class="wp-subscription-action-list">
					<li><?php esc_html_e( 'Analyze customer feedback', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Optimize pricing strategy', 'wp_subscription_pro' ); ?></li>
					<li><?php esc_html_e( 'Expand product offerings', 'wp_subscription_pro' ); ?></li>
				</ul>
			</div>
		</div>
	</div>

	<!-- Detailed Analytics -->
	<div class="wp-subscription-section">
		<h2 class="wp-subscription-section-title"><?php esc_html_e( 'Detailed Analytics', 'wp_subscription_pro' ); ?></h2>
		<canvas id="detailedAnalyticsChart" style="width:100%;height:400px;"></canvas>
	</div>

	<!-- Footer -->
	<div class="wp-subscription-footer">
		<div class="wp-subscription-footer-text">
			<?php // translators: %s: human-readable date/time when stats were generated ?>
			<?php printf( esc_html__( 'Last updated: %s', 'wp_subscription_pro' ), wp_kses_post( current_time( 'F j, Y \a\t g:i A' ) ) ); ?>
		</div>
		<div class="wp-subscription-footer-note">
			<?php esc_html_e( 'Data refreshes automatically. Use the refresh button for real-time updates.', 'wp_subscription_pro' ); ?>
		</div>
	</div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
	// Main Trends Chart
	const mainCtx = document.getElementById('mainTrendsChart').getContext('2d');
	const months = <?php echo wp_json_encode( array_column( $monthly_trends, 'label' ) ); ?>;
	const revenue = <?php echo wp_json_encode( array_column( $monthly_trends, 'revenue' ) ); ?>;
	const actives = <?php echo wp_json_encode( array_column( $monthly_trends, 'active' ) ); ?>;

	new Chart(mainCtx, {
		type: 'bar',
		data: {
			labels: months,
			datasets: [
				{
					label: '<?php echo esc_js( __( 'Revenue', 'wp_subscription_pro' ) ); ?>',
					data: revenue,
					type: 'bar',
					backgroundColor: 'rgba(49, 104, 142, 0.8)',
					borderColor: '#31688e',
					borderWidth: 2,
					borderRadius: 6,
					borderSkipped: false,
					yAxisID: 'y1'
				},
				{
					label: '<?php echo esc_js( __( 'Active Subscriptions', 'wp_subscription_pro' ) ); ?>',
					data: actives,
					type: 'line',
					borderColor: '#35b779',
					backgroundColor: 'rgba(53, 183, 121, 0.1)',
					borderWidth: 4,
					fill: true,
					tension: 0.4,
					pointRadius: 8,
					pointHoverRadius: 12,
					pointBackgroundColor: '#35b779',
					pointBorderColor: '#fff',
					pointBorderWidth: 3,
					yAxisID: 'y'
				}
			]
		},
		options: {
			responsive: true,
			interaction: { mode: 'index', intersect: false },
			plugins: {
				legend: { 
					position: 'top',
					labels: {
						usePointStyle: true,
						padding: 20,
						font: {
							size: 12,
							weight: '600'
						}
					}
				},
				title: { 
					display: true,
					text: '<?php echo esc_js( __( 'Revenue and Subscription Trends', 'wp_subscription_pro' ) ); ?>',
					font: {
						size: 16,
						weight: '600'
					},
					color: '#1a202c',
					padding: 20
				},
				tooltip: {
					backgroundColor: 'rgba(0, 0, 0, 0.9)',
					titleColor: '#fff',
					bodyColor: '#fff',
					borderColor: '#31688e',
					borderWidth: 1,
					cornerRadius: 8,
					displayColors: true,
					padding: 12,
					callbacks: {
						label: function(context) {
							if (context.dataset.label === '<?php echo esc_js( __( 'Revenue', 'wp_subscription_pro' ) ); ?>') {
								return `<?php echo esc_js( __( 'Revenue', 'wp_subscription_pro' ) ); ?>: $${context.parsed.y.toLocaleString()}`;
							} else {
								return `<?php echo esc_js( __( 'Active Subscriptions', 'wp_subscription_pro' ) ); ?>: ${context.parsed.y.toLocaleString()}`;
							}
						}
					}
				}
			},
			scales: {
				x: {
					grid: {
						color: 'rgba(0, 0, 0, 0.05)',
						drawBorder: false
					},
					ticks: {
						font: {
							size: 11,
							weight: '500'
						},
						color: '#718096'
					}
				},
				y: { 
					type: 'linear',
					display: true,
					position: 'left',
					beginAtZero: true, 
					title: { 
						display: true, 
						text: '<?php echo esc_js( __( 'Active Subscriptions', 'wp_subscription_pro' ) ); ?>',
						font: {
							size: 12,
							weight: '600'
						},
						color: '#4a5568'
					},
					grid: {
						color: 'rgba(0, 0, 0, 0.05)',
						drawBorder: false
					},
					ticks: {
						font: {
							size: 11,
							weight: '500'
						},
						color: '#718096'
					}
				},
				y1: {
					type: 'linear',
					display: true,
					position: 'right',
					beginAtZero: true,
					grid: { 
						drawOnChartArea: false,
						color: 'rgba(0, 0, 0, 0.05)'
					},
					title: { 
						display: true, 
						text: '<?php echo esc_js( __( 'Revenue ($)', 'wp_subscription_pro' ) ); ?>',
						font: {
							size: 12,
							weight: '600'
						},
						color: '#4a5568'
					},
					ticks: {
						font: {
							size: 11,
							weight: '500'
						},
						color: '#718096',
						callback: function(value) {
							return '$' + value.toLocaleString();
						}
					}
				}
			}
		}
	});

	// Detailed Analytics Chart
	const detailedCtx = document.getElementById('detailedAnalyticsChart').getContext('2d');
	const totals = <?php echo wp_json_encode( array_column( $monthly_trends, 'total' ) ); ?>;
	const cancelled = <?php echo wp_json_encode( array_column( $monthly_trends, 'cancelled' ) ); ?>;
	const trials = <?php echo wp_json_encode( array_column( $monthly_trends, 'trials' ) ); ?>;
	const conversions = <?php echo wp_json_encode( array_column( $monthly_trends, 'conversions' ) ); ?>;

	// Create bubble chart data - each bubble represents a month with multiple dimensions
	const bubbleData = months.map((month, index) => ({
		x: index, // Month position
		y: totals[index], // Y-axis: Total subscriptions
		r: Math.max(5, (actives[index] / Math.max(...actives)) * 20), // Size: Active subscriptions (normalized)
		month: month,
		active: actives[index],
		cancelled: cancelled[index],
		trial: trials[index],
		conversion: conversions[index]
	}));

	new Chart(detailedCtx, {
		type: 'bubble',
		data: {
			datasets: [{
				label: '<?php echo esc_js( __( 'Subscription Performance', 'wp_subscription_pro' ) ); ?>',
				data: bubbleData,
				backgroundColor: bubbleData.map((_, index) => {
					const colors = ['#31688e', '#35b779', '#dc2626', '#f59e0b', '#8b5cf6', '#06b6d4'];
					return colors[index % colors.length];
				}),
				borderColor: bubbleData.map((_, index) => {
					const colors = ['#31688e', '#35b779', '#dc2626', '#f59e0b', '#8b5cf6', '#06b6d4'];
					return colors[index % colors.length];
				}),
				borderWidth: 2,
				hoverBackgroundColor: bubbleData.map((_, index) => {
					const colors = ['#1e40af', '#059669', '#b91c1c', '#d97706', '#7c3aed', '#0891b2'];
					return colors[index % colors.length];
				}),
				hoverBorderColor: '#fff',
				hoverBorderWidth: 3
			}]
		},
		options: {
			responsive: true,
			interaction: { mode: 'nearest', intersect: false },
			plugins: {
				legend: { 
					display: false
				},
				title: { 
					display: true,
					text: '<?php echo esc_js( __( 'Subscription Performance by Month', 'wp_subscription_pro' ) ); ?>',
					font: {
						size: 16,
						weight: '600'
					},
					color: '#1a202c',
					padding: 20
				},
				tooltip: {
					backgroundColor: 'rgba(0, 0, 0, 0.9)',
					titleColor: '#fff',
					bodyColor: '#fff',
					borderColor: '#31688e',
					borderWidth: 1,
					cornerRadius: 8,
					displayColors: false,
					padding: 12,
					callbacks: {
						title: function(context) {
							return context[0].raw.month;
						},
						label: function(context) {
							const data = context.raw;
							return [
								`<?php echo esc_js( __( 'Total Subscriptions', 'wp_subscription_pro' ) ); ?>: ${data.y}`,
								`<?php echo esc_js( __( 'Active Subscriptions', 'wp_subscription_pro' ) ); ?>: ${data.active}`,
								`<?php echo esc_js( __( 'Cancelled', 'wp_subscription_pro' ) ); ?>: ${data.cancelled}`,
								`<?php echo esc_js( __( 'Trials', 'wp_subscription_pro' ) ); ?>: ${data.trial}`,
								`<?php echo esc_js( __( 'Conversions', 'wp_subscription_pro' ) ); ?>: ${data.conversion}`,
								`<?php echo esc_js( __( 'Bubble Size: Active Subscriptions', 'wp_subscription_pro' ) ); ?>`
							];
						}
					}
				}
			},
			scales: {
				x: {
					type: 'linear',
					position: 'bottom',
					title: {
						display: true,
						text: '<?php echo esc_js( __( 'Month', 'wp_subscription_pro' ) ); ?>',
						font: {
							size: 12,
							weight: '600'
						},
						color: '#4a5568'
					},
					ticks: {
						stepSize: 1,
						callback: function(value) {
							if (value >= 0 && value < months.length) {
								return months[value];
							}
							return '';
						},
						font: {
							size: 11,
							weight: '500'
						},
						color: '#718096'
					},
					grid: {
						color: 'rgba(0, 0, 0, 0.05)',
						drawBorder: false
					}
				},
				y: {
					type: 'linear',
					position: 'left',
					title: {
						display: true,
						text: '<?php echo esc_js( __( 'Total Subscriptions', 'wp_subscription_pro' ) ); ?>',
						font: {
							size: 12,
							weight: '600'
						},
						color: '#4a5568'
					},
					ticks: {
						font: {
							size: 11,
							weight: '500'
						},
						color: '#718096'
					},
					grid: {
						color: 'rgba(0, 0, 0, 0.05)',
						drawBorder: false
					}
				}
			}
		}
	});
});
</script>
