<?php

namespace SpringDevs\SubscriptionPro\Admin;

use SpringDevs\Subscription\Admin\SettingsHelper;

/**
 * Class Settings
 *
 * @package SpringDevs\SubscriptionPro\Admin
 */
class Settings {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_filter( 'subscrpt_settings_fields', array( $this, 'add_setting_fields' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting( 'wp_subscription_settings', 'subscrpt_renewal_price' );
		register_setting( 'wp_subscription_settings', 'subscrpt_early_renew' );
		register_setting( 'wpsubscription_api_key', 'wpsubscription_api_key' );
		register_setting( 'wpsubscription_api_enabled', 'wpsubscription_api_enabled' );

		// Payment Failure Handling Settings
		register_setting( 'wp_subscription_settings', 'subscrpt_default_max_payment_retries' );
		register_setting( 'wp_subscription_settings', 'subscrpt_default_payment_grace_period' );
		register_setting( 'wp_subscription_settings', 'subscrpt_enable_payment_failure_emails' );
		register_setting( 'wp_subscription_settings', 'subscrpt_payment_failure_email_delay' );
		register_setting( 'wp_subscription_settings', 'subscrpt_enable_grace_period_notifications' );
		register_setting( 'wp_subscription_settings', 'subscrpt_grace_period_warning_days' );
	}

	/**
	 * Enqueue admin scripts
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( 'subscrpt_admin_api_js' );
		wp_localize_script(
			'subscrpt_admin_api_js',
			'subscrptPro',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'subscrpt_pro_ajax_nonce' ),
				'i18n'    => array(
					'confirmRegenerate' => __( 'Are you sure you want to regenerate the API key? This will invalidate the existing key.', 'wp_subscription_pro' ),
					'apiKeyRegenerated' => __( 'API key has been regenerated successfully.', 'wp_subscription_pro' ),
					'error'             => __( 'An error occurred. Please try again.', 'wp_subscription_pro' ),
				),
			)
		);
	}

	/**
	 * Add setting fields
	 *
	 * @param array $settings_fields Settings fields.
	 * @return array
	 */
	public function add_setting_fields( $settings_fields ) {
		// Pro core settings fields
		$pro_core_settings_fields = [
			[
				'type'       => 'select',
				'group'      => 'main',
				'priority'   => 7,
				'field_data' => [
					'id'          => 'subscrpt_renewal_price',
					'title'       => __( 'Renewal Price', 'wp_subscription_pro' ),
					'description' => __( 'Choose a price that will be used for subscription renewal.', 'wp_subscription_pro' ),
					'options'     => [
						'subscribed' => __( 'Subscribed Price', 'wp_subscription_pro' ),
						'updated'    => __( 'New/Updated Price', 'wp_subscription_pro' ),
					],
					'selected'    => esc_attr( get_option( 'subscrpt_renewal_price', 'subscribed' ) ),
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'main',
				'priority'   => 8,
				'field_data' => [
					'id'          => 'subscrpt_early_renew',
					'title'       => __( 'Early Renewal', 'wp_subscription_pro' ),
					'label'       => __( 'Accept Early Renewal Payments', 'wp_subscription_pro' ),
					'description' => __( 'With early renewals enabled, customers can renew their subscriptions before the next payment date.', 'wp_subscription_pro' ),
					'value'       => '1',
					'checked'     => '1' === get_option( 'subscrpt_early_renew', '1' ),
				],
			],
		];

		// Grace Period Settings
		$grace_period_fields = [
			[
				'type'       => 'heading',
				'group'      => 'grace_period',
				'priority'   => 3,
				'field_data' => [
					'title' => __( 'Grace Period Settings', 'wp_subscription_pro' ),
				],
			],
			[
				'type'       => 'input',
				'group'      => 'grace_period',
				'priority'   => 1,
				'field_data' => [
					'id'          => 'subscrpt_default_payment_grace_period',
					'title'       => __( 'Grace Period (Days)', 'wp_subscription_pro' ),
					'description' => __( 'Days to maintain access after subscriptions expires. (0 = No grace period. Max 30 days)', 'wp_subscription_pro' ),
					'value'       => esc_attr( get_option( 'subscrpt_default_payment_grace_period', '7' ) ),
					'type'        => 'number',
					'attributes'  => [
						'min' => 0,
						'max' => 30,
					],
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'grace_period',
				'priority'   => 2,
				'field_data' => [
					'id'          => 'subscrpt_enable_grace_period_notifications',
					'title'       => __( 'Grace Period Notifications', 'wp_subscription_pro' ),
					'label'       => __( 'Send notifications during grace period', 'wp_subscription_pro' ),
					'description' => __( 'Customers will receive warnings before their access is suspended.', 'wp_subscription_pro' ),
					'value'       => '1',
					'checked'     => '1' === get_option( 'subscrpt_enable_grace_period_notifications', '1' ),
				],
			],
			[
				'type'       => 'input',
				'group'      => 'grace_period',
				'priority'   => 3,
				'field_data' => [
					'id'          => 'subscrpt_grace_period_warning_days',
					'title'       => __( 'Grace Period Warning (Days Before)', 'wp_subscription_pro' ),
					'description' => __( 'Send warning emails this many days before grace period expires. (1-7 days)', 'wp_subscription_pro' ),
					'value'       => esc_attr( get_option( 'subscrpt_grace_period_warning_days', '2' ) ),
					'type'        => 'number',
					'attributes'  => [
						'min' => 1,
						'max' => 7,
					],
				],
			],
		];

		// Payment Failure Handling Settings
		$payment_failure_fields = [
			[
				'type'       => 'heading',
				'group'      => 'payment_failure',
				'priority'   => 4,
				'field_data' => [
					'title' => __( 'Payment Failure Handling', 'wp_subscription_pro' ),
				],
			],
			[
				'type'       => 'input',
				'group'      => 'payment_failure',
				'priority'   => 1,
				'field_data' => [
					'id'          => 'subscrpt_default_max_payment_retries',
					'title'       => __( 'Default Max Payment Retries', 'wp_subscription_pro' ),
					'description' => __( 'Default number of automatic retry attempts for failed payments. (0 = No retries. Max 10 retries)', 'wp_subscription_pro' ),
					'value'       => esc_attr( get_option( 'subscrpt_default_max_payment_retries', '3' ) ),
					'type'        => 'number',
					'attributes'  => [
						'min' => 0,
						'max' => 10,
					],
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'payment_failure',
				'priority'   => 2,
				'field_data' => [
					'id'          => 'subscrpt_enable_payment_failure_emails',
					'title'       => __( 'Enable Payment Failure Emails', 'wp_subscription_pro' ),
					'label'       => __( 'Send email notifications when payments fail', 'wp_subscription_pro' ),
					'description' => __( 'Customers will receive emails about failed payments and retry attempts.', 'wp_subscription_pro' ),
					'value'       => '1',
					'checked'     => '1' === get_option( 'subscrpt_enable_payment_failure_emails', '1' ),
				],
			],
			[
				'type'       => 'input',
				'group'      => 'payment_failure',
				'priority'   => 3,
				'field_data' => [
					'id'          => 'subscrpt_payment_failure_email_delay',
					'title'       => __( 'Payment Failure Email Delay (Hours)', 'wp_subscription_pro' ),
					'description' => __( 'Delay before sending payment failure emails to avoid spam during temporary issues. (1-168 hours)', 'wp_subscription_pro' ),
					'value'       => esc_attr( get_option( 'subscrpt_payment_failure_email_delay', '24' ) ),
					'type'        => 'number',
					'attributes'  => [
						'min' => 1,
						'max' => 168,
					],
				],
			],
		];

		// API Settings
		$api_fields = [
			[
				'type'       => 'heading',
				'group'      => 'api_settings',
				'priority'   => 99,
				'field_data' => [
					'title' => __( 'API Settings', 'wp_subscription_pro' ),
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'api_settings',
				'priority'   => 1,
				'field_data' => [
					'id'          => 'wpsubscription_api_enabled',
					'title'       => __( 'Enable API', 'wp_subscription_pro' ),
					'description' => __( 'Enable REST API endpoints for subscription actions', 'wp_subscription_pro' ),
					'value'       => 'on',
					'checked'     => 'on' === get_option( 'wpsubscription_api_enabled', 'on' ),
				],
			],
			[
				'type'       => 'join',
				'group'      => 'api_settings',
				'priority'   => 2,
				'field_data' => [
					'title'       => __( 'API Key', 'wp_subscription_pro' ),
					'description' => __( 'API key for external integrations. Keep this secure.', 'wp_subscription_pro' ),
					'elements'    => [
						SettingsHelper::inp_element(
							[
								'id'         => 'wpsubscription_api_key',
								'value'      => esc_attr( get_option( 'wpsubscription_api_key', '' ) ),
								'type'       => 'text',
								'attributes' => [
									'readonly' => true,
								],
							],
							true
						),
						'<button 
							class="button" 
							id="regenerate_api_key" 
							style="border-top-left-radius: 0; border-bottom-left-radius: 0;"
						>' . esc_html__( 'Regenerate API Key', 'wp_subscription_pro' ) . '</button>',
					],
				],
			],
			[
				'type'       => 'input',
				'group'      => 'api_settings',
				'priority'   => 3,
				'field_data' => [
					'id'          => 'subscrpt_api_endpoint_url',
					'title'       => __( 'REST API Endpoint URL', 'wp_subscription_pro' ),
					'description' => __( 'Copy and use this endpoint for all API actions.', 'wp_subscription_pro' ),
					'value'       => esc_url( home_url( '/wp-json/wpsubscription/v1/action' ) ),
					'type'        => 'text',
					'attributes'  => [
						'readonly' => true,
					],
				],
			],
		];

		return array_merge( $settings_fields, $pro_core_settings_fields, $grace_period_fields, $payment_failure_fields, $api_fields );
	}

	public function get_settings_sections() {
		return array(
			array(
				'id'    => 'wpsubscription_general',
				'title' => __( 'General Settings', 'wp_subscription_pro' ),
			),
			array(
				'id'    => 'wpsubscription_payment_failure',
				'title' => __( 'Payment Failure Handling', 'wp_subscription_pro' ),
			),
			array(
				'id'    => 'wpsubscription_api',
				'title' => __( 'API Settings', 'wp_subscription_pro' ),
			),
		);
	}

	/**
	 * Generate a new API key
	 *
	 * @return string
	 */
	private function generate_api_key() {
		$api_key = get_option( 'wpsubscription_api_key' );
		if ( empty( $api_key ) ) {
			$api_key = wp_generate_password( 32, false );
			update_option( 'wpsubscription_api_key', $api_key );
		}
		return $api_key;
	}
}
