<?php

namespace SpringDevs\SubscriptionPro\Admin;

/**
 * Stats class for handling subscription statistics
 */
class Stats {
	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// Cache invalidation hooks
		add_action( 'save_post', array( $this, 'invalidate_cache' ), 10, 2 );
		add_action( 'deleted_post', array( $this, 'invalidate_cache' ) );
		add_action( 'wp_insert_comment', array( $this, 'invalidate_cache' ) );
		add_action( 'delete_comment', array( $this, 'invalidate_cache' ) );

		// Hook into the core plugin's stats page render action
		add_action( 'wp_subscription_render_stats_page', array( $this, 'render_pro_stats_page' ) );
	}

	/**
	 * Render pro stats page
	 */
	public static function render_pro_stats_page() {
		// Check for force refresh
		$self          = new self();
		$force_refresh = isset( $_GET['refresh'] ) && $_GET['refresh'] === 'true';

		if ( $force_refresh ) {
			$self->invalidate_cache();
		}

		// Get comprehensive metrics with caching
		$metrics = $self->get_cached_data(
			'wp_subscription_metrics',
			function () use ( $self ) {
				return $self->get_comprehensive_metrics();
			}
		);

		$popular_subscriptions = $self->get_cached_data(
			'wp_subscription_popular',
			function () use ( $self ) {
				return $self->get_popular_subscriptions();
			}
		);

		$trial_conversion_data = $self->get_cached_data(
			'wp_subscription_trials',
			function () use ( $self ) {
				return $self->get_trial_conversion_data();
			}
		);

		$revenue_metrics = $self->get_cached_data(
			'wp_subscription_revenue',
			function () use ( $self ) {
				return $self->get_revenue_metrics();
			}
		);

		$monthly_trends = $self->get_cached_data(
			'wp_subscription_trends',
			function () use ( $self ) {
				return $self->get_monthly_trends();
			}
		);

		// Include the pro template
		include __DIR__ . '/../../templates/admin/stats.php';
	}

	/**
	 * Enqueue required scripts and styles
	 */
	public function enqueue_scripts( $hook ) {
		if ( 'wp-subscription_page_wp-subscription-stats' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'wp-subscription-stats',
			plugin_dir_url( dirname( __DIR__ ) ) . 'assets/css/admin.css',
			array(),
			defined( 'SUBSCRIPT_PRO_VERSION' ) ? SUBSCRIPT_PRO_VERSION : '1.0.0'
		);

		wp_enqueue_script(
			'wp-subscription-stats',
			plugin_dir_url( dirname( __DIR__ ) ) . 'assets/js/admin-stats.js',
			array( 'jquery' ),
			defined( 'SUBSCRIPT_PRO_VERSION' ) ? SUBSCRIPT_PRO_VERSION : '1.0.0',
			true
		);

		wp_localize_script(
			'wp-subscription-stats',
			'wpSubscriptionStats',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'wp-subscription-stats-nonce' ),
			)
		);
	}

	/**
	 * Get comprehensive subscription metrics
	 *
	 * Note: All calculations exclude subscriptions with status 'trash' and 'pending'
	 * to ensure accurate reporting of actual business metrics.
	 */
	private function get_comprehensive_metrics() {
		global $wpdb;

		$metrics = array(
			'total_subscriptions'        => 0,
			'active_subscriptions'       => 0,
			'cancelled_subscriptions'    => 0,
			'trial_subscriptions'        => 0,
			'total_revenue'              => 0,
			'mrr'                        => 0,
			'trial_conversion_rate'      => 0,
			'churn_rate'                 => 0,
			'average_subscription_value' => 0,
			'total_products'             => 0,
		);

		// Get total subscriptions (excluding pending and trash)
		$total                          = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type='subscrpt_order' 
            AND post_status NOT IN ('trash', 'pending')"
		);
		$metrics['total_subscriptions'] = intval( $total );

		// Get active subscriptions
		$active                          = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type='subscrpt_order' 
            AND post_status='active'"
		);
		$metrics['active_subscriptions'] = intval( $active );

		// Get cancelled subscriptions
		$cancelled                          = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type='subscrpt_order' 
            AND post_status='cancelled'"
		);
		$metrics['cancelled_subscriptions'] = intval( $cancelled );

		// Get trial subscriptions (excluding pending)
		$trial                          = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_trial' 
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''"
		);
		$metrics['trial_subscriptions'] = intval( $trial );

		// Get total revenue (excluding pending)
		$revenue                  = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm.meta_value) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_price'"
		);
		$metrics['total_revenue'] = floatval( $revenue );

		// Get MRR (Monthly Recurring Revenue) - only active subscriptions, excluding trash
		$mrr            = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm.meta_value) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_price'"
		);
		$metrics['mrr'] = floatval( $mrr );

		// Calculate trial conversion rate (excluding pending)
		$total_trials = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_trial' 
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''"
		);

		$converted_trials = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_trial'
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''"
		);

		$metrics['trial_conversion_rate'] = $total_trials > 0 ?
			round( ( $converted_trials / $total_trials ) * 100, 2 ) : 0;

		// Calculate churn rate (cancelled / total, excluding pending)
		$metrics['churn_rate'] = $metrics['total_subscriptions'] > 0 ?
			round( ( $metrics['cancelled_subscriptions'] / $metrics['total_subscriptions'] ) * 100, 2 ) : 0;

		// Calculate average subscription value (excluding pending)
		$metrics['average_subscription_value'] = $metrics['total_subscriptions'] > 0 ?
			round( $metrics['total_revenue'] / $metrics['total_subscriptions'], 2 ) : 0;

		// Get total unique products (excluding pending)
		$total_products            = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(DISTINCT pm.meta_value) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_product_id'"
		);
		$metrics['total_products'] = intval( $total_products );

		return $metrics;
	}

	/**
	 * Get popular subscriptions by count
	 *
	 * Note: Excludes subscriptions with status 'trash' and 'pending' for accurate reporting.
	 */
	private function get_popular_subscriptions() {
		global $wpdb;

		$popular = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT 
                pm.meta_value as product_id,
                COUNT(*) as subscription_count,
                SUM(pm2.meta_value) as total_revenue
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            INNER JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_product_id'
            AND pm2.meta_key='_subscrpt_price'
            GROUP BY pm.meta_value
            ORDER BY subscription_count DESC
            LIMIT 10"
		);

		$popular_subscriptions = array();
		foreach ( $popular as $item ) {
			$product = wc_get_product( $item->product_id );
			if ( $product ) {
				$popular_subscriptions[] = array(
					'product_id'         => $item->product_id,
					'product_name'       => $product->get_name(),
					'subscription_count' => intval( $item->subscription_count ),
					'total_revenue'      => floatval( $item->total_revenue ),
					'average_revenue'    => round( floatval( $item->total_revenue ) / intval( $item->subscription_count ), 2 ),
				);
			}
		}

		return $popular_subscriptions;
	}

	/**
	 * Get trial conversion data
	 *
	 * Note: Excludes subscriptions with status 'trash' and 'pending' for accurate trial metrics.
	 */
	private function get_trial_conversion_data() {
		global $wpdb;

		$data = array(
			'total_trials'     => 0,
			'converted_trials' => 0,
			'active_trials'    => 0,
			'expired_trials'   => 0,
			'conversion_rate'  => 0,
			'trial_revenue'    => 0,
		);

		// Get total trials (excluding pending)
		$total_trials         = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_trial' 
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''"
		);
		$data['total_trials'] = intval( $total_trials );

		// Get converted trials (trials that became active)
		$converted_trials         = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_trial'
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''"
		);
		$data['converted_trials'] = intval( $converted_trials );

		// Get active trials
		$active_trials         = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT COUNT(*) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_trial_mode'
            AND pm.meta_value='on'"
		);
		$data['active_trials'] = intval( $active_trials );

		// Get expired trials
		$data['expired_trials'] = $data['total_trials'] - $data['converted_trials'] - $data['active_trials'];

		// Calculate conversion rate
		$data['conversion_rate'] = $data['total_trials'] > 0 ?
			round( ( $data['converted_trials'] / $data['total_trials'] ) * 100, 2 ) : 0;

		// Get trial revenue
		$trial_revenue         = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm2.meta_value) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            INNER JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_trial'
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''
            AND pm2.meta_key='_subscrpt_price'"
		);
		$data['trial_revenue'] = floatval( $trial_revenue );

		return $data;
	}

	/**
	 * Get revenue metrics
	 *
	 * Note: All revenue calculations exclude subscriptions with status 'trash' and 'pending'
	 * to ensure accurate financial reporting.
	 */
	private function get_revenue_metrics() {
		global $wpdb;

		$metrics = array(
			'total_revenue'           => 0,
			'mrr'                     => 0,
			'arr'                     => 0,
			'average_monthly_revenue' => 0,
			'revenue_growth_rate'     => 0,
			'net_revenue'             => 0,
		);

		// Get total revenue (excluding pending)
		$total_revenue            = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm.meta_value) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_price'"
		);
		$metrics['total_revenue'] = floatval( $total_revenue );

		// Get MRR (Monthly Recurring Revenue) - only active subscriptions, excluding trash
		$mrr            = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm.meta_value) FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='active'
            AND pm.meta_key='_subscrpt_price'"
		);
		$metrics['mrr'] = floatval( $mrr );

		// Calculate ARR (Annual Recurring Revenue)
		$metrics['arr'] = $metrics['mrr'] * 12;

		// Calculate average monthly revenue (last 12 months, excluding pending)
		$now   = current_time( 'timestamp' );
		$start = strtotime( '-11 months', $now );
		$start = strtotime( date( 'Y-m-01 00:00:00', $start ) );

		$monthly_revenue                    = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT SUM(pm.meta_value) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status NOT IN ('trash', 'pending')
            AND pm.meta_key='_subscrpt_price'
            AND p.post_date_gmt >= %s",
				gmdate( 'Y-m-d H:i:s', $start )
			)
		);
		$metrics['average_monthly_revenue'] = round( floatval( $monthly_revenue ) / 12, 2 );

		// Calculate net revenue (total - cancelled)
		$cancelled_revenue      = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT SUM(pm.meta_value) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type='subscrpt_order' 
            AND p.post_status='cancelled'
            AND pm.meta_key='_subscrpt_price'"
		);
		$metrics['net_revenue'] = $metrics['total_revenue'] - floatval( $cancelled_revenue );

		return $metrics;
	}

	/**
	 * Get monthly trends data
	 *
	 * Note: Excludes subscriptions with status 'trash' and 'pending' for accurate trend analysis.
	 */
	private function get_monthly_trends() {
		global $wpdb;
		$months = [];
		$now    = current_time( 'timestamp' );

		// Get last 12 months data
		for ( $i = 11; $i >= 0; $i-- ) {
			$month          = strtotime( "-{$i} month", $now );
			$key            = date( 'Y-m', $month );
			$months[ $key ] = [
				'label'       => date( 'M Y', $month ),
				'total'       => 0,
				'active'      => 0,
				'cancelled'   => 0,
				'revenue'     => 0.0,
				'trials'      => 0,
				'conversions' => 0,
			];
		}

		// Get subscription data (excluding pending)
		$start = strtotime( '-11 months', $now );
		$start = strtotime( date( 'Y-m-01 00:00:00', $start ) );
		$end   = strtotime( date( 'Y-m-t 23:59:59', $now ) );

		$subs = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT ID, post_status, post_date_gmt FROM {$wpdb->posts} 
            WHERE post_type='subscrpt_order' 
            AND post_status NOT IN ('trash', 'pending')
            AND post_date_gmt >= %s 
            AND post_date_gmt <= %s",
				gmdate( 'Y-m-d H:i:s', $start ),
				gmdate( 'Y-m-d H:i:s', $end )
			)
		);

		// Process subscription data
		foreach ( $subs as $sub ) {
			$month = date( 'Y-m', strtotime( $sub->post_date_gmt ) );
			if ( ! isset( $months[ $month ] ) ) {
				continue;
			}

			++$months[ $month ]['total'];
			if ( $sub->post_status === 'active' ) {
				++$months[ $month ]['active'];
			}
			if ( $sub->post_status === 'cancelled' ) {
				++$months[ $month ]['cancelled'];
			}

			$price                        = get_post_meta( $sub->ID, '_subscrpt_price', true );
			$months[ $month ]['revenue'] += floatval( $price );

			// Check if trial
			$trial = get_post_meta( $sub->ID, '_subscrpt_trial', true );
			if ( $trial ) {
				++$months[ $month ]['trials'];
				if ( $sub->post_status === 'active' ) {
					++$months[ $month ]['conversions'];
				}
			}
		}

		return $months;
	}

	/**
	 * Get subscription metrics (legacy method for backward compatibility)
	 */
	private function get_subscription_metrics() {
		return $this->get_comprehensive_metrics();
	}

	/**
	 * Invalidate cache when subscription data changes
	 */
	public function invalidate_cache( $post_id = null, $post = null ) {
		if ( $post && $post->post_type === 'subscrpt_order' ) {
			wp_cache_delete( 'wp_subscription_metrics', 'subscription_pro' );
			wp_cache_delete( 'wp_subscription_popular', 'subscription_pro' );
			wp_cache_delete( 'wp_subscription_trials', 'subscription_pro' );
			wp_cache_delete( 'wp_subscription_revenue', 'subscription_pro' );
			wp_cache_delete( 'wp_subscription_trends', 'subscription_pro' );
		}
	}

	/**
	 * Get cached data or calculate if not cached
	 */
	private function get_cached_data( $cache_key, $callback, $expiration = 3600 ) {
		$cached_data = wp_cache_get( $cache_key, 'subscription_pro' );

		if ( false === $cached_data ) {
			$cached_data = $callback();
			wp_cache_set( $cache_key, $cached_data, 'subscription_pro', $expiration );
		}

		return $cached_data;
	}
}
