<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Helper;

/**
 * Delivery class.
 */
class Delivery {

	/**
	 * Initialize the class.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register' ) );
		add_action( 'subscrpt_subscription_activated', array( $this, 'generate_schedules' ), 20 );
	}

	/**
	 * After WordPress init.
	 *
	 * @return void
	 */
	public function register() {
		$this->register_post_type();
		$this->register_post_status();
	}

	/**
	 * Register custom post type `subscrpt_delivery`.
	 *
	 * @return void
	 */
	private function register_post_type() {
		$labels = array(
			'name'              => __( 'Delivery Schedules', 'wp_subscription_pro' ),
			'singular_name'     => __( 'Delivery Schedule', 'wp_subscription_pro' ),
			'name_admin_bar'    => __( 'Delivery Schedule\'s', 'wp_subscription_pro' ),
			'archives'          => __( 'Item Archives', 'wp_subscription_pro' ),
			'attributes'        => __( 'Item Attributes', 'wp_subscription_pro' ),
			'parent_item_colon' => __( 'Parent :', 'wp_subscription_pro' ),
			'all_items'         => __( 'Delivery Schedules', 'wp_subscription_pro' ),
			'add_new_item'      => __( 'Add New Schedule', 'wp_subscription_pro' ),
			'add_new'           => __( 'Add Schedule', 'wp_subscription_pro' ),
			'new_item'          => __( 'New Schedule', 'wp_subscription_pro' ),
			'edit_item'         => __( 'Edit Schedule', 'wp_subscription_pro' ),
			'update_item'       => __( 'Update Schedule', 'wp_subscription_pro' ),
			'view_item'         => __( 'View Schedule', 'wp_subscription_pro' ),
			'view_items'        => __( 'View Schedule', 'wp_subscription_pro' ),
			'search_items'      => __( 'Search Schedule', 'wp_subscription_pro' ),
			'not_found'         => __( 'No schedules found.', 'wp_subscription_pro' ),
			'item_updated'      => __( 'Delivery Schedule updated.', 'wp_subscription_pro' ),
		);

		$args = array(
			'label'                 => __( 'Delivery Schedules', 'wp_subscription_pro' ),
			'labels'                => $labels,
			'description'           => '',
			'public'                => false,
			'publicly_queryable'    => false,
			'show_ui'               => true,
			'delete_with_user'      => false,
			'show_in_rest'          => true,
			'rest_base'             => '',
			'rest_controller_class' => 'WP_REST_Posts_Controller',
			'has_archive'           => false,
			'show_in_menu'          => false,
			'show_in_nav_menus'     => false,
			'exclude_from_search'   => false,
			'capability_type'       => 'post',
			'capabilities'          => array(
				'create_posts' => false,
				'edit_post'    => false,
				'delete_post'  => false,
			),
			'hierarchical'          => false,
			'rewrite'               => array(
				'slug'       => 'subscrpt_delivery',
				'with_front' => true,
			),
			'query_var'             => true,
			'supports'              => false,
		);

		register_post_type( 'subscrpt_delivery', $args );

		// Remove quick edit link.
		add_filter( 'post_row_actions', array( $this, 'remove_quick_edit' ), 10, 2 );
	}

	/**
	 * Register custom post status for delivery.
	 *
	 * @return void
	 */
	public function register_post_status() {
		register_post_status(
			'waiting',
			array(
				'label'                     => _x( 'Waiting', 'Delivery Status', 'wp_subscription_pro' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s is the count of deliveries.
				'label_count'               => _n_noop( 'Waiting <span class="count">(%s)</span>', 'Waiting <span class="count">(%s)</span>', 'wp_subscription_pro' ),
				'post_type'                 => array( 'subscrpt_delivery' ),
			)
		);

		register_post_status(
			'in_process',
			array(
				'label'                     => _x( 'In Process', 'Delivery Status', 'wp_subscription_pro' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s is the count of deliveries.
				'label_count'               => _n_noop( 'In Process <span class="count">(%s)</span>', 'In Process <span class="count">(%s)</span>', 'wp_subscription_pro' ),
				'post_type'                 => array( 'subscrpt_delivery' ),
			)
		);

		register_post_status(
			'shipped',
			array(
				'label'                     => _x( 'Shipped', 'Delivery Status', 'wp_subscription_pro' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s is the count of deliveries.
				'label_count'               => _n_noop( 'Shipped <span class="count">(%s)</span>', 'Shipped <span class="count">(%s)</span>', 'wp_subscription_pro' ),
				'post_type'                 => array( 'subscrpt_delivery' ),
			)
		);

		register_post_status(
			'cancelled',
			array(
				'label'                     => _x( 'Cancelled', 'Delivery Status', 'wp_subscription_pro' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s is the count of deliveries.
				'label_count'               => _n_noop( 'Cancelled <span class="count">(%s)</span>', 'Cancelled <span class="count">(%s)</span>', 'wp_subscription_pro' ),
				'post_type'                 => array( 'subscrpt_delivery' ),
			)
		);
	}

	/**
	 * Generate delivery schedules.
	 *
	 * @param int $subscription_id Subscription Id.
	 *
	 * @return void
	 */
	public function generate_schedules( int $subscription_id ) {
		error_log( "[Delivery] generate_schedules called for subscription_id: $subscription_id" );
		$order_id          = get_post_meta( $subscription_id, '_subscrpt_order_id', true );
		$already_generated = get_post_meta( $subscription_id, "_subscrpt_delivery_generated_$order_id", true );

		error_log( "[Delivery] order_id: $order_id, already_generated: $already_generated" );

		if ( $already_generated ) {
			error_log( '[Delivery] Already generated, exiting.' );
			return;
		}

		$order_item_id = get_post_meta( $subscription_id, '_subscrpt_order_item_id', true );
		$order_item    = wc_get_order_item_meta( $order_item_id, '_subscrpt_delivery' );
		error_log( "[Delivery] order_item_id: $order_item_id, order_item: " . print_r( $order_item, true ) );
		if ( ! isset( $order_item['delivery_time'] ) || empty( $order_item['delivery_time'] ) ) {
			error_log( '[Delivery] No delivery_time in order_item, exiting.' );
			return;
		}
		$delivery_time = $order_item['delivery_time'];

		// Get the subscription's next date.
		$next_date = (int) get_post_meta( $subscription_id, '_subscrpt_next_date', true );
		error_log( "[Delivery] next_date: $next_date" );

		$delivery_timing_option = $order_item['delivery_option'];
		$type                   = Helper::get_typos( $delivery_time, $delivery_timing_option );

		$delivery_day_sync = $order_item['delivery_day_sync'];
		$delivery_day      = $order_item['delivery_day'];

		// Generate delivery schedule.
		$delivery_date  = time();
		$delivery_posts = array();
		$post_meta      = array();

		$time_string = "+$delivery_time $type";
		if ( $delivery_day_sync ) {
			$time_string   = "next $delivery_day";
			$delivery_date = strtotime( $delivery_time . $type );
		}

		error_log( "[Delivery] Initial delivery_date: $delivery_date, time_string: $time_string" );

		$loop_count = 0;
		while ( $delivery_date < $next_date ) {
			$delivery_date = strtotime( $time_string, $delivery_date );

			error_log( "[Delivery] Loop: delivery_date: $delivery_date, next_date: $next_date" );

			$delivery_posts[] = array(
				'post_type'   => 'subscrpt_delivery',
				'post_status' => 'waiting',
				'post_title'  => 'Delivery for Subscription #%d',
			);

			$post_meta[] = array(
				'_subscrpt_subscription_id' => $subscription_id,
				'_subscrpt_shipping_on'     => $delivery_date,
			);
			++$loop_count;
		}
		error_log( "[Delivery] Total delivery posts to create: $loop_count" );

		// Bulk insert posts.
		$delivery_ids = $this->bulk_insert_posts( $delivery_posts );
		error_log( '[Delivery] Inserted delivery post IDs: ' . print_r( $delivery_ids, true ) );

		// Bulk update post meta.
		$this->bulk_update_post_meta( $delivery_ids, $post_meta );

		// Mark as generated to prevent duplicate generation.
		update_post_meta( $subscription_id, "_subscrpt_delivery_generated_$order_id", time() );
	}

	/**
	 * Bulk insert posts.
	 *
	 * @param array $posts Array of post data.
	 * @return array Array of inserted post IDs.
	 */
	private function bulk_insert_posts( $posts ) {
		global $wpdb;

		if ( empty( $posts ) ) {
			return array();
		}

		$post_ids      = array();
		$values        = array();
		$place_holders = array();

		foreach ( $posts as $post ) {
			$values          = array_merge(
				$values,
				array(
					$post['post_type'],
					$post['post_status'],
					$post['post_title'],
					current_time( 'mysql' ),
					current_time( 'mysql' ),
				)
			);
			$place_holders[] = '(%s, %s, %s, %s, %s)';
		}

		if ( empty( $place_holders ) ) {
			return array();
		}

		$query  = "INSERT INTO {$wpdb->posts} (post_type, post_status, post_title, post_date, post_modified) VALUES ";
		$query .= implode( ', ', $place_holders );

		 // phpcs:ignore WordPress.DB
		$wpdb->query( $wpdb->prepare( $query, $values ) );

		$start_id   = $wpdb->insert_id;
		$post_count = count( $posts );
		$post_ids   = range( $start_id, $start_id + $post_count - 1 );

		return $post_ids;
	}

	/**
	 * Bulk update post meta.
	 *
	 * @param array $post_ids Array of post IDs.
	 * @param array $meta_data Array of meta data for each post.
	 */
	private function bulk_update_post_meta( $post_ids, $meta_data ) {
		global $wpdb;

		if ( empty( $post_ids ) || empty( $meta_data ) ) {
			return;
		}

		$values        = array();
		$place_holders = array();

		foreach ( $post_ids as $index => $post_id ) {
			if ( ! isset( $meta_data[ $index ] ) ) {
				continue;
			}
			foreach ( $meta_data[ $index ] as $meta_key => $meta_value ) {
				$values          = array_merge(
					$values,
					array(
						$post_id,
						$meta_key,
						$meta_value,
					)
				);
				$place_holders[] = '(%d, %s, %s)';
			}
		}

		if ( empty( $place_holders ) ) {
			return;
		}

		$query  = "INSERT INTO {$wpdb->postmeta} (post_id, meta_key, meta_value) VALUES ";
		$query .= implode( ', ', $place_holders );

		 // phpcs:ignore WordPress.DB
		$wpdb->query( $wpdb->prepare( $query, $values ) );
	}

	/**
	 * Remove quick edit link for subscrpt_delivery post type.
	 *
	 * @param array    $actions An array of row action links.
	 * @param \WP_Post $post    The post object.
	 * @return array Modified array of row action links.
	 */
	public function remove_quick_edit( $actions, $post ) {
		if ( 'subscrpt_delivery' === $post->post_type ) {
			unset( $actions['inline hide-if-no-js'] );
			unset( $actions['edit'] );
		}
		return $actions;
	}
}
