<?php
namespace SpringDevs\SubscriptionPro\Illuminate\Emails;

use SpringDevs\SubscriptionPro\Traits\Email;
use WC_Email;

/**
 * Payment Failure Email Class
 */
class PaymentFailure extends WC_Email {
	use Email;

	/**
	 * Initialize the class.
	 */
	public function __construct() {
		$this->customer_email = true;

		$this->id          = 'subscrpt_payment_failure';
		$this->title       = __( 'Subscription Payment Failure', 'wp_subscription_pro' );
		$this->description = __( 'This email is sent to customers when their subscription payment fails.', 'wp_subscription_pro' );

		// Call parent constructor first.
		parent::__construct();

		// Set custom template paths for pro plugin.
		$this->template_base  = WP_SUBSCRIPTION_PRO_TEMPLATES;
		$this->template_html  = 'emails/payment-failure-html.php';
		$this->template_plain = 'emails/plains/payment-failure-plain.php';

		// Triggers for this email.
		add_action( 'subscrpt_payment_failure_email_notification', array( $this, 'trigger' ) );
	}

	/**
	 * Get default subject.
	 *
	 * @return string
	 */
	public function get_default_subject(): string {
		return __( 'Payment Failed - Subscription #{subscription_id}', 'wp_subscription_pro' );
	}

	/**
	 * Get default heading.
	 *
	 * @return string
	 */
	public function get_default_heading(): string {
		return __( 'Payment Failed', 'wp_subscription_pro' );
	}

	/**
	 * Get the email content in HTML format.
	 *
	 * @return string
	 */
	public function get_content_html(): string {
		return $this->render_template( $this->template_html );
	}

	/**
	 * Get the email content in Plain Format.
	 *
	 * @return string
	 */
	public function get_content_plain(): string {
		return $this->render_template( $this->template_plain );
	}

	/**
	 * Render template for email.
	 *
	 * @param string $path Template Path.
	 *
	 * @return string
	 */
	public function render_template( string $path ): string {
		if ( ! $path ) {
			return '';
		}

		return wc_get_template_html(
			$path,
			array_merge(
				array(
					'subscription_id'    => $this->subscription_id,
					'email_heading'      => $this->get_heading(),
					'product_name'       => $this->product_name,
					'qty'                => $this->qty,
					'amount'             => $this->amount,
					'customer_name'      => $this->get_customer_name(),
					'email'              => $this->get_recipient(),
					'failure_count'      => $this->extra['failure_count'] ?? 1,
					'max_retries'        => $this->extra['max_retries'] ?? 3,
					'is_delayed'         => $this->extra['is_delayed'] ?? false,
					'remaining_attempts' => $this->extra['remaining_attempts'] ?? 0,
				),
				$this->extra
			),
			'',
			$this->template_base
		);
	}

	/**
	 * Get customer name for the email template.
	 *
	 * @return string
	 */
	private function get_customer_name(): string {
		if ( ! $this->subscription_id ) {
			return '';
		}

		$customer_id = get_post_meta( $this->subscription_id, '_subscrpt_user_id', true );
		if ( ! $customer_id ) {
			return '';
		}

		$customer = get_user_by( 'id', $customer_id );
		return $customer ? $customer->display_name : '';
	}

	/**
	 * Trigger the email.
	 *
	 * @param int  $subscription_id Subscription ID.
	 * @param int  $failure_count   Failure count.
	 * @param int  $max_retries     Maximum retries.
	 * @param bool $is_delayed      Whether this is a delayed notification.
	 */
	public function trigger( int $subscription_id, $failure_count = 1, $max_retries = 3, $is_delayed = false ) {
		if ( ! $subscription_id ) {
			return;
		}

		$this->subscription_id = $subscription_id;

		// Set extra data for template
		$this->extra = array(
			'failure_count'      => $failure_count,
			'max_retries'        => $max_retries,
			'is_delayed'         => $is_delayed,
			'remaining_attempts' => max( 0, $max_retries - $failure_count ),
		);

		// Set table data for template variables
		$this->set_table_data();

		// Set recipient
		$customer_id = get_post_meta( $subscription_id, '_subscrpt_user_id', true );
		if ( $customer_id ) {
			$customer = get_user_by( 'id', $customer_id );
			if ( $customer ) {
				$this->recipient = $this->get_recipient();
			}
		}

		// Set subject and heading
		$this->subject = $this->get_subject();
		$this->heading = $this->get_heading();

		// Send email
		$this->send( $this->recipient, $this->subject, $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}
}
