<?php
/**
 * Xendit Gateway for WPSubscription Pro
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Xendit
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Gateways\Xendit;

use Exception;
use SpringDevs\Subscription\Illuminate\Helper;
use SpringDevs\Subscription\Illuminate\Action;

/**
 * Xendit Gateway for WPSubscription Pro
 *
 * Handles recurring subscription payments via Xendit credit card tokenization
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Xendit
 */
class Xendit {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into WPSubscription renewal events
		add_action( 'subscrpt_after_create_renew_order', array( $this, 'process_renewal_payment' ), 10, 3 );
		add_action( 'subscrpt_before_saving_renewal_order', array( $this, 'copy_xendit_metadata' ), 10, 3 );

		// Hook to save Xendit metadata after initial payment
		add_action( 'woocommerce_order_status_processing', array( $this, 'save_xendit_metadata' ), 10, 1 );
		add_action( 'woocommerce_order_status_completed', array( $this, 'save_xendit_metadata' ), 10, 1 );

		// Hook for failed renewal payments (debugging and tracking)
		add_action( 'woocommerce_order_status_failed', array( $this, 'on_renewal_failed' ), 10, 1 );

		// Hook for subscription cancellation (extended tracking)
		add_action( 'subscrpt_subscription_cancelled', array( $this, 'on_subscription_cancelled' ), 10, 1 );
	}

	/**
	 * Process renewal payment using Xendit card token
	 *
	 * @param \WC_Order $new_order Renewal order.
	 * @param \WC_Order $old_order Parent order.
	 * @param int       $subscription_id Subscription ID.
	 *
	 * @throws Exception If payment processing fails.
	 */
	public function process_renewal_payment( $new_order, $old_order, $subscription_id ) {
		$payment_method = $new_order->get_payment_method();

		// Only process Xendit credit card payments
		if ( $payment_method !== 'xendit_cc' ) {
			return;
		}

		$order_id = $new_order->get_id();

		try {
			// Get Xendit card token from renewal order metadata
			$card_id = $new_order->get_meta( '_xendit_card_id' );

			if ( ! $card_id ) {
				throw new Exception( 'Missing Xendit card token (_xendit_card_id)' );
			}

			wp_subscrpt_write_log(
				sprintf(
					'Xendit: Processing renewal for Order #%d (Card ID: %s)',
					$order_id,
					$card_id
				)
			);

			// Get Xendit gateway instance
			$xendit_gateway = $this->get_xendit_gateway();
			if ( ! $xendit_gateway ) {
				throw new Exception( 'Xendit gateway not available' );
			}

			// Get Xendit API class
			if ( ! isset( $xendit_gateway->xenditClass ) ) {
				throw new Exception( 'Xendit API class not initialized' );
			}

			$xendit_api = $xendit_gateway->xenditClass;

			// Prepare payment request data
			$request_data = $xendit_gateway->generate_payment_request(
				$new_order,
				$card_id,
				'',
				false,
				true  // is_recurring
			);

			// Create charge via Xendit API
			$response = $xendit_api->createCharge( $request_data );

			// Check for API errors
			if ( ! empty( $response['error_code'] ) ) {
				// Handle duplicate external ID
				if ( $response['error_code'] === 'EXTERNAL_ID_ALREADY_USED_ERROR' ) {
					wp_subscrpt_write_log(
						sprintf(
							'Xendit: External ID conflict for Order #%d, retrying with new ID',
							$order_id
						)
					);

					// Retry with force new external ID
					$request_data = $xendit_gateway->generate_payment_request(
						$new_order,
						$card_id,
						'',
						true,  // force new external ID
						true   // is_recurring
					);

					$response = $xendit_api->createCharge( $request_data );
				}

				// If still error, throw exception
				if ( ! empty( $response['error_code'] ) ) {
					$error_message = isset( $response['message'] ) ? $response['message'] : 'Unknown error';
					if ( ! empty( $response['code'] ) ) {
						$error_message .= ' Code: ' . $response['code'];
					}
					throw new Exception( $error_message );
				}
			}

			// Fire action after payment created
			do_action( 'wpsubscription_xendit_payment_created', $response, $new_order, $subscription_id );

			// Save payment ID
			if ( ! empty( $response['id'] ) ) {
				$new_order->update_meta_data( 'Xendit Payment ID', $response['id'] );
				$new_order->set_transaction_id( $response['id'] );

				// Save fees if available
				if ( ! empty( $response['fee'] ) ) {
					$new_order->update_meta_data( 'Xendit Fee', $response['fee'] );
				}

				$new_order->save();
			}

			// Add order note
			$new_order->add_order_note(
				sprintf(
					// translators: 1: Payment ID, 2: Amount.
					__( 'Xendit renewal payment created: %1$s (Amount: %2$s)', 'wp_subscription_pro' ),
					$response['id'] ?? 'N/A',
					wc_price( $new_order->get_total(), array( 'currency' => $new_order->get_currency() ) )
				)
			);

			// Update order status based on payment status
			if ( ! empty( $response['status'] ) ) {
				if ( $response['status'] === 'CAPTURED' || $response['status'] === 'AUTHORIZED' ) {
					// Payment successful - mark as complete
					$new_order->payment_complete( $response['id'] );
				} elseif ( $response['status'] === 'PENDING' ) {
					// Payment pending
					$new_order->update_status( 'on-hold', __( 'Awaiting Xendit payment confirmation', 'wp_subscription_pro' ) );
				} else {
					// Other statuses
					$new_order->update_status(
						'pending',
						sprintf(
							// translators: 1: Payment status.
							__( 'Xendit payment %s', 'wp_subscription_pro' ),
							$response['status']
						)
					);
				}
			}

			// Log success
			wp_subscrpt_write_log(
				sprintf(
					'Xendit: Renewal payment %s created for Order #%d, Amount: %s %s, Status: %s',
					$response['id'] ?? 'N/A',
					$order_id,
					$new_order->get_total(),
					$new_order->get_currency(),
					$response['status'] ?? 'unknown'
				)
			);

			// Fire action after renewal payment fully processed
			do_action( 'wpsubscription_xendit_after_renewal_payment', $response, $new_order, $subscription_id );

		} catch ( Exception $e ) {
			wp_subscrpt_write_log(
				sprintf(
					'Xendit: Renewal payment failed for Order #%d - %s',
					$order_id,
					$e->getMessage()
				)
			);

			$new_order->add_order_note(
				sprintf(
					// translators: 1: Error message.
					__( 'Xendit renewal payment failed: %s', 'wp_subscription_pro' ),
					$e->getMessage()
				)
			);

			$new_order->update_status(
				'failed',
				sprintf(
					// translators: 1: Error message.
					__( 'Xendit renewal payment failed: %s', 'wp_subscription_pro' ),
					$e->getMessage()
				)
			);
		}
	}

	/**
	 * Copy Xendit metadata from old order to renewal order
	 *
	 * @param \WC_Order $new_order Renewal order.
	 * @param \WC_Order $old_order Parent order.
	 * @param int       $subscription_id Subscription ID.
	 */
	public function copy_xendit_metadata( $new_order, $old_order, $subscription_id ) {
		if ( $old_order->get_payment_method() !== 'xendit_cc' ) {
			return;
		}

		$card_id = $old_order->get_meta( '_xendit_card_id' );

		if ( $card_id ) {
			$new_order->update_meta_data( '_xendit_card_id', $card_id );
		}

		wp_subscrpt_write_log(
			sprintf(
				'Xendit: Copied card token to renewal order #%d (Card ID: %s)',
				$new_order->get_id(),
				$card_id ? $card_id : 'missing'
			)
		);
	}

	/**
	 * Save Xendit card token from initial payment
	 *
	 * @param int $order_id Order ID.
	 */
	public function save_xendit_metadata( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Only process Xendit payments
		if ( $order->get_payment_method() !== 'xendit_cc' ) {
			return;
		}

		// Check if metadata already saved
		if ( $order->get_meta( '_xendit_card_id' ) ) {
			return;
		}

		// Try to get card ID from various sources
		$card_id = $order->get_meta( '_xendit_card_id' );

		// If not found, try to get from payment tokens
		if ( ! $card_id && $order->get_user_id() ) {
			$tokens = \WC_Payment_Tokens::get_customer_tokens( $order->get_user_id(), 'xendit_cc' );
			foreach ( $tokens as $token ) {
				if ( $token->is_default() ) {
					$card_id = $token->get_token();
					break;
				}
			}
		}

		if ( $card_id ) {
			$order->update_meta_data( '_xendit_card_id', $card_id );
			$order->save();

			wp_subscrpt_write_log(
				sprintf(
					'Xendit: Saved card token for Order #%d (Card ID: %s)',
					$order_id,
					$card_id
				)
			);
		} else {
			wp_subscrpt_write_log(
				sprintf(
					'Xendit: Card token not found for Order #%d during metadata save, Order status: %s',
					$order_id,
					$order->get_status()
				)
			);
		}
	}

	/**
	 * Handle failed renewal payment for debugging and tracking
	 *
	 * @param int $order_id Order ID.
	 */
	public function on_renewal_failed( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || $order->get_payment_method() !== 'xendit_cc' ) {
			return;
		}

		// Check if this is a renewal order
		if ( ! $order->get_meta( '_is_renewal' ) ) {
			return;
		}

		// Get subscription ID
		$subscriptions   = Helper::get_subscriptions_from_order( $order_id );
		$subscription_id = ! empty( $subscriptions ) ? $subscriptions[0] : null;

		// Log detailed failure information
		wp_subscrpt_write_log(
			sprintf(
				'Xendit: Renewal payment failed for Order #%d (Subscription: #%s, Card ID: %s, Status: %s)',
				$order_id,
				$subscription_id ? $subscription_id : 'unknown',
				$order->get_meta( '_xendit_card_id' ) ? $order->get_meta( '_xendit_card_id' ) : 'missing',
				$order->get_status()
			)
		);

		// Get payment ID for additional context
		$payment_id = $order->get_meta( 'Xendit Payment ID' ) ? $order->get_meta( 'Xendit Payment ID' ) : $order->get_transaction_id();
		if ( $payment_id ) {
			wp_subscrpt_write_log( sprintf( 'Xendit: Failed payment ID: %s', $payment_id ) );
		}
	}

	/**
	 * Handle subscription cancellation for extended tracking
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function on_subscription_cancelled( $subscription_id ) {
		// Get subscription data to check if it uses Xendit
		$parent_order = Helper::get_parent_order( $subscription_id );

		if ( ! $parent_order || $parent_order->get_payment_method() !== 'xendit_cc' ) {
			return;
		}

		// Get Xendit metadata
		$card_id = $parent_order->get_meta( '_xendit_card_id' );

		// Log cancellation for tracking
		wp_subscrpt_write_log(
			sprintf(
				'Xendit: Subscription #%d cancelled (Card ID: %s)',
				$subscription_id,
				$card_id ? $card_id : 'unknown'
			)
		);

		// Fire custom hook for extensibility
		do_action( 'wpsubscription_xendit_subscription_cancelled', $subscription_id, $card_id );
	}

	/**
	 * Get Xendit gateway instance
	 *
	 * @return \WC_Xendit_CC_Addons|\WC_Xendit_CC|false Gateway instance or false
	 */
	private function get_xendit_gateway() {
		if ( ! function_exists( 'WC' ) || ! WC()->payment_gateways() ) {
			return false;
		}

		$gateways = WC()->payment_gateways()->payment_gateways();

		if ( isset( $gateways['xendit_cc'] ) ) {
			return $gateways['xendit_cc'];
		}

		return false;
	}
}
