<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class Helper || Some Helper Methods
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class Helper {

	/**
	 * Initialize the class.
	 */
	public function __construct() {
		add_filter( 'wpsubs_cart_recurring_items', [ $this, 'get_recurrs_from_cart' ], 10, 2 );
	}

	/**
	 * Get recurring items from cart.
	 *
	 * @param array $recurrs Simple Recurring items.
	 * @param array $cart_items Cart items.
	 */
	public function get_recurrs_from_cart( $recurrs, $cart_items ) {
		foreach ( $cart_items as $key => $cart_item ) {
			$product = $cart_item['data'];
			if ( in_array( $product->get_type(), [ 'variable', 'variation' ], true ) && isset( $cart_item['subscription'] ) ) {
				$cart_subscription = $cart_item['subscription'];
				$type              = $cart_subscription['type'];

				$price_html      = wc_price( (float) $cart_subscription['per_cost'] * (int) $cart_item['quantity'] ) . '/ ' . $type;
				$recurrs[ $key ] = array(
					'trial_status'    => ! is_null( $cart_subscription['trial'] ),
					'price_html'      => $price_html,
					'start_date'      => FreeHelper::start_date( $cart_subscription['trial'] ),
					'next_date'       => FreeHelper::next_date( ( $cart_subscription['time'] ?? 1 ) . ' ' . $cart_subscription['type'], $cart_subscription['trial'] ),
					'can_user_cancel' => $product->get_meta( '_subscrpt_user_cancel' ),
					'max_no_payment'  => $product->get_meta( '_subscrpt_max_no_payment' ),
				);
			}
		}

		return $recurrs;
	}

	/**
	 * Check if variable subscription exits.
	 *
	 * @param int            $variation_id Variation ID.
	 * @param string | array $status       Post Status.
	 *
	 * @return WP_Post | false
	 */
	public static function subscription_variable_exists( $variation_id, $status ) {
		if ( 0 === get_current_user_id() ) {
			return false;
		}

		$args = array(
			'post_type'   => 'subscrpt_order',
			'post_status' => $status,
			'fields'      => 'ids',
			'meta_query'  => array(
				array(
					'key'   => '_subscrpt_variation_id',
					'value' => $variation_id,
				),
			),
			'author'      => get_current_user_id(),
		);

		$posts = get_posts( $args );

		if ( count( $posts ) > 0 ) {
			return $posts[0];
		}

		return false;
	}

	/**
	 * Check trial variable subscription.
	 *
	 * @param int $variation_id Variation ID.
	 *
	 * @return bool
	 */
	public static function check_trial( $variation_id ) {
		return ! self::subscription_variable_exists( $variation_id, array( 'expired', 'pending', 'active', 'on-hold', 'cancelled' ) );
	}

	/**
	 * Create renewal order when subscription expired. [wip]
	 *
	 * @param  int $subscription_id Subscription ID.
	 * @throws \WC_Data_Exception Exception.
	 * @throws \Exception Exception.
	 */
	public static function create_early_renewal_order( int $subscription_id ) {
		$order_item_id = get_post_meta( $subscription_id, '_subscrpt_order_item_id', true );
		$order_id      = wc_get_order_id_by_order_item_id( $order_item_id );
		$old_order     = FreeHelper::check_order_for_renewal( $order_id );

		if ( ! $old_order ) {
			return;
		} else {
			$early_meta = get_post_meta( $subscription_id, '_order_waiting_early_renew_subscrpt_meta', true );
			if ( $early_meta ) {
				if ( ! is_admin() ) {
					return wc_add_notice( __( 'Subscription early renewal order is already processing !', 'wp_subscription_pro' ), 'error' );
				}
				return;
			}
		}

		$order_item         = $old_order->get_item( $order_item_id );
		$subscription_price = get_post_meta( $subscription_id, '_subscrpt_price', true );
		$product_args       = array(
			'name'     => $order_item->get_name(),
			'subtotal' => $subscription_price,
			'total'    => $subscription_price,
		);

		// creating new order.
		$new_order_data = FreeHelper::create_new_order_for_renewal( $old_order, $order_item, $product_args );
		if ( ! $new_order_data ) {
			return;
		}

		$new_order         = $new_order_data['order'];
		$new_order_item_id = $new_order_data['order_item_id'];
		$order_id          = $new_order->get_id();
		self::create_early_renewal_history( $subscription_id, $new_order->get_id(), $new_order_item_id );

		update_post_meta(
			$subscription_id,
			'_order_waiting_early_renew_subscrpt_meta',
			array(
				'order_id'      => $new_order->get_id(),
				'order_item_id' => $new_order_item_id,
			)
		);

		FreeHelper::clone_order_metadata( $new_order, $old_order );
		FreeHelper::clone_stripe_metadata_for_renewal( $subscription_id, $old_order, $new_order );

		$new_order->calculate_totals();
		$new_order->save();

		if ( ! is_admin() ) {
			$message = 'Early Renewal Order(#' . $order_id . ') Created.';
			if ( $new_order->has_status( 'pending' ) ) {
				$message .= ' Please <a href="' . $new_order->get_checkout_payment_url() . '">Pay now</a>';
			}
			wc_add_notice( $message, 'success' );
		}

		do_action( 'subscrpt_after_create_renew_order', $new_order, $old_order, $subscription_id, true );
	}

	/**
	 * Create history for renewal.
	 *
	 * @param int $subscription_id Subscription Id.
	 * @param int $new_order_id New Order Id.
	 * @param int $new_order_item_id New Order Item Id.
	 *
	 * @return void
	 */
	public static function create_early_renewal_history( $subscription_id, $new_order_id, $new_order_item_id ) {
		global $wpdb;
		$history_table = $wpdb->prefix . 'subscrpt_order_relation';
		$wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$history_table,
			array(
				'subscription_id' => $subscription_id,
				'order_id'        => $new_order_id,
				'order_item_id'   => $new_order_item_id,
				'type'            => 'early-renew',
			)
		);

		$comment_id = wp_insert_comment(
			array(
				'comment_author'  => 'Subscription for WooCommerce',
				'comment_content' => sprintf( 'Subscription early renewal order created. order is %s', $new_order_id ),
				'comment_post_ID' => $subscription_id,
				'comment_type'    => 'order_note',
			)
		);
		update_comment_meta( $comment_id, '_subscrpt_activity', 'Early Renewal Order' );
		update_comment_meta( $comment_id, '_subscrpt_activity_type', 'renewal_order_early' );

		// Fire action when early renewal is triggered
		do_action( 'subscrpt_split_payment_early_renew_triggered', $subscription_id, $new_order_id, $new_order_item_id );
	}

	/**
	 * Process switch order.
	 *
	 * @param int   $subscription_id Subscription Id.
	 * @param int   $order_id Order Id.
	 * @param int   $order_item_id Order Item Id.
	 * @param array $switch_context Switch Context.
	 */
	public static function process_switch_order( int $subscription_id, int $order_id, int $order_item_id, array $switch_context ) {
		global $wpdb;
		$history_table = $wpdb->prefix . 'subscrpt_order_relation';

		$switch_type = $switch_context['switch_type'] ?? 'upgrade';

		$switch_comment = 'upgrade' === $switch_type
							? __( 'Subscription upgrade order successfully created.', 'wp_subscription_pro' )
							: __( 'Subscription downgrade order successfully created.', 'wp_subscription_pro' );

		$subscription_switch_comment_meta      = 'upgrade' === $switch_type
											? __( 'Subscription Upgrade Order', 'wp_subscription_pro' )
											: __( 'Subscription Downgrade Order', 'wp_subscription_pro' );
		$subscription_switch_comment_meta_type = 'upgrade' === $switch_type
											? 'upgrade_order' : 'downgrade_order';

		$comment_id = wp_insert_comment(
			array(
				'comment_author'  => 'Subscription for WooCommerce',
				'comment_content' => sprintf(
					// translators: 1: Switch Type, 2: Order ID.
					__( '%1$s Order #%2$s', 'wp_subscription_pro' ),
					$switch_comment,
					$order_id
				),
				'comment_post_ID' => $subscription_id,
				'comment_type'    => 'order_note',
			)
		);

		update_comment_meta( $comment_id, '_subscrpt_activity', $subscription_switch_comment_meta );
		update_comment_meta( $comment_id, '_subscrpt_activity_type', $subscription_switch_comment_meta_type );

		$wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			$history_table,
			array(
				'subscription_id' => $subscription_id,
				'order_id'        => $order_id,
				'order_item_id'   => $order_item_id,
				'type'            => $switch_context['switch_type'] ?? 'upgrade',
			)
		);
	}

	/**
	 * Activity title to activity type mapping.
	 *
	 * ! (⁠╯⁠°⁠□⁠°⁠）⁠╯⁠︵⁠ ⁠┻⁠━⁠┻
	 *
	 * @param string $activity_title Activity Title.
	 * @return null|string Activity Type.
	 */
	protected static function get_activity_type( $activity_title ) {
		$mapping = [
			'New Subscription'                  => 'subs_created',
			'Subscription Activated'            => 'subs_activated',
			'Subscription Pending'              => 'subs_pending',
			'Subscription Cancelled'            => 'subs_cancelled',
			'Subscription Pending Cancellation' => 'subs_pe_cancel',
			'Subscription Expired'              => 'subs_expired',

			'Renewal Order'                     => 'renewal_order',
			'Early Renewal Order'               => 'early_renewal_order',
			'Early Renewal Cancelled'           => 'early_renewal_cancelled',
			'Renewal Count Updated'             => 'renewal_count',

			'Subscription Upgrade Order'        => 'upgrade_order',
			'Subscription Downgrade Order'      => 'downgrade_order',

			'Split Payment - Plan Complete (Lifetime Access)' => 'split_payment',
			'Split Payment - Plan Complete (Full Duration)' => 'split_payment',
			'Split Payment - Plan Complete (Custom Duration)' => 'split_payment',
			'Payment Summary - Complete'        => 'split_payment',
		];

		return $mapping[ $activity_title ] ?? null;
	}

	/**
	 * Get subscription activities.
	 *
	 * All activity types
	 * - subs_created, subs_activated, subs_pending, subs_cancelled, subs_pe_cancel, subs_expired,
	 * - renewal_order, early_renewal_order, early_renewal_cancelled, renewal_count,
	 * - upgrade_order, downgrade_order,
	 * - split_payment
	 *
	 * @param int   $subscription_id Subscription Id.
	 * @param array $include_types Include Types.
	 * @param array $exclude_types Exclude Types.
	 * @return array
	 */
	public static function get_subscription_activities( $subscription_id, $include_types = [], $exclude_types = [] ) {
		// Get all comments for the subscription.
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$activities = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->comments} WHERE comment_post_ID = %d ORDER BY comment_ID DESC",
				$subscription_id
			),
			ARRAY_A
		);

		// Normalize activities from comments.
		$normalized_activities = [];
		foreach ( $activities as $activity ) {
			$subscrpt_activity      = get_comment_meta( $activity['comment_ID'], '_subscrpt_activity', true );
			$subscrpt_activity_type = get_comment_meta( $activity['comment_ID'], '_subscrpt_activity_type', true );

			// Migration.
			if ( empty( $subscrpt_activity_type ) ) {
				$subscrpt_activity_type = self::get_activity_type( $subscrpt_activity );

				if ( ! empty( $subscrpt_activity_type ) ) {
					update_comment_meta( $activity['comment_ID'], '_subscrpt_activity_type', $subscrpt_activity_type );
				} else {
					// Skip if no mapping found.
					continue;
				}
			}

			$normalized_activities[] = [
				'id'          => $activity['comment_ID'],
				'type'        => $subscrpt_activity_type,
				'title'       => $subscrpt_activity,
				'description' => $activity['comment_content'],
				'date'        => $activity['comment_date'],
			];
		}

		// Filter activities based on include and exclude types.
		$filtered_activities = $normalized_activities;
		if ( ! empty( $include_types ) ) {
			foreach ( $filtered_activities as $key => $activity ) {
				if ( ! in_array( $activity['type'], $include_types, true ) ) {
					unset( $filtered_activities[ $key ] );
				}
			}
		}
		if ( ! empty( $exclude_types ) ) {
			foreach ( $filtered_activities as $key => $activity ) {
				if ( in_array( $activity['type'], $exclude_types, true ) ) {
					unset( $filtered_activities[ $key ] );
				}
			}
		}

		return $filtered_activities;
	}
}
