<?php
/**
 * WP Fusion Integration File
 *
 * @package SpringDevs\SubscriptionPro\Integrations\WPFusion
 */

namespace SpringDevs\SubscriptionPro\Integrations\WPFusion;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class WPFusion
 */
class WPFusion extends \WPF_Integrations_Base {
	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @var string $slug
	 */
	public $slug = 'wp-subscription';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @var string $name
	 */
	public $name = 'WP Subscription';

	/**
	 * The link to the documentation on the WP Fusion website.
	 *
	 * @var string $docs_url
	 */
	public $docs_url = 'https://docs.converslabs.com/en/wpsubscription-wpf';

	/**
	 * Gets things started
	 *
	 * @access  public
	 * @return  void
	 */
	public function init() {
		// Product settings.
		add_action( 'wpf_woocommerce_panel', [ $this, 'panel_content' ] );

		// Subscription triggers.
		add_action( 'subscrpt_subscription_activated', [ $this, 'subscription_activated' ] );
		add_action( 'subscrpt_subscription_pending', [ $this, 'subscription_pending' ] );
		add_action( 'subscrpt_subscription_cancelled', [ $this, 'subscription_cancelled' ] );
		add_action( 'subscrpt_subscription_pending_cancellation', [ $this, 'subscription_pending_cancellation' ] );
		add_action( 'subscrpt_subscription_expired', [ $this, 'subscription_expired' ] );
		add_action( 'subscrpt_subscription_trial_ended', [ $this, 'subscription_trial_ended' ] );
		add_action( 'subscrpt_grace_period_started', [ $this, 'subscription_in_grace_period' ] );
	}

	/**
	 * Outputs the content for the WP Fusion product settings panel.
	 *
	 * @param int $product_id The product ID.
	 */
	public function panel_content( $product_id ) {
		// Saved Settings.
		$settings = $this->get_wpf_subscription_settings( $product_id );

		// Subscription Fields.
		$subs_fields = $this->get_wpf_subscription_fields( $product_id );

		// TODO: CRM Fields.
		// $crm_fields = $this->get_wpf_subscription_crm_fields( $product_id );
		$crm_fields = [];

		// Load the view file.
		ob_start();
		require SUBSCRIPT_PRO_INCLUDES . '/Integrations/WPFusion/views/product-settings.php';
		$content = ob_get_clean();

		// ? Ignored intentionally. Escaping breaks styles from WP Fusion function.
		echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Get WPF subscription settings.
	 *
	 * @param int $post_id    The post ID.
	 * @return array
	 */
	public function get_wpf_subscription_settings( $post_id ) {
		$settings = [
			'wpf_subscrpt_remove_tags'            => '0',
			'apply_tags_subscrpt_active'          => [],
			'apply_tags_subscrpt_pending'         => [],
			'apply_tags_subscrpt_cancelled'       => [],
			'apply_tags_subscrpt_pe_cancelled'    => [],
			'apply_tags_subscrpt_expired'         => [],
			'apply_tags_subscrpt_trial_over'      => [],
			'apply_tags_in_subscrpt_grace_period' => [],
		];

		$saved_settings = get_post_meta( $post_id, 'wpf-settings-woo', true );
		if ( ! empty( $saved_settings ) ) {
			$settings = array_merge( $settings, $saved_settings );
		}

		return $settings;
	}

	/**
	 * Gets the WP Fusion subscription fields.
	 *
	 * @param int $post_id   The post ID.
	 * @return array The subscription fields.
	 */
	public function get_wpf_subscription_fields( $post_id ) {
		// Get WPF Settings.
		$settings = $this->get_wpf_subscription_settings( $post_id );

		$meta_name = 'wpf-settings-woo';

		// Subscription Fields
		$subs_fields = [
			[
				'label'       => __( 'Active', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is active.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_active',
				'setting'     => $settings['apply_tags_subscrpt_active'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Pending', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is pending.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_pending',
				'setting'     => $settings['apply_tags_subscrpt_pending'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Cancelled', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is cancelled.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_cancelled',
				'setting'     => $settings['apply_tags_subscrpt_cancelled'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Pending Cancellation', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is pending cancellation.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_pe_cancelled',
				'setting'     => $settings['apply_tags_subscrpt_pe_cancelled'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Expired', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is expired.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_expired',
				'setting'     => $settings['apply_tags_subscrpt_expired'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Trial Over', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription trial is over.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_subscrpt_trial_over',
				'setting'     => $settings['apply_tags_subscrpt_trial_over'] ?? [],
				'meta_name'   => $meta_name,
			],
			[
				'label'       => __( 'Grace Period', 'wp_subscription_pro' ),
				'description' => __( 'Apply these tags when a subscription is in the grace period.', 'wp_subscription_pro' ),
				'field_id'    => 'apply_tags_in_subscrpt_grace_period',
				'setting'     => $settings['apply_tags_in_subscrpt_grace_period'] ?? [],
				'meta_name'   => $meta_name,
			],
		];

		return $subs_fields;
	}

	/**
	 * Gets the WP Fusion subscription CRM fields.
	 *
	 * @param int $post_id The post ID.
	 * @return array The subscription CRM fields.
	 */
	public function get_wpf_subscription_crm_fields( $post_id ) {
		$meta_name = 'wpf_settings_woo_crm_fields';

		$crm_fields = [
			[
				'label'     => __( 'Subscription ID', 'wp_subscription_pro' ),
				'field_id'  => 'wpf_subs_id',
				'setting'   => false,
				'meta_name' => $meta_name,
			],
			[
				'label'     => __( 'Subscription Status', 'wp_subscription_pro' ),
				'field_id'  => 'wpf_subs_status',
				'setting'   => false,
				'meta_name' => $meta_name,
			],
		];

		// Get WPF contact fields.
		$wpf_contact_fields = wp_fusion()->settings->get( 'contact_fields' );

		foreach ( $crm_fields as $index => $crm_field ) {
			$id = $crm_field['field_id'] . '_' . $post_id;

			$crm_fields[ $index ]['field_id'] = $id;
			$crm_fields[ $index ]['setting']  = isset( $wpf_contact_fields[ $id ] ) ? $wpf_contact_fields[ $id ]['crm_field'] : false;
		}

		return $crm_fields;
	}

	/**
	 * Get tags to apply or remove.
	 *
	 * @param int    $subscription_id Subscription ID.
	 * @param string $status          Subscription Status.
	 * @return array
	 */
	public function get_subscription_tags( $subscription_id, $status = 'active' ) {
		$tags = [
			'apply'  => [],
			'remove' => [],
		];

		$subscription = FreeHelper::get_subscription_data( $subscription_id );

		$user_id = $subscription['user_id'] ?? 0;
		if ( empty( $user_id ) ) {
			return $tags;
		}

		$product_id        = $subscription['product']['product_id'] ?? 0;
		$wpf_subs_settings = $this->get_wpf_subscription_settings( $product_id );

		// Get & group user subscriptions.
		$all_subscriptions = FreeHelper::get_subscriptions(
			[
				'user_id' => $user_id,
				'status'  => 'any',
				'return'  => 'subscription_data',
			]
		);

		$groupped_subscriptions = [];
		foreach ( $all_subscriptions as $subscrpt ) {
			$groupped_subscriptions[ strtolower( $subscrpt['status'] ) ][] = $subscrpt;
		}

		// Build keys for tags.
		switch ( strtolower( $status ) ) {
			case 'active':
				$apply_keys = [ 'apply_tags_subscrpt_active' ];
				break;

			case 'pending':
				$apply_keys = [ 'apply_tags_subscrpt_pending' ];
				break;

			case 'cancelled':
				$apply_keys = [ 'apply_tags_subscrpt_cancelled' ];
				break;

			case 'pe_cancelled':
				$apply_keys = [ 'apply_tags_subscrpt_pe_cancelled' ];
				break;

			case 'expired':
				$apply_keys = [ 'apply_tags_subscrpt_expired' ];
				break;

			case 'trial_over':
				$apply_keys = [ 'apply_tags_subscrpt_trial_over' ];
				break;

			case 'in_grace_period':
				$apply_keys = [ 'apply_tags_in_subscrpt_grace_period' ];
				break;

			default:
				$apply_keys  = [];
				$remove_keys = [];
				break;
		}

		// Remove invalid tags.
		$remove_keys = [];
		if ( count( $groupped_subscriptions['active'] ?? [] ) === 0 ) {
			$remove_keys[] = 'apply_tags_subscrpt_active';
		}
		if ( count( $groupped_subscriptions['pending'] ?? [] ) === 0 ) {
			$remove_keys[] = 'apply_tags_subscrpt_pending';
		}
		if ( count( $groupped_subscriptions['cancelled'] ?? [] ) === 0 ) {
			$remove_keys[] = 'apply_tags_subscrpt_cancelled';
		}
		if ( count( $groupped_subscriptions['pe_cancelled'] ?? [] ) === 0 ) {
			$remove_keys[] = 'apply_tags_subscrpt_pe_cancelled';
		}
		if ( count( $groupped_subscriptions['expired'] ?? [] ) === 0 ) {
			$remove_keys[] = 'apply_tags_subscrpt_expired';
		}

		// Empty remove tags if setting is disabled.
		if ( ! (int) $wpf_subs_settings['wpf_subscrpt_remove_tags'] ) {
			$remove_keys = [];
		}

		// Set Tags.
		foreach ( $apply_keys as $key ) {
			if ( ! empty( $wpf_subs_settings[ $key ] ) ) {
				$tags['apply'] = array_merge( $tags['apply'], $wpf_subs_settings[ $key ] );
			}
		}
		foreach ( $remove_keys as $key ) {
			if ( ! empty( $wpf_subs_settings[ $key ] ) ) {
				$tags['remove'] = array_merge( $tags['remove'], $wpf_subs_settings[ $key ] );
			}
		}

		// Ensure tags to be removed are not in the apply list.
		$tags['remove'] = array_diff( $tags['remove'], $tags['apply'] );

		return $tags;
	}

	/**
	 * Handle subscription activated event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_activated( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'active' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription pending event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_pending( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'pending' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription activated event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_cancelled( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'cancelled' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription pending cancellation event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_pending_cancellation( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'pe_cancelled' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription expired event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_expired( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'expired' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription trial ended event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_trial_ended( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'trial_over' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}

	/**
	 * Handle subscription in grace period event.
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function subscription_in_grace_period( $subscription_id ) {
		$subscription = FreeHelper::get_subscription_data( $subscription_id );
		$user_id      = $subscription['user_id'] ?? 0;

		$tags = $this->get_subscription_tags( $subscription_id, 'in_grace_period' );

		// Apply Tags.
		if ( ! empty( $tags['apply'] ) ) {
			wp_fusion()->user->apply_tags( $tags['apply'], $user_id );
		}

		// Remove Tags.
		if ( ! empty( $tags['remove'] ) ) {
			wp_fusion()->user->remove_tags( $tags['remove'], $user_id );
		}
	}
}
