<?php

namespace SpringDevs\SubscriptionPro\Frontend;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;
use SpringDevs\SubscriptionPro\Illuminate\Helper;

/**
 * Class Account
 *
 * @package SpringDevs\SubscriptionPro\Frontend
 */
class Account {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_filter( 'subscrpt_single_action_buttons', array( $this, 'action_buttons' ), 10, 4 );
		add_action( 'subscrpt_execute_actions', array( $this, 'execute_actions' ), 10, 2 );
		add_action( 'subscrpt_after_subscription_totals', array( $this, 'display_delivery_schedules' ) );
	}

	/**
	 * Subscription customer page action buttons related to premium version.
	 *
	 * @param array  $action_buttons Action Buttons.
	 * @param int    $id             subscription id.
	 * @param string $subscrpt_nonce action_nonce.
	 * @param string $status         subscription status.
	 *
	 * @return array $action_buttons
	 */
	public function action_buttons( $action_buttons, $id, $subscrpt_nonce, $status ) {
		$subscription_data = FreeHelper::get_subscription_data( $id );

		$product_deleted = get_post_meta( $id, '_product_deleted', true );
		if ( 'yes' === $product_deleted ) {
			return array();
		}

		// Don't show any renewal-related buttons if payment limit is reached
		if ( function_exists( 'subscrpt_is_max_payments_reached' ) && subscrpt_is_max_payments_reached( $id ) ) {
			return $action_buttons;
		}

		$order_id = get_post_meta( $id, '_subscrpt_order_id', true );
		$order    = wc_get_order( $order_id );

		$is_auto_renew   = $subscription_data['is_auto_renew'];
		$renewal_setting = in_array( get_option( 'wp_subscription_auto_renewal_toggle', '1' ), [ 1, '1', 'true', 'yes' ], true );

		if ( $renewal_setting && class_exists( 'WC_Stripe' ) && $order && $order->get_payment_method() === 'stripe' ) {
			if ( ! $is_auto_renew ) {
				$action_buttons['auto-renew-on'] = array(
					'url'   => subscrpt_get_action_url( 'renew-on', $subscrpt_nonce, $id ),
					'label' => __( 'Turn on Auto Renewal', 'wp_subscription_pro' ),
				);
			} else {
				$action_buttons['auto-renew-off'] = array(
					'url'   => subscrpt_get_action_url( 'renew-off', $subscrpt_nonce, $id ),
					'label' => __( 'Turn off Auto Renewal', 'wp_subscription_pro' ),
				);
			}
		}

		$trial = get_post_meta( get_the_ID(), '_subscrpt_trial', true );
		if ( '1' === get_option( 'subscrpt_early_renew', '' ) && 'active' === $status && empty( $trial ) ) {
			$url   = subscrpt_get_action_url( 'early-renew', $subscrpt_nonce, $id );
			$label = __( 'Early Renew', 'wp_subscription_pro' );

			$early_meta = get_post_meta( $id, '_order_waiting_early_renew_subscrpt_meta', true );
			if ( $early_meta ) {
				$early_renew_order = wc_get_order( $early_meta['order_id'] );
				if ( 'pending' === $early_renew_order->get_status() ) {
					$url   = $early_renew_order->get_checkout_payment_url();
					$label = __( 'Pay for Early Renew', 'wp_subscription_pro' );
				} else {
					return $action_buttons;
				}
			}

			// Allow customization of button text
			$label = apply_filters( 'subscrpt_split_payment_button_text', $label, 'early-renew', $id, $status );

			$action_buttons['early-renew'] = array(
				'url'   => $url,
				'label' => $label,
			);
		}

		return $action_buttons;
	}

	/**
	 * Execute actions thats releated to premium version.
	 *
	 * @param int    $subscription_id Subscription ID.
	 * @param string $action          Action name.
	 */
	public function execute_actions( $subscription_id, $action ) {
		if ( 'early-renew' === $action ) {
			Helper::create_early_renewal_order( $subscription_id );
		}
	}

	public function display_delivery_schedules( $subscription_id ) {
		// Get product to check if it's virtual or downloadable
		$order_id      = get_post_meta( $subscription_id, '_subscrpt_order_id', true );
		$order_item_id = get_post_meta( $subscription_id, '_subscrpt_order_item_id', true );
		$order         = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		$order_item = $order->get_item( $order_item_id );
		if ( ! $order_item ) {
			return;
		}

		$product = $order_item->get_product();
		if ( ! $product ) {
			return;
		}

		// Don't show delivery schedules for virtual or downloadable products
		if ( $product->is_virtual() || $product->is_downloadable() ) {
			return;
		}

		$delivery_schedules = get_posts(
			array(
				'post_type'   => 'subscrpt_delivery',
				'numberposts' => 5,
				'post_status' => array( 'waiting', 'shipped', 'in_process', 'cancelled' ),
				'meta_query'  => array(
					'relation' => 'AND',
					array(
						'key'     => '_subscrpt_subscription_id',
						'value'   => $subscription_id,
						'compare' => '=',
					),
					array(
						'key'     => '_subscrpt_shipping_on',
						'value'   => time(),
						'compare' => '>',
					),
				),
			)
		);
		?>
		<h2><?php echo esc_html_e( 'Delivery Schedules', 'wp_subscription_pro' ); ?></h2>
<table class="woocommerce-table woocommerce-table--order-details shop_table order_details">
	<thead>
		<tr>
			<th class="product-name"><?php echo esc_html_e( 'Shipped on', 'wp_subscription_pro' ); ?></th>
			<th class="product-total"><?php echo esc_html_e( 'Delivery Status', 'wp_subscription_pro' ); ?></th>
			<th class="product-total"><?php echo esc_html_e( 'Delivery On', 'wp_subscription_pro' ); ?></th>
		</tr>
	</thead>
	<tbody>
		<?php
		foreach ( $delivery_schedules as $delivery ) {
			?>
			<tr class="order_item">
				<td class="product-name">
				<?php
				$shipping_date = get_post_meta( $delivery->ID, '_subscrpt_shipping_on', true );
				$shipped_date  = get_post_meta( $delivery->ID, '_subscrpt_shipped_date', true );
				?>
					<?php echo $shipping_date ? esc_html( date_i18n( get_option( 'date_format' ), $shipping_date ) ) : '-'; ?>
				</td>
				<td class="product-total">
					<?php echo esc_html( get_post_status_object( get_post_status( $delivery->ID ) )->label ); ?>
				</td>
				<td class="product-total">
					<?php echo $shipped_date ? esc_html( date_i18n( get_option( 'date_format' ), $shipped_date ) ) : '-'; ?>
				</td>
			</tr>
			<?php
		}
		?>
		<?php if ( empty( $delivery_schedules ) ) : ?>
			<tr class="order_item">
				<td class="product-name" colspan="3"><?php echo esc_html_e( 'No delivery schedules found', 'wp_subscription_pro' ); ?></td>
			</tr>
		<?php endif; ?>
		</tbody>
</table>
		<?php
	}
}
