<?php

namespace SpringDevs\SubscriptionPro\Frontend;

use SpringDevs\SubscriptionPro\Illuminate\Helper;
use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class Checkout
 *
 * @package SpringDevs\SubscriptionPro\Frontend
 */
class Checkout {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'subscrpt_product_checkout', array( $this, 'create_subscription' ), 10, 3 );
	}

	/**
	 * Create subscription.
	 *
	 * @param \WC_Order_Item $order_item Order Item.
	 * @param \WC_Product    $product Product Object.
	 * @param string         $post_status Subscription status.
	 *
	 * @return void
	 */
	public function create_subscription( $order_item, $product, $post_status ) {
		if ( $product->is_type( 'simple' ) ) {
			$this->create_simple_subscription( $order_item, $product, $post_status );
		} elseif ( $product->is_type( 'variable' ) ) {
			$this->create_variable_subscription( $order_item, $product, $post_status );
		}
	}

	/**
	 * Create subscription for simple product.
	 *
	 * @param \WC_Order_Item $order_item Order Item.
	 * @param \WC_Product    $product Product Object.
	 * @param string         $post_status Subscription status.
	 *
	 * @return void
	 */
	public function create_simple_subscription( $order_item, $product, $post_status ) {
		$enabled = $product->get_meta( '_subscrpt_enabled' );
		if ( $enabled ) {
			$is_renew      = isset( $order_item['_renew_subscrpt'] );
			$timing_per    = empty( $product->get_meta( '_subscrpt_timing_per' ) ) ? 1 : $product->get_meta( '_subscrpt_timing_per' );
			$timing_option = $product->get_meta( '_subscrpt_timing_option' );
			$type          = FreeHelper::get_typos( $timing_per, $timing_option );

			$trial     = null;
			$has_trial = FreeHelper::check_trial( $product->get_id() );

			$trial_per    = $product->get_meta( '_subscrpt_trial_timing_per' );
			$trial_option = $product->get_meta( '_subscrpt_trial_timing_option' );

			if ( ! empty( $trial_per ) && $trial_per > 0 && ! $is_renew && $has_trial ) {
				$trial = $trial_per . ' ' . FreeHelper::get_typos( $trial_per, $trial_option );
			}

			wc_update_order_item_meta(
				$order_item->get_id(),
				'_subscrpt_meta',
				array(
					'time'  => $timing_per,
					'type'  => $timing_option,
					'trial' => $trial,
				)
			);

			if ( ! $product->is_virtual() ) {
				$delivery_timing_per    = $product->get_meta( '_subscrpt_delivery_timing_per' );
				$delivery_timing_option = $product->get_meta( '_subscrpt_delivery_timing_option' );
				$delivery_day_sync      = $product->get_meta( '_subscrpt_synchronize_delivery_schedule' );
				$delivery_day           = $product->get_meta( '_subscrpt_synchronize_delivery_day' );
				wc_update_order_item_meta(
					$order_item->get_id(),
					'_subscrpt_delivery',
					array(
						'delivery_time'     => $delivery_timing_per ?? null,
						'delivery_option'   => FreeHelper::get_typos(
							$delivery_timing_per,
							$delivery_timing_option
						),
						'delivery_day_sync' => 'yes' === $delivery_day_sync,
						'delivery_day'      => $delivery_day ?? null,
					)
				);
			}

			// Renew subscription if need!
			$renew_subscription_id    = FreeHelper::subscription_exists( $product->get_id(), 'expired' );
			$selected_subscription_id = null;
			if ( $is_renew && $renew_subscription_id && 'cancelled' !== $post_status ) {
				$selected_subscription_id = $renew_subscription_id;
				FreeHelper::process_order_renewal(
					$selected_subscription_id,
					$order_item->get_order_id(),
					$order_item->get_id()
				);
			} else {
				$selected_subscription_id = FreeHelper::process_new_subscription_order( $order_item, $post_status, $product );
			}

			if ( $selected_subscription_id ) {
				// product related.
				update_post_meta( $selected_subscription_id, '_subscrpt_timing_per', $timing_per );
				update_post_meta( $selected_subscription_id, '_subscrpt_timing_option', $timing_option );
				update_post_meta( $selected_subscription_id, '_subscrpt_price', $product->get_price() );
				update_post_meta( $selected_subscription_id, '_subscrpt_user_cancel', $product->get_meta( '_subscrpt_user_cancel' ) );

				// order related.
				update_post_meta( $selected_subscription_id, '_subscrpt_order_id', $order_item->get_order_id() );
				update_post_meta( $selected_subscription_id, '_subscrpt_order_item_id', $order_item->get_id() );

				// subscription related.
				update_post_meta( $selected_subscription_id, '_subscrpt_trial', $trial );
				if ( $trial ) {
					update_post_meta( $selected_subscription_id, '_subscrpt_trial_mode', 'active' === $post_status ? 'on' : 'off' );
				}
				if ( 'active' === $post_status ) {
					$start_date = time();
					$next_date  = sdevs_wp_strtotime( $timing_per . ' ' . $type, $start_date );
					if ( $trial ) {
						$start_date = sdevs_wp_strtotime( $trial );
						$next_date  = $start_date;
					}
					update_post_meta( $selected_subscription_id, '_subscrpt_start_date', $start_date );
					update_post_meta( $selected_subscription_id, '_subscrpt_next_date', $next_date );
				}

				$fees = $order_item->get_order()->get_fees();
				foreach ( $fees as $fee ) {
					if ( 'Signup Fee' === $fee->get_name() ) {
						update_post_meta( $selected_subscription_id, '_subscrpt_signup_fee', $fee->get_amount() );
					}
				}

				do_action( 'subscrpt_order_checkout', $selected_subscription_id, $order_item );
			}
		}
	}

	/**
	 * Create subscription during checkout for variable product
	 *
	 * @param \WC_Order_Item       $order_item  order-item data.
	 * @param \WC_Product_Variable $product       product object.
	 * @param string               $post_status status.
	 */
	public function create_variable_subscription( $order_item, $product, $post_status ) {
		$product = wc_get_product( $order_item['product_id'] );
		if ( $product->is_type( 'variable' ) ) {
			$variation_id = $order_item->get_variation_id();
			$variation    = sdevs_get_subscription_product( $variation_id );

			if ( $variation->is_enabled() ) {
				$timing_per    = $variation->get_timing_per();
				$timing_option = $variation->get_timing_option();
				$is_renew      = isset( $order_item['_renew_subscrpt'] );
				$type          = FreeHelper::get_typos( $timing_per, $timing_option );

				// Subscription switch.
				$is_switch      = in_array( $order_item->get_meta( '_wp_subs_switch' ), [ true, 1, '1' ], true );
				$switch_context = $order_item->get_meta( '_wp_subs_switch_context' );

				$start_date = time();
				$trial      = null;
				if ( ! $is_renew && ! $is_switch && $variation->has_trial() ) {
					$trial_timing_per    = $variation->get_trial_timing_per();
					$trial_timing_option = $variation->get_trial_timing_option();
					$trial               = $trial_timing_per . ' ' . FreeHelper::get_typos( $trial_timing_per, $trial_timing_option );
					$start_date          = sdevs_wp_strtotime( $trial );
				}
				$next_date = sdevs_wp_strtotime( $timing_per . ' ' . $type, $start_date );

				wc_update_order_item_meta(
					$order_item->get_id(),
					'_subscrpt_meta',
					array(
						'time'       => $timing_per,
						'type'       => $timing_option,
						'trial'      => $trial,
						'start_date' => $start_date,
						'next_date'  => $next_date,
					)
				);

				if ( ! $variation->is_virtual() ) {
					$delivery_timing_per    = get_post_meta( $variation_id, '_subscrpt_delivery_timing_per', true );
					$delivery_timing_option = get_post_meta( $variation_id, '_subscrpt_delivery_timing_option', true );
					$delivery_day_sync      = get_post_meta( $variation_id, '_subscrpt_synchronize_delivery_schedule', true );
					$delivery_day           = get_post_meta( $variation_id, '_subscrpt_synchronize_delivery_day', true );
					wc_update_order_item_meta(
						$order_item->get_id(),
						'_subscrpt_delivery',
						array(
							'delivery_time'     => $delivery_timing_per ?? null,
							'delivery_option'   => FreeHelper::get_typos(
								$delivery_timing_per,
								$delivery_timing_option
							),
							'delivery_day_sync' => 'yes' === $delivery_day_sync,
							'delivery_day'      => $delivery_day ?? null,
						)
					);
				}

				// renew subscription if need.
				$renew_subscription_id    = Helper::subscription_variable_exists( $variation_id, 'expired' );
				$selected_subscription_id = null;

				if ( $is_switch && ! empty( $switch_context['subscription_id'] ?? null ) && 'cancelled' !== $post_status ) {
					// Process subscription switch.
					$selected_subscription_id = $switch_context['subscription_id'];
					Helper::process_switch_order( $selected_subscription_id, $order_item->get_order_id(), $order_item->get_id(), $switch_context );

				} elseif ( $is_renew && $renew_subscription_id && 'cancelled' !== $post_status ) {
					$selected_subscription_id = $renew_subscription_id;
					FreeHelper::process_order_renewal(
						$selected_subscription_id,
						$order_item->get_order_id(),
						$order_item->get_id()
					);

				} else {
					$selected_subscription_id = FreeHelper::process_new_subscription_order( $order_item, $post_status, $product );
				}

				if ( $selected_subscription_id && ! $is_switch ) {
					// ? If it is a switch order, update subscription info after the payment is completed.

					// product related.
					update_post_meta( $selected_subscription_id, '_subscrpt_timing_per', $timing_per );
					update_post_meta( $selected_subscription_id, '_subscrpt_timing_option', $timing_option );
					$variation = wc_get_product( $variation_id );
					update_post_meta( $selected_subscription_id, '_subscrpt_price', $variation->get_price() );
					update_post_meta( $selected_subscription_id, '_subscrpt_variation_id', $variation_id );
					update_post_meta( $selected_subscription_id, '_subscrpt_user_cancel', $variation->get_meta( '_subscrpt_user_cancel' ) );

					// order related.
					update_post_meta( $selected_subscription_id, '_subscrpt_order_id', $order_item->get_order_id() );
					update_post_meta( $selected_subscription_id, '_subscrpt_order_item_id', $order_item->get_id() );

					// subscription related.
					update_post_meta( $selected_subscription_id, '_subscrpt_trial', $trial );
					update_post_meta( $selected_subscription_id, '_subscrpt_start_date', $start_date );
					update_post_meta( $selected_subscription_id, '_subscrpt_next_date', $next_date );

					$fees = $order_item->get_order()->get_fees();
					foreach ( $fees as $fee ) {
						if ( 'Signup Fee' === $fee->get_name() ) {
							update_post_meta( $selected_subscription_id, '_subscrpt_signup_fee', $fee->get_amount() );
						}
					}

					do_action( 'subscrpt_order_checkout', $selected_subscription_id, $order_item );
				}
			}
		}
	}
}
