<?php

namespace SpringDevs\SubscriptionPro\Frontend;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;
use SpringDevs\SubscriptionPro\Frontend\Product\Simple;
use SpringDevs\SubscriptionPro\Illuminate\Helper;
use SpringDevs\Subscription\Utils\Product as UtilsProduct;
use WC_Product;
use WC_Product_Variation;

/**
 * Class Product
 *
 * @package SpringDevs\SubscriptionPro\Frontend
 */
class Product {


	/**
	 * Initialize the class
	 */
	public function __construct() {
		new Simple();
		add_action( 'woocommerce_before_add_to_cart_quantity', array( $this, 'remove_qtybox' ) );
		add_filter( 'woocommerce_get_price_html', array( $this, 'change_product_price_html' ), 10, 2 );
		add_filter( 'woocommerce_available_variation', array( $this, 'change_variation_price_html' ), 10, 3 );
		add_filter( 'woocommerce_cart_item_price', array( $this, 'change_price_cart_html' ), 10, 2 );
		add_filter( 'woocommerce_cart_item_subtotal', array( $this, 'change_price_cart_html' ), 10, 2 );
		add_filter( 'woocommerce_variation_is_purchasable', array( $this, 'variable_check_if_purchasable' ), 10, 2 );
	}

	/**
	 * Filter variation purchasable conditionaly.
	 *
	 * @param boolean              $is_purchasable true/false.
	 * @param WC_Product_Variation $variation Variation object.
	 *
	 * @return boolean
	 */
	public function variable_check_if_purchasable( $is_purchasable, $variation ) {
		$variation = sdevs_get_subscription_product( $variation );
		$limit     = $variation->get_meta( '_subscrpt_limit' );
		if ( $variation->is_enabled() ) {
			if ( 'unlimited' === $limit ) {
				return true;
			}
			if ( 'one' === $limit ) {
				return ! Helper::subscription_variable_exists( $variation->get_id(), array( 'pending', 'active' ) );
			}
			if ( 'only_one' === $limit ) {
				return $variation->has_trial();
			}
		}

		return $is_purchasable;
	}

	/**
	 * Hide quantity box when customer already subscription on this product.
	 */
	public function remove_qtybox() {
		global $product;
		if ( $product->is_type( 'variable' ) ) {
			$variations      = $product->get_available_variations();
			$unexpired_items = array();
			foreach ( $variations as $variation ) {
				$variation = sdevs_get_subscription_product( $variation['variation_id'] );
				$unexpired = Helper::subscription_variable_exists(
					$variation->get_id(),
					array(
						'pending',
						'active',
					)
				);
				if ( ! $variation->is_enabled() ) {
					return false;
				}
				$limit = $variation->get_limit();
				if ( $limit ) {
					if ( 'one' === $limit ) {
						if ( $unexpired ) {
							$unexpired_items[] = $variation->get_id();
						}
					}
					if ( 'only_one' === $limit ) {
						$has_trial = Helper::check_trial( $variation->get_id() );
						if ( ! $has_trial ) {
							$unexpired_items[] = $variation->get_id();
						}
					}
				} elseif ( $unexpired ) {
					$unexpired_items[] = $variation->get_id();
				}
			}
			?>
			<style>
				.quantity.hideQty {
					display: none !important;
				}
			</style>
			<script>
				jQuery(document).ready(function($) {
					let unexpired_items = <?php echo wp_json_encode( $unexpired_items ); ?>;
					$('input.variation_id').change(function() {
						if ('' != $('input.variation_id').val()) {
							let var_id = $('input.variation_id').val();
							for (let index = 0; index < unexpired_items.length; index++) {
								const element = unexpired_items[index];
								if (var_id == element) {
									$(".quantity").addClass('hideQty');
									$(".quantity input.qty").hide();
									$(".single_add_to_cart_button").hide();
									break;
								} else {
									$(".quantity").removeClass('hideQty');
									$(".quantity input.qty").show();
									$(".single_add_to_cart_button").show();
								}
							}
						}
					});
				});
			</script>
			<?php
		}
	}

	/**
	 * Add variation type (daily, weekly, yearly) with price amount
	 *
	 * @param string     $price Price HTML.
	 * @param WC_Product $product Product Object.
	 *
	 * @return string $price
	 */
	public function change_product_price_html( $price, $product ) {
		if ( $product->is_type( 'variable' ) ) {
			$variations    = $product->get_available_variations();
			$min_price     = (float) $product->get_variation_price();
			$max_price     = (float) $product->get_variation_price( 'max' );
			$min_variation = $max_variation = null;

			foreach ( $variations as $variation ) {
				$variation_obj = sdevs_get_subscription_product( $variation['variation_id'] );
				if ( ! $variation_obj->is_enabled() ) {
					continue;
				}

				$timing_per = $variation_obj->get_timing_per();
				$time       = $timing_per === 1 ? null : $timing_per;
				$typo       = subscrpt_get_typos( $timing_per, $variation_obj->get_timing_option() );

				// Get split payment info
				$max_no_payment     = $variation_obj->get_max_no_payment();
				$split_payment_info = '';
				if ( $max_no_payment > 0 ) {
					$split_payment_info = ' x ' . $max_no_payment;
				}

				$variation_data = array(
					'id'                 => $variation_obj->get_id(),
					'time'               => $time,
					'typo'               => $typo,
					'split_payment_info' => $split_payment_info,
				);

				if ( $variation['display_price'] === $min_price ) {
					$min_variation = $variation_data;
				}
				if ( $variation['display_price'] === $max_price ) {
					$max_variation = $variation_data;
				}
			}

			if ( $min_variation || $max_variation ) {
				$price = $this->format_price_range( $min_price, $max_price, $min_variation, $max_variation );
			}
		}

		return $price;
	}

	/**
	 * Add some information(type, signup fee, trial) with price if subscription enabled.
	 *
	 * @param mixed                $data data.
	 * @param WC_Product           $product Product object.
	 * @param WC_Product_Variation $variation Variation object.
	 *
	 * @return array
	 */
	public function change_variation_price_html( $data, $product, $variation ): array {
		$variation = sdevs_get_subscription_product( $variation );
		if ( $variation->is_enabled() ) :
			$timing_per    = $variation->get_timing_per();
			$timing_option = $variation->get_timing_option();
			$time          = $timing_per === 1 ? null : $timing_per;

			// Subscription type string
			$timing_type = FreeHelper::get_typos( $timing_per, $timing_option, true );
			$time_string = ' / ' . ( $time ? $time . '-' : '' ) . ucfirst( $timing_type );

			// Split payment info
			$max_no_payment = $variation->get_max_no_payment();

			$split_payment_info = '';
			if ( $max_no_payment > 0 ) {
				$split_payment_info = ' x ' . $max_no_payment;
			}

			// Trial & Signup fee info
			$trial_signup_string = $this->get_trial_signup_html( $variation );

			// Final price HTML
			$data['price_html'] .= $time_string . $split_payment_info . $trial_signup_string;

			// Limit check messages
			$unexpired = Helper::subscription_variable_exists( $variation->get_id(), array( 'pending', 'active' ) );

			if ( $variation->get_limit() === 'one' && $unexpired ) {
				$data['variation_description'] .= '<strong>' . __( 'You already have a subscription with this product!', 'wp_subscription_pro' ) . '</strong>';
			} elseif ( $variation->get_limit() === 'only_one' && ! $variation->has_trial() ) {
				$data['variation_description'] .= '<strong>' . __( 'You already have a subscription with this product!', 'wp_subscription_pro' ) . '</strong>';
			}
		endif;

		return $data;
	}

	/**
	 * Add some information(type, signup fee, trial) with price in Cart if subscription enabled.
	 *
	 * @param string $price price.
	 * @param array  $cart_item cart item.
	 *
	 * @return string
	 */
	public function change_price_cart_html( $price, $cart_item ) {
		$product = wc_get_product( $cart_item['product_id'] );
		if ( ! $product->is_type( 'variable' ) ) {
			return $price;
		}
		$variation = sdevs_get_subscription_product( $cart_item['variation_id'] );
		if ( $variation->is_enabled() ) :
			$timing_per    = $variation->get_timing_per();
			$timing_option = $variation->get_timing_option();
			$time          = 1 === $timing_per ? null : $timing_per;
			$type          = subscrpt_get_typos( $timing_per, $timing_option );

			// Add split payment information if max_no_payment is set
			$max_no_payment     = $variation->get_max_no_payment();
			$split_payment_info = '';
			if ( $max_no_payment > 0 ) {
				$split_payment_info = ' x ' . $max_no_payment;
			}

			return $price . ' / ' . $time . ' ' . $type . $split_payment_info . $this->get_trial_signup_html( $variation );
		else :
			return $price;
		endif;
	}

	/**
	 * @param UtilsProduct $variation
	 *
	 * @return string
	 */
	private function get_trial_signup_html( UtilsProduct $variation ): string {
		$trial           = null;
		$signup_fee_html = null;

		$has_trial = $variation->has_trial();
		if ( $has_trial ) {
			$trial_timing_per    = $variation->get_trial_timing_per();
			$trial_timing_option = $variation->get_trial_timing_option();

			$trial = '<br/><small> ' . sprintf(
				// translators: %s: trial period.
				'+ %s free trial!',
				$trial_timing_per . ' ' . subscrpt_get_typos( $trial_timing_per, $trial_timing_option )
			) . '</small>';
		}

		$signup_fee = $variation->get_signup_fee();
		if ( $signup_fee > 0 ) {
			$signup_fee_html = '<br/><small> ' . sprintf(
				// translators: %s: signup fee amount.
				'+ Signup fee of %s',
				wc_price( $signup_fee )
			) . '</small>';
		}

		return $signup_fee_html . $trial;
	}

	private function format_price_range( $min_price, $max_price, $min_variation, $max_variation ) {
		$min_price_formatted = wc_price( $min_price );
		$max_price_formatted = wc_price( $max_price );

		if ( $min_variation && $max_variation && $min_variation['id'] === $max_variation['id'] ) {
			$variation = $min_variation;
		} elseif ( $min_variation && ! $max_variation ) {
			$variation           = $min_variation;
			$max_price_formatted = " – $max_price_formatted";
		} elseif ( ! $min_variation && $max_variation ) {
			$variation            = $max_variation;
			$min_price_formatted .= ' – ';
		} else {
			$variation            = $max_variation;
			$min_price_formatted .= ' – ';
		}

		$time_string        = $variation['time'] ? $variation['time'] . '-' : '';
		$split_payment_info = isset( $variation['split_payment_info'] ) ? $variation['split_payment_info'] : '';

		return "$min_price_formatted$max_price_formatted / $time_string{$variation['typo']}$split_payment_info";
	}
}
