<?php

namespace SpringDevs\SubscriptionPro\Frontend\Product;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;
use SpringDevs\Subscription\Utils\Product;

/**
 * Class Simple
 *
 * @package SpringDevs\SubscriptionPro\Frontend\Product
 */
class Simple {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_filter( 'subscrpt_simple_price_html', array( $this, 'change_price_html' ), 10, 4 );
		add_action( 'woocommerce_single_product_summary', array( $this, 'display_delivery_info' ), 11 );
	}

	/**
	 * Add trial, signup fee etc. with product price.
	 *
	 * @param string  $price_html Price HTML.
	 * @param Product $product Product.
	 * @param string  $price Product price-html.
	 * @param string  $trial Product subscription trial HTML.
	 *
	 * @return string
	 */
	public function change_price_html( $price_html, $product, $price, $trial ): string {
		$meta_timing_per = $product->get_timing_per();
		$time            = 1 === $meta_timing_per ? null : ( $meta_timing_per . '-' );
		$type            = FreeHelper::get_typos( $meta_timing_per, $product->get_timing_option(), true );

		$has_trial = $product->has_trial();
		$trial     = '';
		if ( $has_trial ) {
			$trial_timing_per    = $product->get_trial_timing_per();
			$trial_timing_option = $product->get_trial_timing_option();

			$trial = '<br/><small> ' . sprintf(
				/* translators: %s: trial period. */
				__( '+ %s free trial!', 'wp_subscription_pro' ),
				$trial_timing_per . ' ' . FreeHelper::get_typos( $trial_timing_per, $trial_timing_option )
			) . '</small>';
		}

		$meta_signup_fee = $product->get_signup_fee();
		$signup_fee_html = '';
		if ( 0 < $meta_signup_fee ) {
			$signup_fee_html = '<br/><small> ' . sprintf(
				/* translators: 1: Signup fee amount */
				__( '+ Signup fee of %s', 'wp_subscription_pro' ),
				wc_price( $meta_signup_fee )
			) . '</small>';
		}

		// Add split payment information if max_no_payment is set
		$max_no_payment     = $product->get_max_no_payment();
		$split_payment_info = '';
		if ( $max_no_payment > 0 ) {
			$split_payment_info = ' x ' . $max_no_payment;
		}

		return $price . ' / ' . $time . ucfirst( $type ) . $split_payment_info . $signup_fee_html . $trial;
	}

	/**
	 * Display delivery info.
	 *
	 * @return void
	 */
	public function display_delivery_info() {
		global $product;

		// Check if subscription is enabled.
		$is_subscription = '1' === $product->get_meta( '_subscrpt_enabled' );

		if ( ! $product->is_type( 'simple' ) || ! $is_subscription ) {
			return;
		}

		$delivery_timing_per    = $product->get_meta( '_subscrpt_delivery_timing_per' );
		$delivery_timing_option = $product->get_meta( '_subscrpt_delivery_timing_option' );
		$delivery_day_sync      = $product->get_meta( '_subscrpt_synchronize_delivery_schedule' );
		$delivery_day           = $product->get_meta( '_subscrpt_synchronize_delivery_day' );

		$type = FreeHelper::get_typos( $delivery_timing_per, $delivery_timing_option );

		if ( 'yes' === $delivery_day_sync ) {
			$time               = strtotime( "next $delivery_day", strtotime( $delivery_timing_per . $type ) );
			$full_day_name      = gmdate( 'l', $time );
			$next_delivery      = gmdate( 'F j, Y', $time );
			$delivery_schedules = 'Every week on ' . $full_day_name;
		} else {
			$next_delivery      = gmdate( 'F j, Y', strtotime( $delivery_timing_per . $type ) );
			$delivery_schedules = 'Every ' . $delivery_timing_per . ' ' . $type;
		}

		if ( ! empty( $delivery_timing_per ) ) {
			?>
			<br>
			<span>
				<strong><?php esc_html_e( 'Delivery schedules: ', 'wp_subscription_pro' ); ?></strong>
				<?php echo esc_html( $delivery_schedules ); ?>
			</span>
			<br>
			<span>
				<strong><?php esc_html_e( 'Next delivery: ', 'wp_subscription_pro' ); ?></strong>
				<?php echo esc_html( $next_delivery ); ?>
			</span>
			<?php
		}
	}
}
