<?php
/**
 * Mollie Gateway for WPSubscription Pro
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Mollie
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Gateways\Mollie;

use Exception;
use SpringDevs\Subscription\Illuminate\Helper;
use SpringDevs\Subscription\Illuminate\Action;

/**
 * Mollie Gateway for WPSubscription Pro
 * Handles recurring subscription payments via Mollie using mandates
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Mollie
 */
class Mollie {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into WPSubscription renewal events
		add_action( 'subscrpt_after_create_renew_order', array( $this, 'process_renewal_payment' ), 10, 3 );
		add_action( 'subscrpt_before_saving_renewal_order', array( $this, 'copy_mollie_metadata' ), 10, 3 );

		// Hook to save Mollie metadata after initial payment
		add_action( 'woocommerce_order_status_processing', array( $this, 'save_mollie_metadata' ), 10, 1 );
		add_action( 'woocommerce_order_status_completed', array( $this, 'save_mollie_metadata' ), 10, 1 );

		// Hook for failed renewal payments (debugging and tracking)
		add_action( 'woocommerce_order_status_failed', array( $this, 'on_renewal_failed' ), 10, 1 );

		// Hook for subscription cancellation (extended tracking)
		add_action( 'subscrpt_subscription_cancelled', array( $this, 'on_subscription_cancelled' ), 10, 1 );
	}

	/**
	 * Process renewal payment using Mollie mandate
	 *
	 * @param \WC_Order $new_order Renewal order.
	 * @param \WC_Order $old_order Parent order.
	 * @param int       $subscription_id Subscription ID.
	 */
	public function process_renewal_payment( $new_order, $old_order, $subscription_id ) {
		$payment_method = $new_order->get_payment_method();

		// Only process Mollie payment methods
		if ( strpos( $payment_method, 'mollie_wc_gateway_' ) !== 0 ) {
			return;
		}

		$order_id = $new_order->get_id();

		try {
			// Get Mollie customer ID and mandate ID from renewal order metadata
			$customer_id  = $new_order->get_meta( '_mollie_customer_id' );
			$mandate_id   = $new_order->get_meta( '_mollie_mandate_id' );
			$payment_mode = $new_order->get_meta( '_mollie_payment_mode' );

			if ( ! $customer_id ) {
				throw new Exception( 'Missing Mollie customer ID' );
			}

			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Processing renewal for Order #%d (Customer: %s, Mandate: %s, Mode: %s)',
					$order_id,
					$customer_id,
					$mandate_id ? $mandate_id : 'auto-select',
					$payment_mode ? $payment_mode : 'unknown'
				)
			);

			// Get Mollie API client
			$api_key = $this->get_mollie_api_key( $payment_mode );
			if ( ! $api_key ) {
				throw new Exception( 'Mollie API key not configured' );
			}

			// Get API client using Mollie's helper function
			if ( ! function_exists( 'mollieWooCommerceApiHelper' ) ) {
				throw new Exception( 'Mollie plugin functions not available' );
			}

			$api = mollieWooCommerceApiHelper()->getApiClient( $api_key );

			// Validate mandate if provided
			$valid_mandate = $this->validate_mandate( $api, $customer_id, $mandate_id, $payment_method );

			if ( ! $valid_mandate ) {
				throw new Exception(
					sprintf(
						'No valid mandate found for customer %s (Mandate ID: %s)',
						$customer_id,
						$mandate_id ? $mandate_id : 'none'
					)
				);
			}

			// Prepare payment data
			$payment_data = array(
				'amount'       => array(
					'currency' => $new_order->get_currency(),
					'value'    => number_format( $new_order->get_total(), 2, '.', '' ),
				),
				'customerId'   => $customer_id,
				'sequenceType' => 'recurring',
				'description'  => sprintf( 'Renewal for Order #%d', $order_id ),
				'redirectUrl'  => $new_order->get_checkout_order_received_url(),
				'webhookUrl'   => home_url( 'wc-api/' . $payment_method ),
				'metadata'     => array(
					'order_id'        => (string) $order_id,
					'subscription_id' => (string) $subscription_id,
				),
			);

			// Include mandate if available
			if ( $mandate_id ) {
				$payment_data['mandateId'] = $mandate_id;
			}

			// Create payment
			$payment = $api->payments->create( $payment_data );

			// Fire action after payment created
			do_action( 'wpsubscription_mollie_payment_created', $payment, $new_order, $subscription_id );

			// Save payment ID
			$new_order->update_meta_data( '_mollie_payment_id', $payment->id );
			$new_order->set_transaction_id( $payment->id );

			// Update payment method if it changed (e.g., iDEAL -> SEPA Direct Debit)
			// This happens when first payment uses iDEAL but renewals use SEPA
			$payment_method_used = 'mollie_wc_gateway_' . $payment->method;
			if ( $payment_method_used !== $payment_method ) {
				$new_order->set_payment_method( $payment_method_used );
				$new_order->add_order_note(
					sprintf(
						__( 'Payment method updated from %1$s to %2$s for this renewal', 'wp_subscription_pro' ),
						str_replace( 'mollie_wc_gateway_', '', $payment_method ),
						$payment->method
					)
				);
				wp_subscrpt_write_log(
					sprintf(
						'Mollie: Updated payment method for Order #%d from %s to %s',
						$order_id,
						$payment_method,
						$payment_method_used
					)
				);
			}

			// Update mandate if it changed
			if ( property_exists( $payment, 'mandateId' ) && $payment->mandateId && $payment->mandateId !== $mandate_id ) {
				$new_order->update_meta_data( '_mollie_mandate_id', $payment->mandateId );
				$new_order->add_order_note(
					sprintf(
						__( 'Mollie mandate updated to: %s', 'wp_subscription_pro' ),
						$payment->mandateId
					)
				);
				wp_subscrpt_write_log( sprintf( 'Mollie: Updated mandate for Order #%d to %s', $order_id, $payment->mandateId ) );
			}

			$new_order->save();

			// Add order note
			$new_order->add_order_note(
				sprintf(
					__( 'Mollie renewal payment created: %1$s (Mode: %2$s, Amount: %3$s)', 'wp_subscription_pro' ),
					$payment->id,
					$payment->mode,
					wc_price( $new_order->get_total(), array( 'currency' => $new_order->get_currency() ) )
				)
			);

			// Add test mode helper link for easier testing
			if ( isset( $payment->_links->changePaymentState->href ) && $payment->mode === 'test' ) {
				$new_order->add_order_note(
					sprintf(
						'MOLLIE TEST MODE: <a href="%s" target="_blank">Change payment state</a>',
						esc_url( $payment->_links->changePaymentState->href )
					)
				);
			}

			// Update order status based on payment status
			// Recurring payments with valid mandates are often processed immediately
			if ( $payment->status === 'paid' or $payment->status === 'authorized' ) {
				// Payment successful immediately - set to processing
				$new_order->update_status( 'processing', __( 'Mollie payment completed', 'wp_subscription_pro' ) );
			} elseif ( $payment->status === 'pending' ) {
				// Payment pending (e.g., SEPA Direct Debit takes days)
				$new_order->update_status( 'on-hold', __( 'Awaiting Mollie payment confirmation', 'wp_subscription_pro' ) );
			} else {
				// Other statuses( Pending, fraud)
				$new_order->update_status(
					'pending',
					sprintf(
						__( 'Mollie payment %s', 'wp_subscription_pro' ),
						$payment->status
					)
				);
			}

			// For SEPA Direct Debit and similar methods, activate subscription immediately
			// These payment methods take 2-3 days to process, customer shouldn't wait
			$this->maybe_activate_subscription_for_sepa( $subscription_id, $payment_method, $payment->method );

			// Log success
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Renewal payment %s created for Order #%d, Amount: %s %s, Mode: %s',
					$payment->id,
					$order_id,
					$new_order->get_total(),
					$new_order->get_currency(),
					$payment->mode
				)
			);

			// Fire action after renewal payment fully processed
			do_action( 'wpsubscription_mollie_after_renewal_payment', $payment, $new_order, $subscription_id );

		} catch ( Exception $e ) {
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Renewal payment failed for Order #%d - %s',
					$order_id,
					$e->getMessage()
				)
			);

			$new_order->add_order_note(
				sprintf(
					__( 'Mollie renewal payment failed: %s', 'wp_subscription_pro' ),
					$e->getMessage()
				)
			);

			$new_order->update_status(
				'failed',
				sprintf(
					__( 'Mollie renewal payment failed: %s', 'wp_subscription_pro' ),
					$e->getMessage()
				)
			);
		}
	}

	/**
	 * Copy Mollie metadata from old order to renewal order
	 *
	 * @param \WC_Order $new_order Renewal order.
	 * @param \WC_Order $old_order Parent order.
	 * @param int       $subscription_id Subscription ID.
	 */
	public function copy_mollie_metadata( $new_order, $old_order, $subscription_id ) {
		if ( strpos( $old_order->get_payment_method(), 'mollie_wc_gateway_' ) !== 0 ) {
			return;
		}

		$customer_id  = $old_order->get_meta( '_mollie_customer_id' );
		$mandate_id   = $old_order->get_meta( '_mollie_mandate_id' );
		$payment_mode = $old_order->get_meta( '_mollie_payment_mode' );

		if ( $customer_id ) {
			$new_order->update_meta_data( '_mollie_customer_id', $customer_id );
		}

		if ( $mandate_id ) {
			$new_order->update_meta_data( '_mollie_mandate_id', $mandate_id );
		}

		if ( $payment_mode ) {
			$new_order->update_meta_data( '_mollie_payment_mode', $payment_mode );
		}

		wp_subscrpt_write_log(
			sprintf(
				'Mollie: Copied metadata to renewal order #%d (Customer: %s, Mandate: %s, Mode: %s)',
				$new_order->get_id(),
				$customer_id ? $customer_id : 'missing',
				$mandate_id ? $mandate_id : 'missing',
				$payment_mode ? $payment_mode : 'missing'
			)
		);
	}

	/**
	 * Save Mollie customer and mandate ID from initial payment
	 *
	 * @param int $order_id Order ID.
	 */
	public function save_mollie_metadata( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Only process Mollie payments
		$payment_method = $order->get_payment_method();
		if ( strpos( $payment_method, 'mollie_wc_gateway_' ) !== 0 ) {
			return;
		}

		// Check if metadata already saved
		if ( $order->get_meta( '_mollie_customer_id' ) ) {
			return;
		}

		// Get payment ID
		$payment_id = $order->get_meta( '_mollie_payment_id' );
		if ( ! $payment_id ) {
			$payment_id = $order->get_transaction_id();
		}

		if ( ! $payment_id ) {
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Payment ID not found for Order #%d during metadata save. Order status: %s',
					$order_id,
					$order->get_status()
				)
			);
			return;
		}

		try {
			// Get API key and client
			$payment_mode = $order->get_meta( '_mollie_payment_mode' );
			$api_key      = $this->get_mollie_api_key( $payment_mode );

			if ( ! $api_key || ! function_exists( 'mollieWooCommerceApiHelper' ) ) {
				return;
			}

			$api = mollieWooCommerceApiHelper()->getApiClient( $api_key );

			// Fetch payment details from Mollie
			$payment = $api->payments->get( $payment_id );

			if ( $payment ) {
				$customer_saved = false;
				$mandate_saved  = false;

				// Save customer ID
				if ( isset( $payment->customerId ) && $payment->customerId ) {
					$order->update_meta_data( '_mollie_customer_id', $payment->customerId );
					$customer_saved = true;
				}

				// Save mandate ID
				if ( isset( $payment->mandateId ) && $payment->mandateId ) {
					$order->update_meta_data( '_mollie_mandate_id', $payment->mandateId );
					$mandate_saved = true;
				}

				// Save payment mode
				if ( isset( $payment->mode ) ) {
					$order->update_meta_data( '_mollie_payment_mode', $payment->mode );
				}

				$order->save();

				if ( $customer_saved || $mandate_saved ) {
					wp_subscrpt_write_log(
						sprintf(
							'Mollie: Saved metadata for Order #%d (Customer ID: %s, Mandate ID: %s, Mode: %s)',
							$order_id,
							$customer_saved ? 'saved' : 'n/a',
							$mandate_saved ? 'saved' : 'n/a',
							$payment->mode ?? 'unknown'
						)
					);
				}
			}
		} catch ( Exception $e ) {
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Failed to fetch payment details for Order #%d, Payment ID: %s - %s',
					$order_id,
					$payment_id,
					$e->getMessage()
				)
			);
		}
	}

	/**
	 * Validate Mollie mandate
	 *
	 * @param object      $api Mollie API client.
	 * @param string      $customer_id Mollie customer ID.
	 * @param string|null $mandate_id Mandate ID (optional).
	 * @param string      $payment_method Payment method gateway ID.
	 * @return bool True if valid mandate found
	 */
	private function validate_mandate( $api, $customer_id, $mandate_id, $payment_method ) {
		try {
			// Get customer from Mollie
			$customer = $api->customers->get( $customer_id );

			// Extract payment method from gateway ID (e.g., mollie_wc_gateway_creditcard -> creditcard)
			$mollie_method = str_replace( 'mollie_wc_gateway_', '', $payment_method );

			// If specific mandate provided, check if it's valid
			if ( $mandate_id ) {
				try {
					$mandate = $customer->getMandate( $mandate_id );
					if ( $mandate && $mandate->status === 'valid' ) {
						wp_subscrpt_write_log( sprintf( 'Mollie: Mandate %s is valid for customer %s', $mandate_id, $customer_id ) );
						return true;
					}
				} catch ( Exception $e ) {
					wp_subscrpt_write_log( sprintf( 'Mollie: Mandate %s not found or invalid: %s', $mandate_id, $e->getMessage() ) );
				}
			}

			// Check if customer has any valid mandate for this payment method
			if ( method_exists( $customer, 'hasValidMandateForMethod' ) ) {
				$has_valid = $customer->hasValidMandateForMethod( $mollie_method );
				if ( $has_valid ) {
					wp_subscrpt_write_log( sprintf( 'Mollie: Customer %s has valid mandate for method %s', $customer_id, $mollie_method ) );
					return true;
				}
			}

			// Fallback: check all mandates manually
			$mandates = $api->mandates->listFor( $customer );
			foreach ( $mandates as $mandate ) {
				if ( $mandate->status === 'valid' && ( ! $mollie_method || $mandate->method === $mollie_method ) ) {
					wp_subscrpt_write_log( sprintf( 'Mollie: Found valid mandate %s for customer %s', $mandate->id, $customer_id ) );
					return true;
				}
			}

			wp_subscrpt_write_log( sprintf( 'Mollie: No valid mandate found for customer %s and method %s', $customer_id, $mollie_method ) );
			return false;

		} catch ( Exception $e ) {
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Mandate validation failed for customer %s - %s',
					$customer_id,
					$e->getMessage()
				)
			);
			return false;
		}
	}

	/**
	 * Maybe activate subscription for SEPA Direct Debit and similar methods
	 *
	 * SEPA payments take 2-3 days to process. Instead of keeping the subscription
	 * on-hold during this time, we activate immediately to give customers access.
	 *
	 * Uses Action::status() to properly fire subscription_activated hooks and add comments.
	 *
	 * @param int    $subscription_id Subscription ID.
	 * @param string $gateway_method Gateway method ID (e.g., mollie_wc_gateway_directdebit).
	 * @param string $actual_method Actual Mollie method used (e.g., directdebit).
	 */
	private function maybe_activate_subscription_for_sepa( $subscription_id, $gateway_method, $actual_method ) {
		// Methods that need immediate activation (they take days to process)
		$methods_needing_activation = array(
			'directdebit',     // SEPA Direct Debit
			'bancontact',      // Bancontact
			'belfius',         // Belfius
		);

		// Check if actual payment method needs activation
		if ( ! in_array( $actual_method, $methods_needing_activation, true ) ) {
			return;
		}

		try {
			// Get current subscription status
			$current_status = Helper::get_subscription_status( $subscription_id );

			// Only activate if currently on-hold or pending
			if ( $current_status === 'on-hold' || $current_status === 'pending' ) {
				// Use Action::status() to properly fire hooks and add subscription comments
				Action::status( 'active', $subscription_id, true );

				wp_subscrpt_write_log(
					sprintf(
						'Mollie: Auto-activated subscription #%d for %s payment via Action::status() (payment takes days to process)',
						$subscription_id,
						$actual_method
					)
				);
			}
		} catch ( Exception $e ) {
			wp_subscrpt_write_log(
				sprintf(
					'Mollie: Failed to activate subscription #%d for SEPA payment - %s',
					$subscription_id,
					$e->getMessage()
				)
			);
		}
	}

	/**
	 * Handle failed renewal payment for debugging and tracking
	 *
	 * @param int $order_id Order ID.
	 */
	public function on_renewal_failed( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || strpos( $order->get_payment_method(), 'mollie_wc_gateway_' ) !== 0 ) {
			return;
		}

		// Check if this is a renewal order
		if ( ! $order->get_meta( '_is_renewal' ) ) {
			return;
		}

		// Get subscription ID
		$subscriptions   = Helper::get_subscriptions_from_order( $order_id );
		$subscription_id = ! empty( $subscriptions ) ? $subscriptions[0] : null;

		// Log detailed failure information
		wp_subscrpt_write_log(
			sprintf(
				'Mollie: Renewal payment failed for Order #%d (Subscription: #%s, Customer: %s, Mandate: %s, Status: %s)',
				$order_id,
				$subscription_id ? $subscription_id : 'unknown',
				$order->get_meta( '_mollie_customer_id' ) ? $order->get_meta( '_mollie_customer_id' ) : 'missing',
				$order->get_meta( '_mollie_mandate_id' ) ? $order->get_meta( '_mollie_mandate_id' ) : 'missing',
				$order->get_status()
			)
		);

		// Get payment ID for additional context
		$payment_id = $order->get_meta( '_mollie_payment_id' ) ? $order->get_meta( '_mollie_payment_id' ) : $order->get_transaction_id();
		if ( $payment_id ) {
			wp_subscrpt_write_log( sprintf( 'Mollie: Failed payment ID: %s', $payment_id ) );
		}
	}

	/**
	 * Handle subscription cancellation for extended tracking
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function on_subscription_cancelled( $subscription_id ) {
		// Get subscription data to check if it uses Mollie
		$parent_order = Helper::get_parent_order( $subscription_id );

		if ( ! $parent_order ) {
			return;
		}

		// Only track Mollie subscriptions
		if ( strpos( $parent_order->get_payment_method(), 'mollie_wc_gateway_' ) !== 0 ) {
			return;
		}

		// Get Mollie metadata
		$customer_id    = $parent_order->get_meta( '_mollie_customer_id' );
		$mandate_id     = $parent_order->get_meta( '_mollie_mandate_id' );
		$payment_method = str_replace( 'mollie_wc_gateway_', '', $parent_order->get_payment_method() );

		// Log cancellation for tracking
		wp_subscrpt_write_log(
			sprintf(
				'Mollie: Subscription #%d cancelled (Customer: %s, Mandate: %s, Method: %s)',
				$subscription_id,
				$customer_id ? $customer_id : 'unknown',
				$mandate_id ? $mandate_id : 'unknown',
				$payment_method
			)
		);

		// Optional: In the future, you could:
		// - Cancel Mollie subscriptions if using Mollie Subscriptions API
		// - Revoke mandates (not recommended - customer might reactivate)
		// - Send notification to admin about churned subscription

		// For now, we just track it in logs
		do_action( 'wpsubscription_mollie_subscription_cancelled', $subscription_id, $customer_id, $mandate_id );
	}

	/**
	 * Get Mollie API key based on mode
	 *
	 * @param string|null $mode Payment mode (test or live).
	 * @return string|false API key or false if not found
	 */
	private function get_mollie_api_key( $mode = null ) {
		// Check if test mode is enabled
		$test_mode_enabled = get_option( 'mollie-payments-for-woocommerce_test_mode_enabled' ) === 'yes';

		// If mode is specified, use that, otherwise use global setting
		$use_test_mode = $mode === 'test' || ( ! $mode && $test_mode_enabled );

		if ( $use_test_mode ) {
			$api_key = get_option( 'mollie-payments-for-woocommerce_test_api_key' );
		} else {
			$api_key = get_option( 'mollie-payments-for-woocommerce_live_api_key' );
		}

		return $api_key ? $api_key : false;
	}
}
