<?php
/**
 * Payment Gateways File
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Gateways;

use SpringDevs\Subscription\Admin\Integrations;
use SpringDevs\Subscription\Illuminate\Helper;
use SpringDevs\SubscriptionPro\Illuminate\Gateways\Mollie\Mollie;
use SpringDevs\SubscriptionPro\Illuminate\Gateways\Razorpay\Razorpay;
use SpringDevs\SubscriptionPro\Illuminate\Gateways\Xendit\Xendit;

/**
 * Payment Gateways Class
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways
 */
class PaymentGateways {
	/**
	 * Constructor.
	 */
	public function __construct() {
		// Initialize gateways.
		$this->initialize_gateways();

		// Add gateways to the integrations list.
		add_filter( 'wpsubs_integrations', [ $this, 'add_gateways_to_integrations' ] );

		// init hook.
		add_action( 'init', [ $this, 'init' ] );

		// Add settings fields and register settings.
		add_filter( 'subscrpt_settings_fields', [ $this, 'add_settings_fields' ] );
		add_action( 'subscrpt_register_settings', [ $this, 'register_settings_fields' ] );
	}

	/**
	 * Initialize Gateways
	 */
	public function initialize_gateways() {
		// Razorpay
		if ( class_exists( 'WC_Razorpay' ) ) {
			new Razorpay();
		}

		// Mollie
		if ( class_exists( 'Mollie\WooCommerce\Activation\ActivationModule' ) ) {
			new Mollie();
		}

		// Xendit
		if ( class_exists( 'WC_Xendit_CC' ) || class_exists( 'WC_Xendit_CC_Addons' ) ) {
			new Xendit();
		}
	}

	/**
	 * Add Gateways to Integrations
	 *
	 * @param array $integrations Integrations list.
	 */
	public function add_gateways_to_integrations( $integrations ) {
		// Mollie
		$integrations['mollie'] = [
			'title'              => 'Mollie',
			'description'        => 'Pay for subscriptions with Mollie Payments for WooCommerce.',
			'icon_url'           => SUBSCRIPT_PRO_ASSETS . '/images/mollie.png',
			'is_beta'            => true,
			'is_installed'       => class_exists( 'Mollie\WooCommerce\Activation\ActivationModule' ),
			'is_active'          => Integrations::is_gateway_enabled( 'mollie_wc_gateway', true ),
			'supports_recurring' => true,
			'actions'            => [
				[
					'action' => 'install',
					'label'  => 'Install Now',
					'type'   => 'link',
					'url'    => admin_url( 'plugin-install.php?s=Mollie%2520Payments%2520for%2520WooCommerce&tab=search&type=term' ),
				],
				[
					'action' => 'settings',
					'label'  => 'Settings',
					'type'   => 'link',
					'url'    => admin_url( 'admin.php?page=wc-settings&tab=mollie_settings' ),
				],
				[
					'label' => 'More Details',
					'type'  => 'external_link',
					'url'   => 'https://docs.converslabs.com/en/wpsubscription-payment-with-mollie',
				],
			],
		];

		// Razorpay
		$integrations['razorpay'] = [
			'title'              => 'Razorpay',
			'description'        => 'Pay for subscriptions securely with Razorpay for WooCommerce.',
			'icon_url'           => SUBSCRIPT_PRO_ASSETS . '/images/razorpay.png',
			'is_beta'            => true,
			'is_installed'       => class_exists( 'WC_Razorpay' ),
			'is_active'          => Integrations::is_gateway_enabled( 'razorpay' ),
			'supports_recurring' => true,
			'actions'            => [
				[
					'action' => 'install',
					'label'  => 'Install Now',
					'type'   => 'link',
					'url'    => admin_url( 'plugin-install.php?s=Razorpay%2520for%2520WooCommerce&tab=search&type=term' ),
				],
				[
					'action' => 'settings',
					'label'  => 'Settings',
					'type'   => 'link',
					'url'    => admin_url( 'admin.php?page=wc-settings&tab=checkout&section=razorpay' ),
				],
				[
					'label' => 'More Details',
					'type'  => 'external_link',
					'url'   => 'https://docs.converslabs.com/en/wpsubscription-payment-with-razorpay',
				],
			],
		];

		// Xendit
		$integrations['xendit'] = [
			'title'              => 'Xendit',
			'description'        => 'Pay for subscriptions securely with Xendit for WooCommerce.',
			'icon_url'           => SUBSCRIPT_PRO_ASSETS . '/images/xendit.png',
			'is_beta'            => true,
			'is_installed'       => class_exists( 'WC_Xendit_CC' ),
			'is_active'          => Integrations::is_gateway_enabled( 'xendit' ),
			'supports_recurring' => true,
			'actions'            => [
				[
					'action' => 'install',
					'label'  => 'Install Now',
					'type'   => 'link',
					'url'    => admin_url( 'plugin-install.php?s=Xendit%2520Payment&tab=search&type=term' ),
				],
				[
					'action' => 'settings',
					'label'  => 'Settings',
					'type'   => 'link',
					'url'    => admin_url( 'admin.php?page=wc-settings&tab=checkout&section=xendit_gateway' ),
				],
				[
					'label' => 'More Details',
					'type'  => 'external_link',
					'url'   => 'https://docs.converslabs.com/en/wpsubscription-payment-with-xendit',
				],
			],
		];

		return $integrations;
	}

	/**
	 * Initialize the class
	 */
	public function init() {
		// Remove hidden payment gateways from checkout.
		add_filter( 'woocommerce_available_payment_gateways', [ $this, 'filter_hidden_payment_gateways' ], 99 );

		// Remove blocks of the hidden payment gateways.
		add_action(
			'woocommerce_blocks_loaded',
			function () {
				add_action( 'woocommerce_blocks_payment_method_type_registration', [ $this, 'unregister_payment_gateway_block' ], 99 );
			},
			99
		);

		// JS Fallback for hiding payment gateways in blocks checkout.
		add_action( 'wp_footer', [ $this, 'hide_payment_gateways_js' ] );
	}

	/**
	 * Get Hidden Gateways
	 */
	public function get_hidden_gateways() {
		return get_option( 'hidden_payment_gateways', [] );
	}

	/**
	 * Register Settings Fields
	 *
	 * @param array $settings_fields Settings fields.
	 */
	public function add_settings_fields( $settings_fields ) {
		// Build payment gateway options
		$wc_payment_gateways = function_exists( 'WC' ) ? WC()->payment_gateways->get_available_payment_gateways() : [];
		$gateway_options     = [];
		foreach ( $wc_payment_gateways as $gateway_id => $gateway ) {
			$gateway_options[ $gateway_id ] = $gateway->get_title();
		}

		// Add payment gateway settings fields
		$payment_gateway_fields = [
			[
				'type'       => 'heading',
				'group'      => 'payment_gateways',
				'priority'   => 0.1,
				'field_data' => [
					'title' => __( 'Payment Gateway Settings', 'wp_subscription_pro' ),
				],
			],
			[
				'type'       => 'multi_select',
				'group'      => 'payment_gateways',
				'priority'   => 2,
				'field_data' => [
					'id'          => 'hidden_payment_gateways',
					'title'       => __( 'Hide Payment Gateways', 'wp_subscription_pro' ),
					'description' => __( 'Select payment gateways to hide for subscription products. (keep empty for not hiding any)', 'wp_subscription_pro' ),
					'options'     => $gateway_options,
					'selected'    => get_option( 'hidden_payment_gateways', [] ),
				],
			],
		];

		return array_merge( $settings_fields, $payment_gateway_fields );
	}

	/**
	 * Register Settings Fields
	 */
	public function register_settings_fields() {
		register_setting(
			'wp_subscription_settings',
			'hidden_payment_gateways',
			[
				'type'              => 'array',
				'sanitize_callback' => [ $this, 'sanitize_multi_select_field' ],
			]
		);
	}

	/**
	 * Sanitize Multi Select Field
	 *
	 * @param mixed $value Field value.
	 */
	public function sanitize_multi_select_field( $value ) {
		if ( ! is_array( $value ) ) {
			return [];
		}
		return array_map( 'sanitize_text_field', $value );
	}

	/**
	 * Check cart for subscription products.
	 */
	public function cart_has_subscription_products() {
		$cart       = WC()->cart;
		$cart_items = $cart ? $cart->get_cart() : [];
		$recurrs    = Helper::get_recurrs_from_cart( $cart_items );
		return ! empty( $recurrs );
	}

	/**
	 * Filter Hidden Payment Gateways
	 *
	 * @param array $available_gateways Available gateways.
	 */
	public function filter_hidden_payment_gateways( $available_gateways ) {
		if ( ! is_checkout() && ! is_cart() ) {
			return $available_gateways;
		}

		// Only hide gateways if cart has subscription products
		$cart_has_subs = $this->cart_has_subscription_products();
		if ( ! $cart_has_subs ) {
			return $available_gateways;
		}

		// Get hidden payment gateways from settings
		$hidden_gateways = $this->get_hidden_gateways();

		// Remove hidden gateways from available gateways
		foreach ( $hidden_gateways as $gateway_id ) {
			if ( isset( $available_gateways[ $gateway_id ] ) ) {
				unset( $available_gateways[ $gateway_id ] );
			}
		}

		return $available_gateways;
	}

	/**
	 * Unregister Payment Gateway Block
	 *
	 * @param object $payment_method_registry Payment method registry.
	 */
	public function unregister_payment_gateway_block( $payment_method_registry ) {
		if ( ! is_checkout() && ! is_cart() ) {
			return;
		}

		$cart_has_subs = $this->cart_has_subscription_products();
		if ( ! $cart_has_subs ) {
			return;
		}

		// Get hidden gateways
		$hidden_gateways = $this->get_hidden_gateways();

		$all_registered = $payment_method_registry->get_all_registered();

		// Unregister each hidden gateway
		foreach ( $hidden_gateways as $gateway_id ) {
			if ( array_key_exists( $gateway_id, $all_registered ) ) {
				$payment_method_registry->unregister( $gateway_id );
			}
		}
	}

	/**
	 * Hide Payment Gateways JS
	 */
	public function hide_payment_gateways_js() {
		if ( ! is_checkout() && ! is_cart() ) {
			return;
		}

		$hidden_gateways = $this->get_hidden_gateways();

		if ( empty( $hidden_gateways ) ) {
			return;
		}

		$gateway_ids = array_map( 'esc_js', $hidden_gateways );
		?>
		<script type="text/javascript">
		(function() {
			'use strict';
			
			const hiddenGateways = <?php echo wp_json_encode( $gateway_ids ); ?>;
			
			function hidePaymentGateways() {
				hiddenGateways.forEach(function(gatewayId) {
					// Target the label by its 'for' attribute
					const selector = 'label[for="radio-control-wc-payment-method-options-' + gatewayId + '"]';
					const labels = document.querySelectorAll(selector);
					
					labels.forEach(function(label) {
						// Find the parent accordion option div
						const parentDiv = label.closest('.wc-block-components-radio-control-accordion-option');
						if (parentDiv) {
							parentDiv.style.display = 'none';
						}
					});
					
					// Also hide saved payment method tokens for this gateway
					const savedTokenSelector = 'label[for^="radio-control-wc-payment-method-saved-tokens-"][for*="' + gatewayId + '"]';
					const savedTokens = document.querySelectorAll(savedTokenSelector);
					
					savedTokens.forEach(function(token) {
						const parentDiv = token.closest('.wc-block-components-radio-control__option');
						if (parentDiv) {
							parentDiv.style.display = 'none';
						}
					});
				});
			}

			function hideExpressCheckout() {
				// Target the express payment content container
				const expressContainer = document.querySelector('.wc-block-components-express-payment__content');
				
				if (!expressContainer) {
					return;
				}
				
				hiddenGateways.forEach(function(gatewayId) {
					// Find all elements within the express payment container that have the gateway ID in their class or id
					// Example: id="express-payment-method-paymentplugins_ppcp_express" contains "ppcp"
					const selector = '[id*="' + gatewayId + '"], [class*="' + gatewayId + '"]';
					const gatewayElements = expressContainer.querySelectorAll(selector);
					
					gatewayElements.forEach(function(el) {
						if(el.style.display !== 'none'){
							el.style.display = 'none';
						}
					});
				});
			}
			
			// Continuously check for changes in the number of gateways when
			let lastGatewayCount = 0;
			function checkAndHide() {
				const currentGatewayCount = document.querySelectorAll('.wc-block-components-radio-control-accordion-option').length;
				
				// Only run hiding logic if gateway count changed
				if (currentGatewayCount !== lastGatewayCount) {
					console.log('Gateway count changed:', lastGatewayCount, '->', currentGatewayCount);

					hidePaymentGateways();
					lastGatewayCount = currentGatewayCount;
				}

				// Always attempt to hide express checkout methods
				// hideExpressCheckout();
				
				// Continue checking but only when browser is ready to paint
				requestAnimationFrame(checkAndHide);
			}

			// Start the efficient polling
			requestAnimationFrame(checkAndHide);

			// Hide express checkout on initial load
			window.addEventListener('load', function() {
				hideExpressCheckout();
			});
			
			// Also listen for WooCommerce checkout update events
			if (typeof jQuery !== 'undefined') {
				jQuery(document.body).on('updated_checkout checkout_updated', function() {
					hidePaymentGateways();
					hideExpressCheckout();
				});
			}
		})();
		</script>
		<?php
	}
}
