<?php
/**
 * Razorpay Gateway for WPSubscription Pro
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Razorpay
 */

namespace SpringDevs\SubscriptionPro\Illuminate\Gateways\Razorpay;

use WC_Razorpay;
use WC_Order;
use Exception;
use Razorpay\Api\Api;
use SpringDevs\Subscription\Illuminate\Helper;
use SpringDevs\Subscription\Illuminate\Action;

/**
 * Class Razorpay
 *
 * @package SpringDevs\SubscriptionPro\Illuminate\Gateways\Razorpay
 */
class Razorpay extends WC_Razorpay {

	/**
	 * Constructor
	 */
	public function __construct() {
		// We don't call parent::__construct() because we don't want to re-initialize the gateway settings/hooks
		// that the main plugin already does. We just want to use its methods and add our own hooks.
		// However, we might need some properties setup.
		// Let's check if we need to call parent constructor.
		// WC_Razorpay constructor calls init_form_fields and init_settings.
		// If we don't call it, $this->id, $this->api, etc might not be set.
		// But if we call it, we might duplicate hooks.
		// For now, let's manually setup what we need or call parent with $hooks=false if supported.
		// Looking at WC_Razorpay: public function __construct($hooks = true)

		parent::__construct( false ); // Initialize without hooks

		add_action( 'subscrpt_after_create_renew_order', array( $this, 'after_create_renew_order' ), 10, 3 );
		add_action( 'subscrpt_before_saving_renewal_order', array( $this, 'copy_razorpay_meta' ), 10, 3 );

		// Hook to save token/customer info after initial payment if needed
		add_action( 'woocommerce_order_status_processing', array( $this, 'save_subscription_meta' ), 10, 1 );
		add_action( 'woocommerce_order_status_completed', array( $this, 'save_subscription_meta' ), 10, 1 );

		// Hook for failed renewal payments (debugging and tracking)
		add_action( 'woocommerce_order_status_failed', array( $this, 'on_renewal_failed' ), 10, 1 );

		// Hook for subscription cancellation (extended tracking)
		add_action( 'subscrpt_subscription_cancelled', array( $this, 'on_subscription_cancelled' ), 10, 1 );
	}

	/**
	 * Handle the renewal order payment
	 *
	 * @param WC_Order $new_order new order.
	 * @param WC_Order $old_order old order.
	 * @param int      $subscription_id subscription ID.
	 */
	public function after_create_renew_order( $new_order, $old_order, $subscription_id ) {
		$payment_method = $new_order->get_payment_method();

		if ( 'razorpay' === $payment_method ) {
			$this->pay_renew_order( $new_order );
		}
	}

	/**
	 * Process the renewal payment
	 *
	 * @param WC_Order $renewal_order Renewal order.
	 *
	 * @throws Exception Exception on payment failure.
	 */
	public function pay_renew_order( $renewal_order ) {
		$order_id = $renewal_order->get_id();

		try {
			$customer_id = $renewal_order->get_meta( '_razorpay_customer_id' );
			$token_id    = $renewal_order->get_meta( '_razorpay_token_id' );

			if ( ! $customer_id || ! $token_id ) {
				wp_subscrpt_write_log( sprintf( 'Razorpay: Renewal failed for Order #%d - Missing Customer ID or Token ID (Customer: %s, Token: %s)', $order_id, $customer_id ? $customer_id : 'missing', $token_id ? $token_id : 'missing' ) );
				$renewal_order->add_order_note( __( 'Razorpay Renewal Failed: Missing Customer ID or Token ID.', 'wp_subscription_pro' ) );
				$renewal_order->update_status( 'failed' );
				return;
			}

			$amount   = $renewal_order->get_total() * 100; // Amount in paise
			$currency = $renewal_order->get_currency();

			$api = $this->getRazorpayApiInstance();

			$payment_data = array(
				'amount'      => $amount,
				'currency'    => $currency,
				'customer_id' => $customer_id,
				'token'       => $token_id,
				'recurring'   => '1',
				'description' => 'Renewal Order #' . $order_id,
				'notes'       => array(
					'woocommerce_order_id' => $order_id,
					'subscription_renewal' => 'true',
				),
			);

			// Check if payment action is set to authorize only
			if ( 'authorize' === $this->getSetting( 'payment_action' ) ) {
				$payment_data['payment_capture'] = 0;
			} else {
				$payment_data['payment_capture'] = 1;
			}

			// Create a recurring payment
			$payment = $api->payment->create( $payment_data );

			if ( isset( $payment['id'] ) ) {
				wp_subscrpt_write_log( sprintf( 'Razorpay: Renewal payment created for Order #%d, Payment ID: %s, Amount: %s %s, Status: %s', $order_id, $payment['id'], $renewal_order->get_total(), $currency, isset( $payment['status'] ) ? $payment['status'] : 'unknown' ) );

				// Set transaction ID
				$renewal_order->set_transaction_id( $payment['id'] );

				// Add order note
				// translators: 1: Payment ID, 2: Amount.
				$renewal_order->add_order_note( sprintf( __( 'Razorpay Renewal Payment Created. Payment ID: %1$s, Amount: %2$s', 'wp_subscription_pro' ), $payment['id'], wc_price( $renewal_order->get_total(), array( 'currency' => $currency ) ) ) );

				// Update order status based on payment status
				if ( isset( $payment['status'] ) && ( $payment['status'] === 'captured' || $payment['status'] === 'authorized' ) ) {
					// Payment successful - mark as complete
					$renewal_order->payment_complete( $payment['id'] );
				} elseif ( isset( $payment['status'] ) && $payment['status'] === 'created' ) {
					// Payment created but not captured yet
					$renewal_order->update_status( 'on-hold', __( 'Awaiting Razorpay payment confirmation', 'wp_subscription_pro' ) );
					$renewal_order->save();
				} else {
					// Other statuses - set to pending
					// translators: 1: Payment status.
					$renewal_order->update_status( 'pending', sprintf( __( 'Razorpay payment %s', 'wp_subscription_pro' ), isset( $payment['status'] ) ? $payment['status'] : 'created' ) );
					$renewal_order->save();
				}
			} else {
				wp_subscrpt_write_log( sprintf( 'Razorpay: Payment creation failed for Order #%d - No payment ID returned', $order_id ) );
				throw new Exception( 'Payment creation failed - No payment ID returned.' );
			}
		} catch ( Exception $e ) {
			wp_subscrpt_write_log( sprintf( 'Razorpay: Renewal payment failed for Order #%d - %s', $order_id, $e->getMessage() ) );

			// translators: 1: Error message.
			$renewal_order->add_order_note( sprintf( __( 'Razorpay Renewal Failed: %s', 'wp_subscription_pro' ), $e->getMessage() ) );
			$renewal_order->update_status( 'failed' );
		}
	}

	/**
	 * Copy Razorpay metadata to renewal order
	 *
	 * @param WC_Order $new_order New order.
	 * @param WC_Order $old_order Old order.
	 * @param int      $subscription_id Subscription ID.
	 */
	public function copy_razorpay_meta( $new_order, $old_order, $subscription_id ) {
		if ( 'razorpay' === $old_order->get_payment_method() ) {
			$customer_id = $old_order->get_meta( '_razorpay_customer_id' );
			$token_id    = $old_order->get_meta( '_razorpay_token_id' );

			if ( $customer_id ) {
				$new_order->update_meta_data( '_razorpay_customer_id', $customer_id );
			}
			if ( $token_id ) {
				$new_order->update_meta_data( '_razorpay_token_id', $token_id );
			}
		}
	}

	/**
	 * Save Razorpay Customer and Token ID from the initial order
	 * This attempts to fetch data from Razorpay API if not already saved
	 *
	 * @param int $order_id Order ID.
	 */
	public function save_subscription_meta( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order || 'razorpay' !== $order->get_payment_method() ) {
			return;
		}

		// Check if we already have the data
		if ( $order->get_meta( '_razorpay_token_id' ) ) {
			return;
		}

		// Get Razorpay Payment ID
		// woo-razorpay usually saves it in transaction_id or meta
		$payment_id = $order->get_transaction_id();
		if ( ! $payment_id ) {
			wp_subscrpt_write_log( sprintf( 'Razorpay: Payment ID not found for Order #%d during subscription meta save. Order status: %s', $order_id, $order->get_status() ) );
			// Try to find it in meta if not set as transaction id yet
			// Based on woo-razorpay code, it might be in _razorpay_payment_id (guessing) or just not saved yet if we are too early
			// But this hook is on processing/completed, so it should be there.
			return;
		}

		try {
			$api     = $this->getRazorpayApiInstance();
			$payment = $api->payment->fetch( $payment_id );

			if ( $payment ) {
				$customer_saved = false;
				$token_saved    = false;

				if ( isset( $payment['customer_id'] ) && $payment['customer_id'] ) {
					$order->update_meta_data( '_razorpay_customer_id', $payment['customer_id'] );
					$customer_saved = true;
				}

				if ( isset( $payment['token_id'] ) && $payment['token_id'] ) {
					$order->update_meta_data( '_razorpay_token_id', $payment['token_id'] );
					$token_saved = true;
				}

				$order->save();

				if ( $customer_saved || $token_saved ) {
					wp_subscrpt_write_log( sprintf( 'Razorpay: Saved subscription meta for Order #%d (Customer ID: %s, Token ID: %s)', $order_id, $customer_saved ? 'yes' : 'no', $token_saved ? 'yes' : 'no' ) );
				}
			}
		} catch ( Exception $e ) {
			wp_subscrpt_write_log( sprintf( 'Razorpay: Failed to fetch payment details for Order #%d, Payment ID: %s - %s', $order_id, $payment_id, $e->getMessage() ) );
		}
	}

	/**
	 * Handle failed renewal payment for debugging and tracking
	 *
	 * @param int $order_id Order ID.
	 */
	public function on_renewal_failed( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || $order->get_payment_method() !== 'razorpay' ) {
			return;
		}

		// Check if this is a renewal order
		if ( ! $order->get_meta( '_is_renewal' ) ) {
			return;
		}

		// Get subscription ID
		$subscriptions   = Helper::get_subscriptions_from_order( $order_id );
		$subscription_id = ! empty( $subscriptions ) ? $subscriptions[0] : null;

		// Log detailed failure information
		wp_subscrpt_write_log(
			sprintf(
				'Razorpay: Renewal payment failed for Order #%d (Subscription: #%s, Customer: %s, Token: %s, Status: %s)',
				$order_id,
				$subscription_id ? $subscription_id : 'unknown',
				$order->get_meta( '_razorpay_customer_id' ) ? $order->get_meta( '_razorpay_customer_id' ) : 'missing',
				$order->get_meta( '_razorpay_token_id' ) ? $order->get_meta( '_razorpay_token_id' ) : 'missing',
				$order->get_status()
			)
		);

		// Get payment ID for additional context
		$payment_id = $order->get_transaction_id();
		if ( $payment_id ) {
			wp_subscrpt_write_log( sprintf( 'Razorpay: Failed payment ID: %s', $payment_id ) );
		}
	}

	/**
	 * Handle subscription cancellation for extended tracking
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function on_subscription_cancelled( $subscription_id ) {
		// Get subscription data to check if it uses Razorpay
		$parent_order = Helper::get_parent_order( $subscription_id );

		if ( ! $parent_order || $parent_order->get_payment_method() !== 'razorpay' ) {
			return;
		}

		// Get Razorpay metadata
		$customer_id = $parent_order->get_meta( '_razorpay_customer_id' );
		$token_id    = $parent_order->get_meta( '_razorpay_token_id' );

		// Log cancellation for tracking
		wp_subscrpt_write_log(
			sprintf(
				'Razorpay: Subscription #%d cancelled (Customer: %s, Token: %s)',
				$subscription_id,
				$customer_id ? $customer_id : 'unknown',
				$token_id ? $token_id : 'unknown'
			)
		);

		// Fire custom hook for extensibility
		do_action( 'wpsubscription_razorpay_subscription_cancelled', $subscription_id, $customer_id, $token_id );
	}
}
