<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class LimitChecker
 * Handles subscription purchase and renewal limit checking
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class LimitChecker {

	/**
	 * Get the maximum number of payments for a product with migration support.
	 *
	 * @param int $product_id Product ID.
	 * @return int Maximum number of payments, 0 for unlimited.
	 */
	private static function get_max_no_payment( $product_id ) {
		// Try new field name first
		$max_payments = get_post_meta( $product_id, '_subscrpt_max_no_payment', true ) ?: 0;

		return (int) $max_payments;
	}

	/**
	 * Check if a user can purchase a subscription product
	 *
	 * @param int $product_id Product ID.
	 * @param int $user_id User ID (optional, defaults to current user).
	 * @return bool|string True if allowed, error message if not allowed.
	 */
	public static function check_purchase_limit( $product_id, $user_id = null ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( ! $user_id ) {
			return __( 'Please log in to purchase this subscription.', 'wp_subscription_pro' );
		}

		$limit_type = get_post_meta( $product_id, '_subscrpt_purchase_limit_type', true );

		// Default to unlimited if no limit is set
		if ( empty( $limit_type ) ) {
			$limit_type = 'unlimited';
		}

		switch ( $limit_type ) {
			case 'unlimited':
				return true;

			case 'one_active':
				if ( self::has_active_subscription( $product_id, $user_id ) ) {
					return __( 'You already have an active subscription for this product.', 'wp_subscription_pro' );
				}
				return true;

			case 'one_total':
				if ( self::has_any_subscription( $product_id, $user_id ) ) {
					return __( 'You can only have one subscription for this product.', 'wp_subscription_pro' );
				}
				return true;

			case 'numeric':
				$limit_count   = (int) get_post_meta( $product_id, '_subscrpt_purchase_limit_count', true );
				$current_count = self::get_purchase_count( $product_id, $user_id );

				if ( $current_count >= $limit_count ) {
					return sprintf(
						// translators: %d: maximum number of purchases allowed
						__( 'You have reached the purchase limit of %d for this product.', 'wp_subscription_pro' ),
						$limit_count
					);
				}
				return true;

			default:
				return true;
		}
	}

	/**
	 * Check if a subscription can be renewed
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return bool|string True if allowed, error message if not allowed.
	 */
	public static function check_renewal_limit( $subscription_id ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );

		// Admin override: bypass all limits
		if ( get_post_meta( $product_id, '_subscrpt_admin_override', true ) === 'yes' ) {
			self::log_limit_action( 'override', 'renewal', $product_id, get_current_user_id() );
			return true;
		}

		$limit_count = self::get_max_no_payment( $product_id );

		// 0 means unlimited
		if ( $limit_count == 0 ) {
			return true;
		}

		$current_count = (int) get_post_meta( $subscription_id, '_subscrpt_renewal_count', true );

		if ( $current_count >= $limit_count ) {
			return sprintf(
				// translators: %d: maximum number of renewal cycles
				__( 'This subscription has reached its renewal limit of %d cycles.', 'wp_subscription_pro' ),
				$limit_count
			);
		}

		return true;
	}

	/**
	 * Check if user has an active subscription for a product
	 *
	 * @param int $product_id Product ID.
	 * @param int $user_id User ID.
	 * @return bool
	 */
	public static function has_active_subscription( $product_id, $user_id ) {
		$args = array(
			'post_type'      => 'subscrpt_order',
			'post_status'    => 'active',
			'author'         => $user_id,
			'meta_query'     => array(
				array(
					'key'   => '_subscrpt_product_id',
					'value' => $product_id,
				),
			),
			'posts_per_page' => 1,
		);

		$subscriptions = get_posts( $args );
		return ! empty( $subscriptions );
	}

	/**
	 * Check if user has any subscription for a product (any status)
	 *
	 * @param int $product_id Product ID.
	 * @param int $user_id User ID.
	 * @return bool
	 */
	public static function has_any_subscription( $product_id, $user_id ) {
		$args = array(
			'post_type'      => 'subscrpt_order',
			'post_status'    => array( 'active', 'pending', 'on-hold', 'cancelled', 'expired' ),
			'author'         => $user_id,
			'meta_query'     => array(
				array(
					'key'   => '_subscrpt_product_id',
					'value' => $product_id,
				),
			),
			'posts_per_page' => 1,
		);

		$subscriptions = get_posts( $args );
		return ! empty( $subscriptions );
	}

	/**
	 * Get the number of times a user has purchased a product
	 *
	 * @param int $product_id Product ID.
	 * @param int $user_id User ID.
	 * @return int
	 */
	public static function get_purchase_count( $product_id, $user_id ) {
		$args = array(
			'post_type'   => 'subscrpt_order',
			'post_status' => array( 'active', 'pending', 'on-hold', 'cancelled', 'expired' ),
			'author'      => $user_id,
			'meta_query'  => array(
				array(
					'key'   => '_subscrpt_product_id',
					'value' => $product_id,
				),
			),
			'fields'      => 'ids',
		);

		$subscriptions = get_posts( $args );
		return count( $subscriptions );
	}

	/**
	 * Get the current renewal count for a subscription
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int
	 */
	public static function get_renewal_count( $subscription_id ) {
		return (int) get_post_meta( $subscription_id, '_subscrpt_renewal_count', true );
	}

	/**
	 * Increment the renewal count for a subscription
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int New count.
	 */
	public static function increment_renewal_count( $subscription_id ) {
		$current_count = self::get_renewal_count( $subscription_id );
		$new_count     = $current_count + 1;
		update_post_meta( $subscription_id, '_subscrpt_renewal_count', $new_count );

		// Log the renewal
		self::log_renewal( $subscription_id, $new_count );

		return $new_count;
	}

	/**
	 * Get renewal limit for a product
	 *
	 * @param int $product_id Product ID.
	 * @return int Renewal limit, 0 for unlimited.
	 */
	public static function get_renewal_limit( $product_id ) {
		return self::get_max_no_payment( $product_id );
	}

	/**
	 * Check if subscription has reached its renewal limit
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return bool True if limit reached.
	 */
	public static function is_renewal_limit_reached( $subscription_id ) {
		$product_id  = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		$limit_count = self::get_max_no_payment( $product_id );

		// 0 means unlimited
		if ( $limit_count == 0 ) {
			return false;
		}

		$current_count = (int) get_post_meta( $subscription_id, '_subscrpt_renewal_count', true );
		return $current_count >= $limit_count;
	}

	/**
	 * Get remaining renewals for a subscription
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return int|string Remaining renewals or 'unlimited'.
	 */
	public static function get_remaining_renewals( $subscription_id ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );

		// Get payment type using consistent helper function
		$payment_type = function_exists( 'subscrpt_get_payment_type' ) ? subscrpt_get_payment_type( $subscription_id ) : 'recurring';

		// For split payments, use different logic than recurring subscriptions
		if ( $payment_type === 'split_payment' ) {
			// Get the basic data we need
			$max_payments = subscrpt_get_max_payments( $subscription_id );
			if ( ! $max_payments || intval( $max_payments ) <= 0 ) {
				return 'unlimited';
			}

			$payments_made = subscrpt_count_payments_made( $subscription_id );
			$max_payments  = intval( $max_payments );
			$payments_made = intval( $payments_made );

			// Debug logging (can be removed in production)
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log(
					sprintf(
						'[LimitChecker] Subscription #%d: max_payments=%d, payments_made=%d, remaining=%d',
						$subscription_id,
						$max_payments,
						$payments_made,
						( $max_payments - $payments_made )
					)
				);
			}

			// Simple completion check: all payments made
			if ( $payments_made >= $max_payments ) {
				return 0; // Completed
			}

			// Calculate remaining payments
			$remaining = $max_payments - $payments_made;

			// For split payments, remaining payments ARE the renewals
			return max( 0, $remaining );
		}

		// For recurring subscriptions, use the original logic
		$renewal_limit = self::get_max_no_payment( $product_id );

		// 0 means unlimited
		if ( $renewal_limit == 0 ) {
			return 'unlimited';
		}

		$current_renewals = (int) get_post_meta( $subscription_id, '_subscrpt_renewal_count', true );
		return max( 0, $renewal_limit - $current_renewals );
	}

	public static function get_subscrpt_renewals( $subscription_id ) {
		$product_id = get_post_meta( $subscription_id, '_subscrpt_product_id', true );
		$limit      = get_post_meta( $product_id, '_subscrpt_renewal_limit', true );

		if ( $limit == '0' ) {
			return 'unlimited';
		}

		return $limit;
	}


	/**
	 * Check if subscription is approaching its renewal limit
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $warning_threshold Number of renewals before limit to start warning.
	 * @return bool
	 */
	public static function is_approaching_limit( $subscription_id, $warning_threshold = 2 ) {
		$remaining = self::get_remaining_renewals( $subscription_id );

		if ( $remaining === 'unlimited' ) {
			return false;
		}

		return $remaining <= $warning_threshold;
	}

	/**
	 * Log limit-related actions for audit purposes
	 *
	 * @param string $action Action taken ('check', 'block', 'override').
	 * @param string $type Type of limit ('purchase', 'renewal').
	 * @param int    $product_id Product ID.
	 * @param int    $user_id User ID.
	 */
	private static function log_limit_action( $action, $type, $product_id, $user_id ) {
		$log_message = sprintf(
			'Limit %s: %s limit for product #%d by user #%d',
			$action,
			$type,
			$product_id,
			$user_id
		);

		// Log to WooCommerce logger if available
		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();
			$logger->info( $log_message, array( 'source' => 'subscription-limits' ) );
		}

		// Also log to debug log
		if ( function_exists( 'wp_subscrpt_write_debug_log' ) ) {
			wp_subscrpt_write_debug_log( $log_message );
		}
	}

	/**
	 * Log renewal activity
	 *
	 * @param int $subscription_id Subscription ID.
	 * @param int $renewal_count Current renewal count.
	 */
	private static function log_renewal( $subscription_id, $renewal_count ) {
		$comment_id = wp_insert_comment(
			array(
				'comment_post_ID' => $subscription_id,
				'comment_author'  => 'System',
				'comment_content' => sprintf(
					// translators: %d: renewal count
					__( 'Subscription renewed. Renewal count: %d', 'wp_subscription_pro' ),
					$renewal_count
				),
				'comment_type'    => 'subscription_note',
				'user_id'         => 0,
			)
		);

		if ( $comment_id ) {
			update_comment_meta( $comment_id, '_subscrpt_activity', 'Renewal Count Updated' );
		}
	}

	/**
	 * Get limit information for display
	 *
	 * @param int $product_id Product ID.
	 * @return array Limit information.
	 */
	public static function get_limit_info( $product_id ) {
		$purchase_limit = get_post_meta( $product_id, '_subscrpt_limit', true );
		$renewal_limit  = get_post_meta( $product_id, '_subscrpt_renewal_limit', true );

		return array(
			'purchase_limit' => $purchase_limit,
			'renewal_limit'  => $renewal_limit ?: 'unlimited',
		);
	}

	/**
	 * Get user-friendly limit description
	 *
	 * @param int $product_id Product ID.
	 * @return string Description.
	 */
	public static function get_limit_description( $product_id ) {
		$info         = self::get_limit_info( $product_id );
		$descriptions = array();

		// Purchase limit description
		switch ( $info['purchase_limit_type'] ) {
			case 'unlimited':
				$descriptions[] = __( 'No purchase restrictions', 'wp_subscription_pro' );
				break;
			case 'one_active':
				$descriptions[] = __( 'One active subscription per customer', 'wp_subscription_pro' );
				break;
			case 'one_total':
				$descriptions[] = __( 'One subscription total per customer', 'wp_subscription_pro' );
				break;
			case 'numeric':
				$descriptions[] = sprintf(
					// translators: %d: maximum number of purchases per customer
					__( 'Maximum %d purchases per customer', 'wp_subscription_pro' ),
					$info['purchase_limit_count']
				);
				break;
		}

		// Renewal limit description
		switch ( $info['renewal_limit_type'] ) {
			case 'unlimited':
				$descriptions[] = __( 'No renewal restrictions', 'wp_subscription_pro' );
				break;
			case 'numeric':
				$descriptions[] = sprintf(
					// translators: %d: maximum number of renewal cycles
					__( 'Maximum %d renewal cycles', 'wp_subscription_pro' ),
					$info['renewal_limit_count']
				);
				break;
		}

		return implode( '. ', $descriptions );
	}
}
