<?php
/**
 * Live QR File
 *
 * @package SubscriptionPro\Illuminate
 */

namespace SpringDevs\SubscriptionPro\Illuminate\LiveQR;

use chillerlan\QRCode\Common\Version;
use chillerlan\QRCode\Data\QRMatrix;
use chillerlan\QRCode\Output\QRMarkupSVG;
use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;
use SpringDevs\SubscriptionPro\Illuminate\Helper;

/**
 * LiveQR class
 */
class LiveQR {
	/**
	 * Page title
	 *
	 * @var string
	 */
	private const PAGE_TITLE = 'Subscription Basic Details';

	/**
	 * Page slug
	 *
	 * @var string
	 */
	private const PAGE_SLUG = 'wp-subscription-basic-details';

	/**
	 * Constructor
	 */
	public function __construct() {
		// Create & Delete frontend page
		register_activation_hook( SUBSCRIPT_PRO_FILE, [ $this, 'create_subscription_basic_status_page' ] );
		register_deactivation_hook( SUBSCRIPT_PRO_FILE, [ $this, 'delete_subscription_basic_status_page' ] );

		// Add settings
		add_filter( 'subscrpt_settings_fields', [ $this, 'setting_fields' ] );
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Register Ajax actions
		add_action( 'wp_ajax_wpsubs_regenerate_qr', [ $this, 'ajax_regenerate_qr' ] );

		// Init hook
		add_action( 'init', [ $this, 'init' ] );
	}

	/**
	 * Init hook
	 */
	public function init() {
		// Register shortcode
		add_shortcode( 'wp_subs_basic_details', [ $this, 'render_subs_basic_details_page' ] );

		// Remove page title
		add_filter( 'the_title', [ $this, 'remove_subs_details_page_title' ] );

		// Create meta boxes in subscription edit page
		add_action( 'add_meta_boxes', [ $this, 'create_meta_boxes' ] );

		// Render QR code in frontend subscription details page
		add_action( 'subscrpt_after_subscription_totals', [ $this, 'render_frontend_qr_code' ] );
	}

	/**
	 * Setting fields
	 *
	 * @param array $settings_fields Existing settings fields.
	 * @return array Modified settings fields.
	 */
	public function setting_fields( $settings_fields ) {
		$qr_settings = [
			[
				'type'       => 'heading',
				'group'      => 'live_qr_settings',
				'priority'   => 5,
				'field_data' => [
					'title' => __( 'Quick Details QR Settings', 'wp_subscription_pro' ),
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'live_qr_settings',
				'priority'   => 1,
				'field_data' => [
					'id'      => 'subscrpt_live_qr_show_product',
					'title'   => __( 'Product Details', 'wp_subscription_pro' ),
					'label'   => __( 'Show product details in the subscription quick view', 'wp_subscription_pro' ),
					'value'   => '1',
					'checked' => '1' === get_option( 'subscrpt_live_qr_show_product', '1' ),
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'live_qr_settings',
				'priority'   => 2,
				'field_data' => [
					'id'      => 'subscrpt_live_qr_show_billing',
					'title'   => __( 'Billing Details', 'wp_subscription_pro' ),
					'label'   => __( 'Show billing details in the subscription quick view', 'wp_subscription_pro' ),
					'value'   => '1',
					'checked' => '1' === get_option( 'subscrpt_live_qr_show_billing', '0' ),
				],
			],
			[
				'type'       => 'toggle',
				'group'      => 'live_qr_settings',
				'priority'   => 3,
				'field_data' => [
					'id'      => 'subscrpt_live_qr_show_timeline',
					'title'   => __( 'Timeline', 'wp_subscription_pro' ),
					'label'   => __( 'Show subscription timeline in the subscription quick view', 'wp_subscription_pro' ),
					'value'   => '1',
					'checked' => '1' === get_option( 'subscrpt_live_qr_show_timeline', '0' ),
				],
			],
		];

		return array_merge( $settings_fields, $qr_settings );
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting(
			'wp_subscription_settings',
			'subscrpt_live_qr_show_product',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
		register_setting(
			'wp_subscription_settings',
			'subscrpt_live_qr_show_billing',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
		register_setting(
			'wp_subscription_settings',
			'subscrpt_live_qr_show_timeline',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
	}

	/**
	 * Get Settings
	 *
	 * @param string $id Setting ID.
	 */
	public static function get_settings( $id = '' ) {
		$settings = [
			'subscrpt_live_qr_show_product'  => get_option( 'subscrpt_live_qr_show_product', '1' ) === '1',
			'subscrpt_live_qr_show_billing'  => get_option( 'subscrpt_live_qr_show_billing', '0' ) === '1',
			'subscrpt_live_qr_show_timeline' => get_option( 'subscrpt_live_qr_show_timeline', '0' ) === '1',
		];
		return ! empty( $id ) ? $settings[ $id ] ?? false : $settings;
	}

	/**
	 * Get Subscription Basic Status Page URL
	 *
	 * @param int $subscription_id Subscription ID.
	 * @return string
	 */
	public static function get_page_url( int $subscription_id = 0 ) {
		$page = get_page_by_path( self::PAGE_SLUG );
		$url  = '';

		if ( $page ) {
			$url = get_permalink( $page );
		} else {
			( new self() )->create_subscription_basic_status_page();
			$page = get_page_by_path( self::PAGE_SLUG );
			$url  = $page ? get_permalink( $page ) : '';
		}

		if ( $subscription_id ) {
			$url = add_query_arg( 'subscription_id', $subscription_id, $url );
		}
		return $url;
	}

	/**
	 * Create Subscription Basic Status Page
	 */
	public function create_subscription_basic_status_page() {
		$page = get_page_by_path( self::PAGE_SLUG );

		if ( ! $page ) {
			$page_id = wp_insert_post(
				[
					'post_title'     => self::PAGE_TITLE,
					'post_name'      => self::PAGE_SLUG,
					'post_content'   => "<!-- wp:shortcode -->\n[wp_subs_basic_details]\n<!-- /wp:shortcode -->",
					'post_status'    => 'publish',
					'post_type'      => 'page',
					'post_author'    => 1,
					'comment_status' => 'closed',
				]
			);

			// Add a flag to mark this page as hidden from menus
			if ( $page_id && ! is_wp_error( $page_id ) ) {
				update_post_meta( $page_id, '_wpsubs_hidden_page', 1 );

				update_post_meta( $page_id, '_wp_page_template', 'default' );
				update_post_meta( $page_id, '_exclude_from_nav', true );
			}
		}
	}

	/**
	 * Delete Subscription Basic Status Page
	 */
	public function delete_subscription_basic_status_page() {
		$page = get_page_by_path( self::PAGE_SLUG );

		if ( $page ) {
			wp_delete_post( $page->ID, true );
		}
	}

	/**
	 * Remove Subscription Basic Details Page Title
	 *
	 * @param string $title Page title.
	 * @return string
	 */
	public function remove_subs_details_page_title( $title ) {
		if ( is_page( self::PAGE_SLUG ) && in_the_loop() ) {
			return '';
		}
		return $title;
	}

	/**
	 * Render Subscription Basic Details Page
	 */
	public function render_subs_basic_details_page() {
		// ? escaped intentionally
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$subscription_id   = isset( $_GET['subscription_id'] ) ? sanitize_text_field( wp_unslash( $_GET['subscription_id'] ) ) : 0;
		$subscription_data = FreeHelper::get_subscription_data( $subscription_id );

		$template_args = [
			'subscription_data' => $subscription_data,
		];

		if ( ! empty( $subscription_data ) ) {
			$status         = $subscription_data['status'] ?? '';
			$verbose_status = FreeHelper::get_verbose_status( $status );

			$is_grace_period = isset( $subscription_data['grace_period'] );
			$grace_remaining = $subscription_data['grace_period']['remaining_days'] ?? 0;
			$grace_end_date  = $subscription_data['grace_period']['end_date'] ?? '';
			$grace_end_date  = ! empty( $grace_end_date ) ? gmdate( 'M j, Y', strtotime( $grace_end_date ) ) : '-';

			$product_id          = $subscription_data['product']['product_id'] ?? 0;
			$product_name        = wc_get_product( $product_id ) ? wc_get_product( $product_id )->get_name() : '';
			$product_name        = ( strlen( $product_name ) > 50 ) ? substr( $product_name, 0, 47 ) . '...' : $product_name;
			$product_description = wc_get_product( $product_id ) ? wc_get_product( $product_id )->get_description() : '';
			$product_description = ( strlen( $product_description ) > 100 ) ? substr( $product_description, 0, 97 ) . '...' : $product_description;
			$product_description = ! empty( $product_description ) ? $product_description : '...';
			$product_image_url   = wc_get_product( $product_id ) ? wp_get_attachment_url( wc_get_product( $product_id )->get_image_id() ) : '';
			$product_image_url   = ! empty( $product_image_url ) ? $product_image_url : wc_placeholder_img_src();

			$order_id      = $subscription_data['order']['order_id'] ?? 0;
			$order_item_id = $subscription_data['order']['order_item_id'] ?? 0;

			$order      = wc_get_order( $order_id );
			$order_item = $order ? $order->get_item( $order_item_id ) : null;

			$payment_method = $order->get_payment_method_title() ?? '-';

			$price          = $subscription_data['price'] ?? 0;
			$price_excl_tax = (float) $order_item->get_total();
			$tax_amount     = (float) $order_item->get_total_tax();

			if ( $tax_amount > 0 ) {
				$price = $price_excl_tax + $tax_amount;
				$price = number_format( (float) $price, 2, '.', '' );
			} else {
				$tax_amount = 0;
			}
			$plan_price_formatted = FreeHelper::format_price_with_order_item( $price, $order_item_id );

			$start_date = $subscription_data['start_date'] ?? '';
			$start_date = ! empty( $start_date ) ? gmdate( 'M j, Y', strtotime( $start_date ) ) : '-';

			$next_date = $subscription_data['next_date'] ?? '';
			$next_date = ! empty( $next_date ) ? gmdate( 'M j, Y', strtotime( $next_date ) ) : '-';

			$include_activities      = [ 'subs_created', 'subs_activated', 'subs_cancelled', 'renewal_order' ];
			$subscription_activities = Helper::get_subscription_activities( $subscription_id, $include_activities );

			$subscription_activated_activities = array_filter(
				$subscription_activities,
				function ( $activity ) {
					return in_array( $activity['type'], [ 'subs_activated' ], true );
				}
			);

			$recent_payment_date = '-';
			if ( ! empty( $subscription_activated_activities ) ) {
				$last_activated_activity    = reset( $subscription_activated_activities );
				$recent_payment_date_string = $last_activated_activity['date'] ?? '';
				$recent_payment_date        = ! empty( $recent_payment_date_string ) ? gmdate( 'M j, Y', strtotime( $recent_payment_date_string ) ) : '-';
			}

			// Add to template args
			$template_args = array_merge(
				$template_args,
				[
					'subscription_id'         => $subscription_id,
					'status'                  => $status,
					'verbose_status'          => $verbose_status,
					'is_grace_period'         => $is_grace_period,
					'grace_remaining'         => $grace_remaining,
					'grace_end_date'          => $grace_end_date,
					'product_name'            => $product_name,
					'product_description'     => $product_description,
					'product_image_url'       => $product_image_url,
					'plan_price_formatted'    => $plan_price_formatted,
					'start_date'              => $start_date,
					'next_date'               => $next_date,
					'recent_payment_date'     => $recent_payment_date,
					'payment_method'          => $payment_method,
					'subscription_activities' => $subscription_activities,
				]
			);
		}

		// Settings
		$qr_settings               = $this->get_settings();
		$template_args['settings'] = [
			'show_product_details' => $qr_settings['subscrpt_live_qr_show_product'] ?? true,
			'show_billing_details' => $qr_settings['subscrpt_live_qr_show_billing'] ?? false,
			'show_timeline'        => $qr_settings['subscrpt_live_qr_show_timeline'] ?? false,
		];

		ob_start();
		wc_get_template(
			'subscription-basic-details.php',
			$template_args,
			'subscription-pro/LiveQR/', // Template directory inside theme
			SUBSCRIPT_PRO_PATH . '/templates/LiveQR/' // Default path in plugin
		);
		return ob_get_clean();
	}

	/**
	 * Create Meta Boxes
	 */
	public function create_meta_boxes() {
		add_meta_box(
			'subscrpt_live_qr',
			__( 'Quick Details QR', 'wp_subscription_pro' ),
			[ $this, 'render_live_qr_metabox' ],
			'subscrpt_order',
			'side',
			'default'
		);
	}

	/**
	 * Get qr image from subscription meta or generate new one
	 *
	 * @param int  $subscription_id Subscription ID.
	 * @param bool $force Force regenerate QR code.
	 * @return null|string Base64 encoded SVG image of the QR code.
	 */
	public function get_qr_image( $subscription_id, $force = false ) {
		$url      = self::get_page_url( $subscription_id );
		$qr_image = get_post_meta( $subscription_id, '_subscrpt_live_qr_image', true );

		if ( empty( $qr_image ) || $force ) {
			$args     = [
				'data' => $url,
			];
			$qr_image = self::generate_qr_code( $args );

			if ( $qr_image ) {
				update_post_meta( $subscription_id, '_subscrpt_live_qr_image', $qr_image );
			}
		}

		return $qr_image;
	}

	/**
	 * Render Live QR Code Meta Box
	 *
	 * @param WP_Post $post Post object.
	 */
	public function render_live_qr_metabox( $post ) {
		$subscription_id = $post->ID;
		$qr_image        = $this->get_qr_image( $subscription_id );

		// Load the view file.
		include 'views/liveqr-metabox.php';
	}

	/**
	 * Generate QR Code image
	 *
	 * @param array $args Data and other parameters to encode in the QR code.
	 * @return null|string Base64 encoded PNG image of the QR code.
	 */
	public static function generate_qr_code( $args ) {
		$default_args = [
			'data'    => '',
			'version' => Version::AUTO,
		];
		$args         = wp_parse_args( $args, $default_args );

		// ? escaped intentionally
		// phpcs:disable WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$options                  = new QROptions();
		$options->version         = $args['version'];
		$options->outputInterface = QRMarkupSVG::class;
		$options->outputBase64    = true;
		$options->connectPaths    = true;

		$options->moduleValues = [
			// finder patterns
			// QRMatrix::M_FINDER_DARK => 'red',
			// QRMatrix::M_FINDER_DOT  => 'red',

			// data modules
			QRMatrix::M_DATA_DARK => '#000000',   // qr modules
			QRMatrix::M_DATA      => '#ffffffff',   // background
		];

		/*
		$options->drawCircularModules = true;
		$options->circleRadius        = 0.4;
		$options->keepAsSquare        = [
			QRMatrix::M_FINDER_DARK,
			QRMatrix::M_FINDER_DOT,
			QRMatrix::M_ALIGNMENT_DARK,
		];
		*/
		// phpcs:enable WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase

		try {
			$qrcode = ( new QRCode( $options ) )->render( $args['data'] );

			return $qrcode;
		} catch ( \Exception $e ) {
			$subscription_id = $args['subscription_id'] ?? 0;

			$log_message = "QR Code generation failed for Subscription: #{$subscription_id}. Error: " . $e->getMessage();
			wp_subscrpt_write_log( $log_message );
			wp_subscrpt_write_debug_log( $log_message );

			return null;
		}
	}

	/**
	 * Ajax handler to regenerate QR code
	 */
	public function ajax_regenerate_qr() {
		check_ajax_referer( 'wpsubs_live_qr', 'nonce' );

		$subscription_id = isset( $_POST['subscription_id'] ) ? sanitize_text_field( wp_unslash( $_POST['subscription_id'] ) ) : 0;
		$qr_image        = $this->get_qr_image( $subscription_id, true );

		if ( $qr_image ) {
			wp_send_json_success( [ 'qr_image' => $qr_image ] );
		} else {
			wp_send_json_error( 'Failed to regenerate QR code.' );
		}
	}

	/**
	 * Render Frontend QR Code
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function render_frontend_qr_code( int $subscription_id ) {
		$qr_image = $this->get_qr_image( $subscription_id );

		if ( ! $qr_image ) {
			return;
		}

		?>
		<h2><?php echo esc_html_e( 'Quick Details QR Code', 'wp_subscription_pro' ); ?></h2>

		<div style="padding: 1em; border: 1px solid rgba(0, 0, 0, .1); width: 40%;">
			<img src="<?php echo esc_attr( $qr_image ); ?>" alt="Subscription Live QR Code" />
		</div>
		<?php
	}
}
