<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Action;
use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class Order
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class Order {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'subscrpt_order_status_changed', array( $this, 'handle_order_status_changed' ), 10, 2 );

		// Add auto complete order settings
		add_filter( 'subscrpt_settings_fields', [ $this, 'setting_fields' ] );
		add_action( 'subscrpt_register_settings', array( $this, 'register_settings' ) );

		// Auto complete processing orders
		// add_action( 'woocommerce_order_status_processing', [ $this, 'auto_complete_subscription_order' ] );
		add_action( 'after_subscription_update_cron', [ $this, 'auto_complete_processing_orders' ] );

		// Queue auto order complete tasks
		add_action( 'queue_auto_order_complete', [ $this, 'auto_complete_subscription_order' ] );
	}

	/**
	 * Add Auto Complete Order Setting Field.
	 *
	 * @param array $settings_fields Existing fields.
	 * @return array
	 */
	public function setting_fields( $settings_fields ) {
		$auto_complete_order_field = [
			[
				'type'       => 'toggle',
				'group'      => 'payment_gateways',
				'priority'   => 1,
				'field_data' => [
					'id'          => 'wp_subscription_auto_complete_order',
					'title'       => __( 'Auto Complete Orders', 'wp_subscription_pro' ),
					'description' => __( 'Automatically change status of processing orders to completed.', 'wp_subscription_pro' ),
					'value'       => '1',
					'checked'     => '1' === get_option( 'wp_subscription_auto_complete_order', '1' ),
				],
			],
		];

		return array_merge( $settings_fields, $auto_complete_order_field );
	}

	/**
	 * Register settings option.
	 */
	public function register_settings() {
		register_setting(
			'wp_subscription_settings',
			'wp_subscription_auto_complete_order',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
	}

	/**
	 * Handle Order status changed.
	 *
	 * @param \WC_Order $order   Order.
	 * @param mixed     $history Order relations.
	 */
	public function handle_order_status_changed( $order, $history ) {
		if ( 'early-renew' === $history->type ) {
			$early_meta = get_post_meta( $history->subscription_id, '_order_waiting_early_renew_subscrpt_meta', true );
			if ( $early_meta && 'completed' === $order->get_status() ) {
				update_post_meta( $history->subscription_id, '_subscrpt_order_id', $early_meta['order_id'] );
				update_post_meta( $history->subscription_id, '_subscrpt_order_item_id', $early_meta['order_item_id'] );
				wp_update_post(
					array(
						'ID'          => $history->subscription_id,
						'post_status' => 'active',
					)
				);

				// Increment renewal count for completed early renewal orders
				if ( class_exists( '\\SpringDevs\\SubscriptionPro\\Illuminate\\LimitChecker' ) ) {
					LimitChecker::increment_renewal_count( $history->subscription_id );
				}

				$order_item = $order->get_item( $early_meta['order_item_id'] );
				$product    = $order_item->get_product();
				if ( $product && $product->is_type( 'variation' ) ) {
					update_post_meta( $history->subscription_id, '_subscrpt_variation_id', $product->get_id() );
				}

				Action::write_comment( 'active', $history->subscription_id );
			}
			if ( in_array( $order->get_status(), array( 'cancelled', 'completed' ), true ) ) {
				delete_post_meta( $history->subscription_id, '_order_waiting_early_renew_subscrpt_meta' );
				if ( 'cancelled' === $order->get_status() ) {
					$comment_id = wp_insert_comment(
						array(
							'comment_author'  => 'Subscription for WooCommerce',
							'comment_content' => "Early renewal order {$order->get_id()} is cancelled.",
							'comment_post_ID' => $history->subscription_id,
							'comment_type'    => 'order_note',
						)
					);
					update_comment_meta( $comment_id, '_subscrpt_activity', 'Early Renewal Cancelled' );
					update_comment_meta( $comment_id, '_subscrpt_activity_type', 'early_renewal_cancelled' );
				}
			}
		}

		if ( in_array( $history->type, [ 'upgrade', 'downgrade' ], true ) ) {
			$order_status = $order->get_status();

			if ( in_array( $order_status, [ 'completed' ], true ) ) {
				$order_item     = $order->get_item( $history->order_item_id );
				$switch_context = $order_item->get_meta( '_wp_subs_switch_context' );

				// Update subscription after switch order is completed
				( new SubscriptionSwitch() )->subs_switch_update_subscription( $switch_context, $history, $order );

				$switch_type = $switch_context['switch_type'] ?? 'upgrade';

				unset( $switch_context['nonce'] );
				unset( $switch_context['redirect_back_url'] );

				/**
				 * Action fired after subscription switch completed and order is completed.
				 *
				 * @param string    $switch_type    Switch type (upgrade/downgrade).
				 * @param \WC_Order $order          Order object.
				 * @param \WC_Order_Item_Product $order_item Order Item object.
				 * @param array     $switch_context Switch context data. [ 'switch_type', 'subscription_id', 'order_id', 'product_id', 'old_variation_id', 'new_variation_id' ]
				 */
				do_action( 'subscrpt_switch_completed', $switch_type, $order, $order_item, $switch_context );
			}
		}
	}

	/**
	 * Auto complete order
	 *
	 * @param int $order_id Order ID.
	 */
	public function auto_complete_subscription_order( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$subscription_order         = FreeHelper::order_has_subscription_item( $order );
		$is_auto_complete_order     = in_array( get_option( 'wp_subscription_auto_complete_order', '1' ), [ 1, '1' ], true );
		$is_order_already_completed = 'completed' === $order->get_status();

		if ( $subscription_order && $is_auto_complete_order && ! $is_order_already_completed ) {
			$order->update_status( 'completed', __( 'Auto status change by WPSubscription Pro.', 'wp_subscription_pro' ) );
		}
	}

	/**
	 * Auto complete processing orders.
	 * Queues auto complete tasks for all processing orders.
	 */
	public function auto_complete_processing_orders() {
		$orders = wc_get_orders(
			[
				'status' => 'processing',
				'limit'  => -1,
			]
		);

		// Queue auto complete task
		foreach ( $orders as $order ) {
			$order_id = $order->get_id() ?? 0;
			as_enqueue_async_action( 'queue_auto_order_complete', [ 'order_id' => $order_id ] );

			$log_message = "Queued auto complete task for order ID: {$order_id}";
			wp_subscrpt_write_log( $log_message );
			wp_subscrpt_write_debug_log( $log_message );
		}
	}
}
