<?php

namespace SpringDevs\SubscriptionPro\Illuminate;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SubscriptionSwitch
 *
 * @package SpringDevs\SubscriptionPro\Illuminate
 */
class SubscriptionSwitch {
	/**
	 * Initialize the class
	 */
	public function __construct() {
		// Add switch subscription button.
		add_filter( 'subscrpt_single_action_buttons', [ $this, 'add_switch_subscription_button' ], 10, 4 );

		// Hide downgrade options.
		add_filter( 'woocommerce_get_children', [ $this, 'possibly_hide_downgrade_variations' ], 10, 2 );

		// Hook into product details page.
		add_action( 'wp', [ $this, 'handle_switch_subscription_selection' ] );

		// Add to cart handling for switch subscription.
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'subs_switch_add_cart_item_data' ], 10, 3 );
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_switch_meta_to_order_item' ], 10, 3 );

		// Validate cart item data for switch subscription.
		add_filter( 'woocommerce_add_to_cart_validation', [ $this, 'subs_switch_validate_cart' ], 10, 5 );

		// Persist url params in while adding to cart.
		add_filter( 'woocommerce_add_to_cart_redirect', [ $this, 'persist_params_on_cart_redirect' ], 10, 1 );

		// Safe redirect after switch subscription error.
		add_action( 'template_redirect', [ $this, 'maybe_redirect_after_add_to_cart_failure' ] );

		// Adjust order price.
		add_action( 'woocommerce_cart_calculate_fees', [ $this, 'switch_add_adjust_price' ], 20, 1 );
	}

	/**
	 * Add switch subscription button to single subscription page.
	 *
	 * @param array  $buttons Array of buttons.
	 * @param int    $id Subscription ID.
	 * @param string $subscrpt_nonce Nonce for subscription actions.
	 * @param string $status Subscription status.
	 */
	public function add_switch_subscription_button( array $buttons, int $id, string $subscrpt_nonce, string $status ): array {
		// Return if subscription is not active.
		if ( 'active' !== $status ) {
			return $buttons;
		}

		$order_id = get_post_meta( $id, '_subscrpt_order_id', true );
		$order    = wc_get_order( $order_id );

		$product_id   = null;
		$variation_id = null;
		if ( $order ) {
			$items = $order->get_items() ?? [];
			$item  = reset( $items );

			if ( $item instanceof \WC_Order_Item_Product ) {
				$product_id   = $item->get_product_id();
				$variation_id = $item->get_variation_id();
			}
		}

		// Return if the order item is not a variable item.
		if ( ! $variation_id || 0 === $variation_id ) {
			return $buttons;
		}

		$product     = wc_get_product( $product_id );
		$product_url = $product->get_permalink();

		$redirect_url = add_query_arg(
			[
				'wp_subs_switch' => '',
				'subscription'   => $id,
				'order'          => $order_id,
				'nonce'          => $subscrpt_nonce,
			],
			$product_url
		);

		$buttons['wp_subs_switch'] = [
			'url'   => $redirect_url,
			'label' => 'Switch Subscription',
			'class' => 'wp-subs-switch',
		];

		return $buttons;
	}

	/**
	 * Possibly hide downgrade variations.
	 *
	 * @param array       $children Array of children variation IDs.
	 * @param \WC_Product $product Product object.
	 */
	public function possibly_hide_downgrade_variations( $children, $product ) {
		// Get the current subscription context.
		$switch_context = $this->get_switch_subscription_context();

		// Check if downgrade is allowed.
		$downgrade_allowed = in_array( get_option( 'subscrpt_downgrade_allowed', '0' ), [ 1, '1', 'yes', 'on' ], true );

		// Return if not a subscription switch context.
		if ( empty( $switch_context ) || (int) $switch_context['product_id'] !== $product->get_id() || $downgrade_allowed ) {
			return $children;
		}

		$current_variation_id = $switch_context['old_variation_id'] ?? 0;
		$current_variation    = wc_get_product( $current_variation_id );

		foreach ( $children as $key => $child_id ) {
			$variation = wc_get_product( $child_id );

			if ( $variation->get_price() < $current_variation->get_price() ) {
				unset( $children[ $key ] );
			}
		}

		return $children;
	}

	/**
	 * Set subscription switch context in session.
	 *
	 * @param array $subscription_switch_data Subscription switch data.
	 */
	public function set_switch_subscription_context( $subscription_switch_data ) {
		if ( ! WC()->session ) {
			return;
		}

		WC()->session->set( 'wp_subs_switch_context', $subscription_switch_data );
	}

	/**
	 * Unset subscription switch context in session.
	 */
	public function unset_switch_subscription_context() {
		if ( ! WC()->session ) {
			return;
		}

		WC()->session->__unset( 'wp_subs_switch_context' );
	}

	/**
	 * Get subscription switch context from session.
	 */
	public function get_switch_subscription_context() {
		if ( ! WC()->session ) {
			return null;
		}

		return WC()->session->get( 'wp_subs_switch_context' );
	}

	/**
	 * Handle switch subscription selection.
	 */
	public function handle_switch_subscription_selection() {
		if ( ! is_product() || ! isset( $_GET['wp_subs_switch'] ) ) {
			// Clear any existing switch context.
			if ( WC()->session ) {
				WC()->session->__unset( 'wp_subs_switch_context' );
			}
			return;
		}

		$order_id        = isset( $_GET['order'] ) ? absint( $_GET['order'] ) : 0;
		$subscription_id = isset( $_GET['subscription'] ) ? absint( $_GET['subscription'] ) : 0;
		$nonce           = isset( $_GET['nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['nonce'] ) ) : null;

		$order = wc_get_order( $order_id );

		// Show Error or Notice.
		$current_user_id = get_current_user_id();
		$verify_nonce    = wp_verify_nonce( $nonce, 'subscrpt_nonce' );
		if ( ( $order && $current_user_id !== $order->get_user_id() ) || ! $verify_nonce ) {
			$notice = __( 'You are not allowed to switch this subscription.', 'wp_subscription_pro' );
			wc_add_notice( $notice, 'error' );
			return;
		} else {
			$notice = __( 'Please select an option to change the current subscription and checkout.', 'wp_subscription_pro' );
			wc_add_notice( $notice, 'notice' );
		}

		// Generate data for switch context.
		$product_id   = null;
		$variation_id = null;
		if ( $order ) {
			$items = $order->get_items() ?? [];
			$item  = reset( $items );

			if ( $item instanceof \WC_Order_Item_Product ) {
				$product_id   = $item->get_product_id();
				$variation_id = $item->get_variation_id();
			}
		}

		// Build redirect URL for errors.
		$redirect_url = '';
		$product      = wc_get_product( $product_id );
		if ( ! empty( $product ) ) {
			$product_url  = $product->get_permalink();
			$redirect_url = add_query_arg(
				[
					'wp_subs_switch' => '',
					'subscription'   => $subscription_id,
					'order'          => $order_id,
					'nonce'          => $nonce,
				],
				$product_url
			);
		}

		// Set data for switch context.
		$subscription_switch_data = [
			'switch_type'       => 'upgrade', // Default. It will be updated in 'subs_switch_add_cart_item_data' method.
			'subscription_id'   => $subscription_id,
			'order_id'          => $order_id,
			'product_id'        => $product_id,
			'old_variation_id'  => $variation_id,
			'new_variation_id'  => 0, // Default. It will be updated in 'subs_switch_add_cart_item_data' method.
			'nonce'             => $nonce,
			'redirect_back_url' => $redirect_url,
		];

		$this->set_switch_subscription_context( $subscription_switch_data );
	}

	/**
	 * Persist URL parameters while adding to cart.
	 *
	 * @param string $url Redirect URL.
	 */
	public function persist_params_on_cart_redirect( string $url ): string {
		if ( ! WC()->session || empty( $this->get_switch_subscription_context() ) ) {
			return $url;
		}

		$switch_context = $this->get_switch_subscription_context();

		return $switch_context['redirect_back_url'] ?? $url;
	}

	/**
	 * Add switch subscription data to cart item.
	 *
	 * @param array $cart_item_data Cart item data.
	 * @param int   $product_id Product ID.
	 * @param int   $variation_id Variation ID.
	 */
	public function subs_switch_add_cart_item_data( array $cart_item_data, int $product_id, int $variation_id = 0 ): array {
		// Get context from session.
		$switch_context = $this->get_switch_subscription_context();

		if ( ! empty( $switch_context ) ) {
			$new_variation = wc_get_product( $variation_id );
			$old_variation = wc_get_product( $switch_context['old_variation_id'] );

			$switch_type                        = ( $new_variation->get_price() < $old_variation->get_price() ) ? 'downgrade' : 'upgrade';
			$switch_context['switch_type']      = $switch_type;
			$switch_context['new_variation_id'] = $variation_id;

			$cart_item_data['wp_subs_switch'] = true;
			$cart_item_data['switch_context'] = $switch_context;
		}

		return $cart_item_data;
	}

	/**
	 * Add switch subscription meta to order item.
	 *
	 * @param \WC_Order_Item_Product $item Order item.
	 * @param string                 $cart_item_key Cart item key.
	 * @param array                  $cart_item Cart item data.
	 */
	public function add_switch_meta_to_order_item( \WC_Order_Item_Product $item, string $cart_item_key, array $cart_item ) {
		// ! We are doing this in free version. That is running before this.
		// ? check "SpringDevs\Subscription\Frontend\Checkout::save_order_item_product_meta"

		/* // phpcs:ignore
		if ( ! empty( $cart_item['wp_subs_switch'] ?? null ) && ! empty( $cart_item['switch_context'] ?? null ) ) {
			$switch_context = $cart_item['switch_context'];

			// Add switch context data to order item meta.
			$item->add_meta_data( 'wp_subs_switch', true, true );
			$item->add_meta_data( 'wp_subs_switch_context', $switch_context, true );
		}
		*/
	}

	/**
	 * Validate cart item data for switch subscription.
	 *
	 * @param bool  $valid Whether the item is valid.
	 * @param int   $product_id Product ID.
	 * @param int   $quantity Quantity of the product.
	 * @param int   $variation_id Variation ID.
	 * @param array $cart_item_data Cart item data.
	 */
	public function subs_switch_validate_cart( bool $valid, int $product_id, int $quantity, int $variation_id = 0, array $cart_item_data = [] ): bool {
		// Get context from session.
		$switch_context = $this->get_switch_subscription_context();

		// If no context is set, return as it is.
		if ( ! WC()->session || empty( $switch_context ) ) {
			return $valid;
		}

		// Build redirect URL.
		$redirect_url = $switch_context['redirect_back_url'] ?? '';
		if ( empty( $redirect_url ) ) {
			$product      = wc_get_product( $product_id );
			$product_url  = $product->get_permalink();
			$redirect_url = add_query_arg(
				[
					'wp_subs_switch' => '',
					'subscription'   => $switch_context['subscription_id'],
					'order'          => $switch_context['order_id'],
					'nonce'          => $switch_context['nonce'],
				],
				$product_url
			);
		}

		$flag = true;

		// Check if the product ID matches the subscription product ID.
		if ( $flag && ( (int) wp_get_post_parent_id( $variation_id ) !== (int) $switch_context['product_id'] ) ) {
			$error_notice = __( 'You can only switch to another variation of your current subscription product.', 'wp_subscription_pro' );
			wc_add_notice( $error_notice, 'error' );
			$flag = false;
		}

		// Check if the product selected is different from the current subscription product.
		if ( (int) $switch_context['old_variation_id'] === $variation_id ) {
			$error_notice = __( 'You are already subscribed to this product. Please select a different one.', 'wp_subscription_pro' );
			wc_add_notice( $error_notice, 'error' );
			$flag = false;
		}

		// Prevent adding additional products to cart.
		if ( $flag && ( ! WC()->cart->is_empty() ) ) {
			$error_notice = __( 'You can only have one item in the cart when switching a subscription.', 'wp_subscription_pro' );
			wc_add_notice( $error_notice, 'error' );

			WC()->session->set( 'wp_subs_switch_redirect_back_url', $redirect_url );

			return false;
		}

		// Clear cart if validation fails.
		if ( ! $flag ) {
			// Set redirect URL in session to redirect.
			WC()->session->set( 'wp_subs_switch_redirect_back_url', $redirect_url );

			// Clear cart.
			WC()->cart->empty_cart();
			return false;
		}
		return true;
	}

	/**
	 * Redirect after switch subscription error.
	 */
	public function maybe_redirect_after_add_to_cart_failure() {
		if ( ! WC()->session ) {
			return;
		}

		$redirect_url = WC()->session->get( 'wp_subs_switch_redirect_back_url' );

		if ( $redirect_url ) {
			WC()->session->__unset( 'wp_subs_switch_redirect_back_url' );
			wp_safe_redirect( $redirect_url );
			exit;
		}
	}

	/**
	 * Adjust order price for switch subscription.
	 *
	 * @param \WC_Cart $cart Cart object.
	 */
	public function switch_add_adjust_price( \WC_Cart $cart ) {
		if ( ! is_cart() && is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		// Check if a subscription is on the cart.
		$has_subs_in_cart = false;
		$cart_items       = WC()->cart->cart_contents;
		foreach ( $cart_items as $cart_item ) {
			if (
				isset( $cart_item['subscription'] ) ||
				$cart_item['data']->get_meta( '_subscrpt_enabled' )
			) {
				$has_subs_in_cart = true;
				break;
			}
		}

		// Subscription switch informations.
		$switch_context = $this->get_switch_subscription_context();

		if ( empty( $switch_context ) ) {
			foreach ( $cart_items as $cart_item ) {
				if ( ! empty( $cart_item['switch_context'] ?? null ) ) {
					$switch_context = $cart_item['switch_context'];
					break;
				}
			}
		}

		if ( ! empty( $switch_context ) && $has_subs_in_cart ) {
			$cart_total = $cart->get_cart_contents_total();

			$wc_product = wc_get_product( $switch_context['old_variation_id'] );
			$prev_price = $wc_product->get_price() ?? 0;

			$price_diff = $cart_total - $prev_price;
			$fee        = ( $cart_total - max( $price_diff, 0 ) ) * -1;

			$cart->add_fee( __( 'Price Adjustment', 'wp_subscription_pro' ), $fee );
		}
	}

	/**
	 * Update subscription information after switch order is processed.
	 *
	 * @param array     $switch_context Switch context information.
	 * @param object    $history Order relation information.
	 * @param \WC_Order $order Order object.
	 */
	public function subs_switch_update_subscription( $switch_context, $history, $order ) {
		$order_item   = $order->get_item( $history->order_item_id );
		$variation_id = $switch_context['new_variation_id'] ?? $order_item->get_variation_id();

		$subs_meta = wc_get_order_item_meta( $history->order_item_id, '_subscrpt_meta', true );

		$subscription_id = $switch_context['subscription_id'] ?? $history->subscription_id;

		if ( ! empty( $subs_meta ) ) {
			$timing_per    = $subs_meta['time'];
			$timing_option = $subs_meta['type'];
			$trial         = $subs_meta['trial'];
			$start_date    = $subs_meta['start_date'];
			$next_date     = $subs_meta['next_date'];

			// product related.
			update_post_meta( $subscription_id, '_subscrpt_timing_per', $timing_per );
			update_post_meta( $subscription_id, '_subscrpt_timing_option', $timing_option );
			$variation = wc_get_product( $variation_id );
			update_post_meta( $subscription_id, '_subscrpt_price', $variation->get_price() );
			update_post_meta( $subscription_id, '_subscrpt_variation_id', $variation_id );

			// order related.
			update_post_meta( $subscription_id, '_subscrpt_order_id', $order_item->get_order_id() );
			update_post_meta( $subscription_id, '_subscrpt_order_item_id', $order_item->get_id() );

			// subscription related.
			update_post_meta( $subscription_id, '_subscrpt_trial', $trial );
			// update_post_meta( $subscription_id, '_subscrpt_start_date', $start_date );
			// update_post_meta( $subscription_id, '_subscrpt_next_date', $next_date );

			$fees = $order_item->get_order()->get_fees();
			foreach ( $fees as $fee ) {
				if ( 'Signup Fee' === $fee->get_name() ) {
					update_post_meta( $subscription_id, '_subscrpt_signup_fee', $fee->get_amount() );
				}
			}
		}
	}
}
