<?php
/**
 * LearnPress integration class file.
 *
 * @package SpringDevs\SubscriptionPro\Integrations
 */

namespace SpringDevs\SubscriptionPro\Integrations;

use SpringDevs\Subscription\Illuminate\Helper as FreeHelper;

/**
 * Class LearnPress
 */
class LearnPress {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'init' ] );

		// Disable LP Order creation for renewal orders
		add_filter( 'subscrpt_before_saving_renewal_order', [ $this, 'disable_lp_order_creation' ], 10, 3 );
	}

	/**
	 * Initialize the LearnPress integration.
	 */
	public function init() {
		if ( ! $this->check_plugin() ) {
			return;
		}

		// Activate courses when subscription is activated
		add_action( 'subscrpt_subscription_activated', [ $this, 'maybe_complete_lp_order' ] );

		// Deactivate courses when subscription is cancelled/expired/grace period ended
		add_action( 'subscrpt_subscription_cancelled', [ $this, 'maybe_cancel_lp_order' ] );
		add_action( 'subscrpt_subscription_expired', [ $this, 'maybe_cancel_lp_order' ] );
		add_action( 'subscrpt_grace_period_ended', [ $this, 'maybe_cancel_lp_order' ] );
	}

	/**
	 * Check if LearnPress plugin is active.
	 */
	public function check_plugin() {
		$is_lp_plugin_active = is_plugin_active( 'learnpress/learnpress.php' );
		$is_lp_class_exists  = class_exists( 'LearnPress' );
		$is_lp_active        = $is_lp_plugin_active && $is_lp_class_exists;

		$is_lpw_plugin_active = is_plugin_active( 'learnpress-woo-payment/learnpress-woo-payment.php' );
		$is_lpw_class_exists  = class_exists( 'LP_Addon_Woo_Payment' );
		$is_lpw_active        = $is_lpw_plugin_active && $is_lpw_class_exists;

		return $is_lp_active && $is_lpw_active;
	}

	/**
	 * Enroll Courses
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function maybe_complete_lp_order( $subscription_id ) {
		$parent_order = FreeHelper::get_parent_order( $subscription_id );

		// Return if no parent order found
		if ( empty( $parent_order ) ) {
			return;
		}

		$lp_order_id = $parent_order->get_meta( '_learn_press_order_id' );
		if ( empty( $lp_order_id ) ) {
			return;
		}

		$lp_order = learn_press_get_order( $lp_order_id );
		if ( ! $lp_order ) {
			wp_subscrpt_write_log( 'Invalid LP Order [LP Order ID: ' . $lp_order_id . ']' );
			return;
		}

		$user_id = $parent_order->get_user_id();
		if ( ! $user_id ) {
			wp_subscrpt_write_log( 'Invalid LP UserID [LP Order ID: ' . $lp_order_id . ']' );
			return;
		}

		$lp_order->update_status( LP_ORDER_COMPLETED );
	}

	/**
	 * Unenroll Courses
	 *
	 * @param int $subscription_id Subscription ID.
	 */
	public function maybe_cancel_lp_order( $subscription_id ) {
		$subscription_data = FreeHelper::get_subscription_data( $subscription_id );
		$parent_order      = FreeHelper::get_parent_order( $subscription_id );

		// Return if no parent order found
		if ( empty( $parent_order ) ) {
			return;
		}

		// Check grace period. Do not cancel if in grace period.
		$grace_period = isset( $subscription_data['grace_period'] ) && $subscription_data['grace_period']['remaining_days'] > 0;
		if ( $grace_period ) {
			return;
		}

		$lp_order_id = $parent_order->get_meta( '_learn_press_order_id' );
		if ( empty( $lp_order_id ) ) {
			return;
		}

		$lp_order = learn_press_get_order( $lp_order_id );
		if ( ! $lp_order ) {
			wp_subscrpt_write_log( 'Invalid LP Order [LP Order ID: ' . $lp_order_id . ']' );
			return;
		}

		$user_id = $parent_order->get_user_id();
		if ( ! $user_id ) {
			wp_subscrpt_write_log( 'Invalid LP UserID [LP Order ID: ' . $lp_order_id . ']' );
			return;
		}

		$lp_order->update_status( LP_ORDER_CANCELLED );
	}

	/**
	 * Disable LP Order creation for renewal orders
	 *
	 * @param \WC_Order $new_order New WC Order ID.
	 * @param \WC_Order $old_order Old WC Order ID.
	 * @param int       $subscription_id Subscription ID.
	 */
	public function disable_lp_order_creation( $new_order, $old_order, $subscription_id ) {
		$parent_order = FreeHelper::get_parent_order( $subscription_id );

		// Return if no parent order found
		if ( empty( $parent_order ) ) {
			return $new_order;
		}

		$lp_order_id = $parent_order->get_meta( '_learn_press_order_id' );
		if ( ! empty( $lp_order_id ) ) {
			$new_order->update_meta_data( '_learn_press_order_id', $lp_order_id );
		}

		return $new_order;
	}
}
