<?php
/**
 * Delivery Schedules page template
 *
 * @var array $deliveries Array of delivery post objects
 * @var array $available_statuses Array of available delivery statuses
 * @var array $status Array of currently selected statuses
 * @var string $date_filter Selected date filter
 * @var string $search Search term
 * @var array $months Array of months for date filter
 * @var int $page Current page number
 * @var int $total_pages Total number of pages
 * @var int $total_deliveries Total number of deliveries
 * @var int $per_page Number of items per page
 */

// Ensure the filter is registered before calling the header
if ( class_exists( 'SpringDevs\Subscription\Admin\Menu' ) ) {
	$menu       = new \SpringDevs\Subscription\Admin\Menu();
	$reflection = new \ReflectionClass( $menu );
	$method     = $reflection->getMethod( 'render_admin_header' );
	$method->setAccessible( true );
	$method->invoke( $menu );
}
?>

<div class="wp-subscription-admin-content list-page">
	<div class="wp-subscription-list-title">
		<h1 class="wp-heading-inline"><?php esc_html_e( 'Delivery Schedules', 'wp_subscription_pro' ); ?></h1>
	</div>

	<?php if ( \Sdevs_Wc_Subscription_Pro::check_pro_license_or_alert() ) : ?>
		
		<form method="get" id="deliveries-form">
			<div class="wp-subscription-list-header">
				<div class="wp-subscription-filters">
					<input type="hidden" name="page" value="wp-subscription-delivery" />
					<div class="wp-subscription-status-filter">
						<label for="status-filter-toggle" class="status-filter-label">
							<span class="status-filter-text">
								<?php esc_html_e( 'Status Filter', 'wp_subscription_pro' ); ?>
							</span>

							<?php if ( ! empty( $status ) ) : ?>
								<span class="filter-count">(<?php echo count( $status ); ?>)</span>
							<?php endif; ?>
						</label>
						<div class="status-filter-dropdown">
							<?php foreach ( $available_statuses as $status_key => $status_label ) : ?>
								<label class="status-filter-option">
									<input type="checkbox" 
											name="subscrpt_status[]" 
											value="<?php echo esc_attr( $status_key ); ?>"
											<?php checked( in_array( $status_key, $status ) ); ?>>
									<?php echo esc_html( $status_label ); ?>
								</label>
							<?php endforeach; ?>
						</div>
					</div>
					<select name="date_filter" value="<?php echo esc_attr( $date_filter ); ?>">
						<option value=""><?php esc_html_e( 'All Dates', 'wp_subscription_pro' ); ?></option>
						<?php foreach ( $months as $val => $label ) : ?>
							<option value="<?php echo esc_attr( $val ); ?>" <?php selected( $date_filter, $val ); ?>><?php echo esc_html( $label ); ?></option>
						<?php endforeach; ?>
					</select>
					<input type="search" name="s" value="<?php echo esc_attr( $search ); ?>" placeholder="<?php esc_attr_e( 'Search by subscription ID...', 'wp_subscription_pro' ); ?>" />
					<select name="per_page">
						<?php foreach ( array( 10, 20, 50, 100 ) as $n ) : ?>
							<option value="<?php echo esc_html( $n ); ?>" <?php selected( $per_page, $n ); ?>><?php echo esc_html( $n ); ?> per page</option>
						<?php endforeach; ?>
					</select>
					<button type="submit" name="filter_action" value="filter" class="button">Search</button>
					<?php if ( ! empty( $status ) || ! empty( $date_filter ) || ! empty( $search ) ) : ?>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=wp-subscription-delivery' ) ); ?>" class="button">Reset</a>
					<?php endif; ?>
					<?php if ( ! empty( $status ) || ! empty( $date_filter ) || ! empty( $search ) ) : ?>
						<span>Filters applied</span>
					<?php endif; ?>
				</div>

				<div class="wp-subscription-actions" style="margin-top:8px;">
					<button type="button" id="subscrpt-print-all" class="button button-primary">Print Addresses</button>
					<button type="button" id="subscrpt-print-selected" class="button button-primary" style="display:none;">Print Addresses (Selected)</button>
				</div>
			</div>

		<table class="wp-list-table widefat fixed striped wp-subscription-modern-table">
			<thead>
				<tr>
					<th class="manage-column column-cb check-column">
						<input type="checkbox" id="subscrpt-select-all" />
					</th>
					<th><?php esc_html_e( 'Subscription ID', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Product', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Customer', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Delivery Status', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Shipping on', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Shipped on', 'wp_subscription_pro' ); ?></th>
					<th><?php esc_html_e( 'Delivery Info', 'wp_subscription_pro' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php if ( ! empty( $deliveries ) ) : ?>
					<?php foreach ( $deliveries as $delivery ) : ?>
						<?php
						// Get delivery data
						$subscription_id = get_post_meta( $delivery->ID, '_subscrpt_subscription_id', true );
						$order_item_id   = get_post_meta( $subscription_id, '_subscrpt_order_item_id', true );
						$order_id        = get_post_meta( $subscription_id, '_subscrpt_order_id', true );
						$order           = $order_id ? wc_get_order( $order_id ) : null;
						$order_item      = $order && $order_item_id ? $order->get_item( $order_item_id ) : null;

						// Customer data
						$customer       = $order ? $order->get_formatted_billing_full_name() : '-';
						$customer_id    = $order ? $order->get_customer_id() : 0;
						$customer_url   = $customer_id ? admin_url( 'user-edit.php?user_id=' . $customer_id ) : '';
						$customer_email = $order ? $order->get_billing_email() : '';

						// Product data
						$product_name = $order_item ? $order_item->get_name() : '-';
						$product_link = '';
						if ( $order_item && is_a( $order_item, 'WC_Order_Item_Product' ) ) {
							$variation_id = $order_item->get_variation_id();
							$product_id   = $order_item->get_product_id();
							$product_link = get_permalink( $variation_id !== 0 ? $variation_id : $product_id );
						}

						// Other data
						$subscription_link = $subscription_id ? get_edit_post_link( $subscription_id ) : '';
						$shipping_date     = get_post_meta( $delivery->ID, '_subscrpt_shipping_on', true );
						$shipped_date      = get_post_meta( $delivery->ID, '_subscrpt_shipped_date', true );
						$delivery_info     = $order ? $order->get_formatted_shipping_address() : '';
						$current_status    = get_post_status( $delivery->ID );
						?>

						<tr>
							<td class="check-column">
								<input type="checkbox" class="subscrpt-select-item" name="delivery_ids[]" value="<?php echo esc_attr( $delivery->ID ); ?>" />
							</td>
							<td>
								<?php if ( $subscription_link ) : ?>
									<a href="<?php echo esc_url( $subscription_link ); ?>" target="_blank">#<?php echo esc_html( $subscription_id ); ?></a>
								<?php else : ?>
									<?php echo esc_html( $subscription_id ); ?>
								<?php endif; ?>
							</td>

							<td>
								<?php if ( $product_link ) : ?>
									<a href="<?php echo esc_url( $product_link ); ?>" target="_blank"><?php echo esc_html( $product_name ); ?></a>
								<?php else : ?>
									<?php echo esc_html( $product_name ); ?>
								<?php endif; ?>
							</td>

							<td>
								<?php if ( $customer_url ) : ?>
									<a href="<?php echo esc_url( $customer_url ); ?>" target="_blank"><?php echo esc_html( $customer ); ?></a>
								<?php else : ?>
									<?php echo esc_html( $customer ); ?>
								<?php endif; ?>
								<?php if ( $customer_email ) : ?>
									<div class="wp-subscription-customer-email"><?php echo esc_html( $customer_email ); ?></div>
								<?php endif; ?>
							</td>

							<td>
								<select class="subscrpt-delivery-status-dropdown" data-delivery-id="<?php echo esc_attr( $delivery->ID ); ?>">
									<?php foreach ( $available_statuses as $status_key => $status_label ) : ?>
										<option value="<?php echo esc_attr( $status_key ); ?>" <?php selected( $current_status, $status_key ); ?>>
											<?php echo esc_html( $status_label ); ?>
										</option>
									<?php endforeach; ?>
								</select>
							</td>

							<td>
								<?php echo $shipping_date ? esc_html( date_i18n( get_option( 'date_format' ), $shipping_date ) ) : '-'; ?>
							</td>

							<td>
								<?php echo $shipped_date ? esc_html( date_i18n( get_option( 'date_format' ), $shipped_date ) ) : '-'; ?>
							</td>

							<td>
								<?php echo $delivery_info ? wp_kses_post( $delivery_info ) : '-'; ?>
							</td>
						</tr>
					<?php endforeach; ?>
				<?php else : ?>
					<tr>
						<td colspan="8" class="wp-subscription-list-empty">
							<?php esc_html_e( 'No deliveries found.', 'wp_subscription_pro' ); ?>
						</td>
					</tr>
				<?php endif; ?>
			</tbody>
		</table>

		<?php if ( $total_pages > 1 ) : ?>
		<div class="wp-subscription-pagination">
			<span class="total">Total <?php echo intval( $total_deliveries ); ?></span>
			<?php
			$base_url = remove_query_arg( 'paged' );
			for ( $i = 1; $i <= $total_pages; $i++ ) :
				$url        = add_query_arg(
					array(
						'paged'    => $i,
						'per_page' => $per_page,
					),
					$base_url
				);
				$is_current = $i == $page;
				?>
				<a href="<?php echo esc_url( $url ); ?>" 
					class="button<?php echo $is_current ? ' button-primary' : ''; ?>" 
					<?php echo $is_current ? 'disabled' : ''; ?>>
					<?php echo esc_html( $i ); ?>
				</a>
			<?php endfor; ?>
			<span class="goto-label">Go to</span>
			<form method="get">
				<input type="hidden" name="page" value="wp-subscription-delivery" />
				<input type="number" name="paged" min="1" max="<?php echo esc_attr( $total_pages ); ?>" value="<?php echo esc_attr( $page ); ?>" />
				<input type="hidden" name="per_page" value="<?php echo esc_attr( $per_page ); ?>" />
				<button type="submit" class="button">OK</button>
			</form>
		</div>
		<?php endif; ?>
	</form>

	<?php endif; ?>
</div>

<style>
.wp-subscription-status-filter {
	position: relative;
	display: inline-block;
}

.status-filter-label {
	display: inline-block;
	padding: 5px 12px;
	background: #fff;
	border: 1px solid #8c8f94;
	border-radius: 3px;
	cursor: pointer;
	user-select: none;
	font-size: 14px;
	min-width: 120px;
}

.status-filter-label:hover {
	color: #2271b1;
}

.status-filter-dropdown {
	position: absolute;
	top: 100%;
	left: 0;
	right: 0;
	background: #fff;
	border: 1px solid #8c8f94;
	border-radius: 3px;
	box-shadow: 0 1px 2px rgba(0,0,0,0.1);
	z-index: 1000;
	display: none;
	max-height: 180px;
	overflow-y: auto;
}

.status-filter-dropdown.show {
	display: block;
}

.status-filter-option {
	display: block;
	padding: 6px 8px;
	cursor: pointer;
	font-size: 14px;
	border-bottom: 1px solid #f0f0f0;
}

.status-filter-option:last-child {
	border-bottom: none;
}

.status-filter-option:hover {
	background: #f6f7f7;
}

.status-filter-option input[type="checkbox"] {
	margin-right: 8px;
	vertical-align: middle;
}

.filter-count {
	color: #0073aa;
	font-weight: bold;
}
</style>
