<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Handle subscription background process.
 * 
 * @class SUMOSubs_Background_Process
 */
class SUMOSubs_Background_Process {

    /**
     * Init SUMOSubs_Background_Process
     */
    public static function init() {
        add_action( 'sumosubscriptions_find_products_to_bulk_update', __CLASS__ . '::find_products_to_bulk_update', 10 );
        add_action( 'sumosubscriptions_find_subscriptions_to_bulk_update', __CLASS__ . '::find_subscriptions_to_bulk_update', 10, 2 );
        add_action( 'sumosubscriptions_update_products_in_bulk', __CLASS__ . '::update_products_in_bulk', 10, 1 );
        add_action( 'sumosubscriptions_update_subscriptions_in_bulk', __CLASS__ . '::update_subscriptions_in_bulk', 10, 3 );

        add_action( 'delete_post', __CLASS__ . '::delete_post' );

        add_action( 'sumosubs_scheduled_subscription_activation', __CLASS__ . '::subscription_activation', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_reactivation', __CLASS__ . '::subscription_reactivation', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_due_order_creation', __CLASS__ . '::subscription_due_order_creation', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_due_payment_automatic', __CLASS__ . '::subscription_due_payment_automatic', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_due_payment_retry', __CLASS__ . '::subscription_due_payment_automatic', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_reminder', __CLASS__ . '::subscription_reminder', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_overdue', __CLASS__ . '::subscription_overdue', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_suspend', __CLASS__ . '::subscription_suspend', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_cancel', __CLASS__ . '::subscription_cancel', 10, 1 );
        add_action( 'sumosubs_scheduled_subscription_expiration', __CLASS__ . '::subscription_expiration', 10, 1 );
    }

    /**
     * Check if the subscription is valid to continue.
     * 
     * @param int $subscription_id
     * @return bool
     */
    private static function is_subscription_valid( $subscription_id ) {
        if ( ! sumo_is_subscription_exists( $subscription_id ) ) {
            return false;
        }

        /**
         * Can restrict subscription product when out of stock.
         *
         * @since 15.6.0
         * @param $subscription_id subscription ID
         * @return bool
         */
        if ( apply_filters( 'sumosubscriptions_can_restrict_subscription_product_when_outofstock', false, $subscription_id ) ) {
            return false;
        }

        return true;
    }

    /**
     * Find products to update in bulk.
     * 
     * @since 15.5.0
     */
    public static function find_products_to_bulk_update() {
        $found_products = get_transient( 'sumosubs_found_products_to_bulk_update' );

        if ( empty( $found_products ) || ! is_array( $found_products ) ) {
            return;
        }

        $found_products = array_filter( array_chunk( $found_products, 10 ) );
        foreach ( $found_products as $index => $chunked_products ) {
            WC()->queue()->schedule_single(
                    time() + $index, 'sumosubscriptions_update_products_in_bulk', array(
                'products' => $chunked_products,
                    ), 'sumosubscriptions-product-bulk-updates' );
        }
    }

    /**
     * Find subscriptions to update in bulk
     * 
     * @since 15.5.0
     * @param int $deleted_product_id Deleted Product ID
     * @param int $replace_product_id Replace Product ID
     */
    public static function find_subscriptions_to_bulk_update( $deleted_product_id, $replace_product_id ) {
        $subscription_ids = get_transient( 'sumosubs_update_found_subscriptions' );
        if ( empty( $subscription_ids ) || ! is_array( $subscription_ids ) ) {
            return;
        }

        $subscription_ids = array_filter( array_chunk( $subscription_ids, 10 ) );
        foreach ( $subscription_ids as $index => $chunked_payments ) {
            WC()->queue()->schedule_single(
                    time() + $index, 'sumosubscriptions_update_subscriptions_in_bulk', array(
                'subscriptions'      => $chunked_payments,
                'deleted_product_id' => $deleted_product_id,
                'replace_product_id' => $replace_product_id,
                    ), 'sumosubs-subscription-bulk-updates' );
        }
    }

    /**
     * Start bulk updation of products.
     */
    public static function update_products_in_bulk( $products ) {
        $product_props = array();

        foreach ( SUMOSubs_Admin_Product_Settings::get_subscription_fields() as $field_name => $type ) {
            $meta_key                   = "sumo_{$field_name}";
            $product_props[ $meta_key ] = get_option( "bulk_{$meta_key}" );
        }

        foreach ( $products as $product_id ) {
            $_product = wc_get_product( $product_id );

            if ( ! $_product ) {
                continue;
            }

            switch ( $_product->get_type() ) {
                case 'simple':
                case 'variation':
                    SUMOSubs_Admin_Product_Settings::save_meta( $product_id, '', $product_props );
                    break;
                case 'variable':
                    $variations = get_children( array(
                        'post_parent' => $product_id,
                        'post_type'   => 'product_variation',
                        'fields'      => 'ids',
                        'post_status' => array( 'publish', 'private' ),
                        'numberposts' => -1,
                            ) );

                    if ( empty( $variations ) ) {
                        continue 2;
                    }

                    foreach ( $variations as $variation_id ) {
                        if ( $variation_id ) {
                            SUMOSubs_Admin_Product_Settings::save_meta( $variation_id, '', $product_props );
                        }
                    }
                    break;
            }
        }
    }

    /**
     * Start bulk update of subscriptions.
     * 
     * @since 15.5.0
     * @param array $subscription_ids Subscription Ids
     * @param int $deleted_product_id Deleted Product ID
     * @param int $replace_product_id Replace Product ID
     */
    public static function update_subscriptions_in_bulk( $subscription_ids, $deleted_product_id, $replace_product_id ) {
        $replace_product = wc_get_product( $replace_product_id );
        if ( ! $replace_product ) {
            return;
        }

        foreach ( $subscription_ids as $subscription_id ) {
            $subscription_meta = sumo_get_subscription_meta( $subscription_id );
            if ( ! $subscription_meta ) {
                continue;
            }

            $parent_order_id  = get_post_meta( $subscription_id, 'sumo_get_parent_order_id', true );
            $old_parent_order = wc_get_order( $parent_order_id );
            if ( ! $old_parent_order ) {
                continue;
            }

            $parent_order_item_data          = get_post_meta( $subscription_id, 'sumo_subscription_parent_order_item_data', true );
            $subscriptions_from_parent_order = $old_parent_order->get_meta( 'sumo_subsc_get_available_postids_from_parent_order', true );
            $payment_info                    = $old_parent_order->get_meta( 'sumosubscription_payment_order_information', true );

            unset( $subscriptions_from_parent_order[ $subscription_meta[ 'productid' ] ] );
            $subscriptions_from_parent_order[ $replace_product_id ] = absint( $subscription_id );

            if ( isset( $payment_info[ $subscription_meta[ 'productid' ] ] ) ) {
                $payment_info[ $replace_product_id ] = $payment_info[ $subscription_meta[ 'productid' ] ];
                unset( $payment_info[ $subscription_meta[ 'productid' ] ] );
            }

            foreach ( $old_parent_order->get_items() as $item_id => $item ) {
                $old_product_id                             = $item[ 'variation_id' ] > 0 ? $item[ 'variation_id' ] : $item[ 'product_id' ];
                $parent_order_item_data[ $item_id ][ 'id' ] = $replace_product_id;

                if ( empty( $old_product_id ) || $old_product_id == $deleted_product_id ) {
                    $item->set_product_id( $replace_product_id );
                    $item->set_name( $replace_product->get_name() );
                    $item->save();
                }
            }

            $subscription_meta[ 'productid' ] = $replace_product_id;
            update_post_meta( $subscription_id, 'sumo_subscription_product_details', $subscription_meta );
            update_post_meta( $subscription_id, 'sumo_subscription_parent_order_item_data', $parent_order_item_data );
            update_post_meta( $subscription_id, 'sumo_product_name', wc_get_product( $replace_product_id )->get_title() );

            $old_parent_order->update_meta_data( 'sumo_subsc_get_available_postids_from_parent_order', $subscriptions_from_parent_order );
            $old_parent_order->update_meta_data( 'sumosubscription_payment_order_information', $payment_info );
            $old_parent_order->save_meta_data();

            $old_renewal_order_id    = get_post_meta( $subscription_id, 'sumo_get_renewal_id', true );
            $unpaid_due_order_exists = sumosubs_unpaid_renewal_order_exists( $subscription_id );
            if ( $unpaid_due_order_exists ) {
                wc_get_order( $old_renewal_order_id )->delete( 'true' );
                $renewal_order_id = SUMOSubs_Order::create_renewal_order( $parent_order_id, $subscription_id );
                sumo_add_subscription_note( sprintf( __( 'Renewal order #%1$s has been deleted and replaced with a new order #%2$s.', 'sumosubscriptions' ), $old_renewal_order_id, $renewal_order_id ), $subscription_id, 'success', __( 'Renewal Order Created', 'sumosubscriptions' ) );
            }
        }
    }

    /**
     * Removes schedules belonging to a deleted post.
     *
     * @param mixed $id ID of post being deleted.
     */
    public static function delete_post( $id ) {
        if ( ! $id ) {
            return;
        }

        if ( 'sumosubscriptions' === get_post_type( $id ) ) {
            SUMOSubs_Action_Scheduler::unschedule_all_actions( $id );
        }
    }

    /**
     * Subscription activation.
     * 
     * @param array $args
     */
    public static function subscription_activation( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id' => 0
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        if ( 'Pending' === get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ) ) {
            $parent_order_id = get_post_meta( $args[ 'subscription_id' ], 'sumo_get_parent_order_id', true );

            SUMOSubs_Order::maybe_activate_subscription( $args[ 'subscription_id' ], $parent_order_id, 'pending', 'active', true );
        }
    }

    /**
     * Subscription reactivation.
     * 
     * @param array $args
     */
    public static function subscription_reactivation( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id' => 0
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        if ( 'Pause' === get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ) ) {
            update_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', 'Active' );
            sumo_add_subscription_note( __( 'Subscription resumed automatically.', 'sumosubscriptions' ), $args[ 'subscription_id' ], sumo_note_status( 'Active' ), __( 'Subscription resumed automatically', 'sumosubscriptions' ) );
            SUMOSubs_Order::set_next_payment_date( $args[ 'subscription_id' ] );
        }
    }

    /**
     * Subscription due order creation.
     * 
     * @param array $args
     */
    public static function subscription_due_order_creation( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id' => 0,
            'next_due_on'     => ''
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        if ( in_array( get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ), array( 'Active', 'Trial', 'Pending' ) ) ) {
            $due_order_id            = absint( get_post_meta( $args[ 'subscription_id' ], 'sumo_get_renewal_id', true ) );
            $unpaid_due_order_exists = sumosubs_unpaid_renewal_order_exists( $args[ 'subscription_id' ] );

            if ( ! $unpaid_due_order_exists ) {
                $due_order_id = SUMOSubs_Order::create_renewal_order( get_post_meta( $args[ 'subscription_id' ], 'sumo_get_parent_order_id', true ), $args[ 'subscription_id' ] );
            }

            switch ( sumo_get_payment_type( $args[ 'subscription_id' ] ) ) {
                case 'auto':
                    SUMOSubs_Action_Scheduler::schedule_due_payment_automatic( $args[ 'subscription_id' ], $args[ 'next_due_on' ], $due_order_id );

                    if ( ! $unpaid_due_order_exists ) {
                        SUMOSubs_Action_Scheduler::schedule_subscription_reminders( $args[ 'subscription_id' ], $args[ 'next_due_on' ], $due_order_id, 'subscription_auto_renewal_reminder' );
                    }
                    break;
                default:
                    SUMOSubs_Action_Scheduler::schedule_due_payment( $args[ 'subscription_id' ], $args[ 'next_due_on' ] );

                    if ( ! $unpaid_due_order_exists ) {
                        SUMOSubs_Action_Scheduler::schedule_subscription_reminders( $args[ 'subscription_id' ], $args[ 'next_due_on' ], $due_order_id );
                    }
                    break;
            }
        }
    }

    /**
     * Subscription due payment for automatic subscription.
     * 
     * @param array $args
     */
    public static function subscription_due_payment_automatic( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'             => 0,
            'parent_order_id'             => 0,
            'renewal_order_id'            => 0,
            'next_eligible_status'        => '',
            'payment_charging_days'       => 0,
            'payment_retry_times_per_day' => 0,
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        $args[ 'parent_order_id' ]  = absint( $args[ 'parent_order_id' ] > 0 ? $args[ 'parent_order_id' ] : get_post_meta( $args[ 'subscription_id' ], 'sumo_get_parent_order_id', true ) );
        $args[ 'renewal_order_id' ] = absint( $args[ 'renewal_order_id' ] > 0 ? $args[ 'renewal_order_id' ] : get_post_meta( $args[ 'subscription_id' ], 'sumo_get_renewal_id', true ) );
        $next_due_date              = get_post_meta( $args[ 'subscription_id' ], 'sumo_get_next_payment_date', true );

        $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
        if ( ! $renewal_order ) {
            return;
        }

        if ( sumosubs_is_order_paid( $renewal_order ) ) {
            return;
        }

        $payment_method = sumo_get_subscription_payment_method( $args[ 'subscription_id' ] );
        update_post_meta( $args[ 'subscription_id' ], '_subscription_payment_method', $payment_method );
        $renewal_order->set_payment_method( $payment_method );
        $renewal_order->save();

        if ( sumo_is_next_renewal_possible( $args[ 'subscription_id' ] ) ) {
            if ( SUMOSubs_Preapproval::is_valid( $args[ 'subscription_id' ], $renewal_order ) ) {
                if ( SUMOSubs_Preapproval::is_payment_txn_success( $args[ 'subscription_id' ], $renewal_order ) ) {
                    /**
                     * Fire after Pre approved payment Transaction is Successfully.
                     * 
                     * @since 1.0
                     */
                    do_action( 'sumosubscriptions_preapproved_payment_transaction_success', $args );
                } else {
                    /**
                     * Fire after Pre approved payment Transaction is Failed.
                     * 
                     * @since 1.0
                     */
                    do_action( 'sumosubscriptions_preapproved_payment_transaction_failed', $args );
                }
            } else {
                /**
                 * Fire after Pre approved access is revoked.
                 * 
                 * @since 1.0
                 */
                do_action( 'sumosubscriptions_preapproved_access_is_revoked', $args );
            }
        } else {
            SUMOSubs_Action_Scheduler::unschedule_all_actions( $args[ 'subscription_id' ] );
            SUMOSubs_Action_Scheduler::schedule_subscription_expiration( $args[ 'subscription_id' ], $next_due_date );

            update_post_meta( $args[ 'subscription_id' ], 'sumo_get_saved_due_date', $next_due_date );
            update_post_meta( $args[ 'subscription_id' ], 'sumo_get_next_payment_date', '--' );
        }
    }

    /**
     * Subscription reminder.
     * 
     * @param array $args
     */
    public static function subscription_reminder( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'  => 0,
            'renewal_order_id' => 0,
            'mail_template_id' => 'subscription_invoice',
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        switch ( $args[ 'mail_template_id' ] ) {
            case 'subscription_invoice':
            case 'subscription_auto_renewal_reminder':
            case 'subscription_pending_authorization':
                $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
                if ( ! $renewal_order ) {
                    break;
                }

                if ( sumosubs_is_order_paid( $renewal_order ) ) {
                    break;
                }

                if ( $renewal_order->get_total() > 0 && in_array( get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ), array( 'Active', 'Trial', 'Pending', 'Pending_Authorization' ) ) ) {
                    sumo_trigger_subscription_email( $args[ 'mail_template_id' ], $args[ 'renewal_order_id' ], $args[ 'subscription_id' ] );
                }
                break;
            case 'subscription_overdue_automatic':
            case 'subscription_suspended_automatic':
            case 'subscription_overdue_manual':
            case 'subscription_suspended_manual':
                $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
                if ( ! $renewal_order ) {
                    break;
                }

                if ( sumosubs_is_order_paid( $renewal_order ) ) {
                    break;
                }

                if ( $renewal_order->get_total() > 0 && in_array( get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ), array( 'Overdue', 'Suspended' ) ) ) {
                    sumo_trigger_subscription_email( $args[ 'mail_template_id' ], $args[ 'renewal_order_id' ], $args[ 'subscription_id' ] );
                }
                break;
            case 'subscription_expiry_reminder':
                if ( 'Active' === get_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', true ) ) {
                    sumo_trigger_subscription_email( $args[ 'mail_template_id' ], 0, $args[ 'subscription_id' ] );
                }
                break;
        }
    }

    /**
     * Subscription overdue.
     * 
     * @param array $args
     */
    public static function subscription_overdue( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'             => 0,
            'renewal_order_id'            => 0,
            'next_due_on'                 => '',
            'payment_charging_days'       => 0,
            'payment_retry_times_per_day' => sumosubs_get_payment_retry_times_per_day_in( 'Overdue' ),
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
        if ( ! $renewal_order ) {
            return;
        }

        if ( sumosubs_is_order_paid( $renewal_order ) ) {
            return;
        }

        if ( $renewal_order->get_total() <= 0 ) {
            $renewal_order->payment_complete();
            return;
        }

        update_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', 'Overdue' );

        $mail_template = 'auto' === sumo_get_payment_type( $args[ 'subscription_id' ] ) ? 'subscription_overdue_automatic' : 'subscription_overdue_manual';
        $remind_before = sumo_get_subscription_timestamp() + ( $args[ 'payment_charging_days' ] * 86400 );

        sumo_add_subscription_note( __( 'Since the renewal payment has not been paid so far, subscription status moved to Overdue.', 'sumosubscriptions' ), $args[ 'subscription_id' ], sumo_note_status( 'Overdue' ), __( 'Subscription Overdue', 'sumosubscriptions' ) );

        SUMOSubs_Action_Scheduler::unschedule_all_actions( $args[ 'subscription_id' ] );
        SUMOSubs_Action_Scheduler::schedule_due_payment( $args[ 'subscription_id' ], $args[ 'next_due_on' ], $args[ 'payment_charging_days' ], $args[ 'payment_retry_times_per_day' ] );
        SUMOSubs_Action_Scheduler::schedule_subscription_reminders( $args[ 'subscription_id' ], $remind_before, $args[ 'renewal_order_id' ], $mail_template );

        if ( apply_filters( 'sumosubscriptions_send_instant_email_overdue', true, $args ) ) {
            sumo_trigger_subscription_email( $mail_template, $args[ 'renewal_order_id' ], $args[ 'subscription_id' ] );
        }

        /**
         * Subscription active still.
         * 
         * @since 1.0
         */
        do_action( 'sumosubscriptions_active_subscription', $args[ 'subscription_id' ], $args[ 'renewal_order_id' ] );
    }

    /**
     * Subscription suspend.
     * 
     * @param array $args
     */
    public static function subscription_suspend( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'             => 0,
            'renewal_order_id'            => 0,
            'next_due_on'                 => '',
            'payment_charging_days'       => 0,
            'payment_retry_times_per_day' => 0,
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
        if ( ! $renewal_order ) {
            return;
        }

        if ( sumosubs_is_order_paid( $renewal_order ) ) {
            return;
        }

        if ( $renewal_order->get_total() <= 0 ) {
            $renewal_order->payment_complete();
            return;
        }

        update_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', 'Suspended' );

        $mail_template = 'auto' === sumo_get_payment_type( $args[ 'subscription_id' ] ) ? 'subscription_suspended_automatic' : 'subscription_suspended_manual';
        $remind_before = sumo_get_subscription_timestamp() + ( $args[ 'payment_charging_days' ] * 86400 );

        sumo_add_subscription_note( __( 'Since the renewal payment has not been paid so far, subscription status moved to Suspended.', 'sumosubscriptions' ), $args[ 'subscription_id' ], sumo_note_status( 'Suspended' ), __( 'Subscription Suspended', 'sumosubscriptions' ) );

        SUMOSubs_Action_Scheduler::unschedule_all_actions( $args[ 'subscription_id' ] );
        SUMOSubs_Action_Scheduler::schedule_due_payment( $args[ 'subscription_id' ], $args[ 'next_due_on' ], $args[ 'payment_charging_days' ], $args[ 'payment_retry_times_per_day' ] );
        SUMOSubs_Action_Scheduler::schedule_subscription_reminders( $args[ 'subscription_id' ], $remind_before, $args[ 'renewal_order_id' ], $mail_template );

        sumo_trigger_subscription_email( $mail_template, $args[ 'renewal_order_id' ], $args[ 'subscription_id' ] );

        /**
         * Subscription not active here.
         * 
         * @since 1.0
         */
        do_action( 'sumosubscriptions_pause_subscription', $args[ 'subscription_id' ], $args[ 'renewal_order_id' ] );
    }

    /**
     * Subscription cancel.
     * 
     * @param array $args
     */
    public static function subscription_cancel( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'  => 0,
            'renewal_order_id' => 0,
            'force_cancel'     => false,
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        $args[ 'renewal_order_id' ] = absint( $args[ 'renewal_order_id' ] > 0 ? $args[ 'renewal_order_id' ] : get_post_meta( $args[ 'subscription_id' ], 'sumo_get_renewal_id', true ) );

        if ( ! $args[ 'force_cancel' ] ) {
            $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
            if ( ! $renewal_order ) {
                return;
            }

            if ( sumosubs_is_order_paid( $renewal_order ) ) {
                return;
            }

            if ( $renewal_order->get_total() <= 0 ) {
                $renewal_order->payment_complete();
                return;
            }
        }

        sumosubs_cancel_subscription( $args[ 'subscription_id' ], array(
            'note' => __( 'Subscription automatically Cancelled.', 'sumosubscriptions' ),
        ) );
    }

    /**
     * Subscription expiration.
     * 
     * @param array $args
     */
    public static function subscription_expiration( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id' => 0,
            'expiry_on'       => '',
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        sumo_expire_subscription( $args[ 'subscription_id' ], $args[ 'expiry_on' ] );
    }

    /**
     * Subscription pending authorization.
     *
     * @param array $args
     */
    public static function subscription_pending_authorization( $args ) {
        $args = wp_parse_args( $args, array(
            'subscription_id'       => 0,
            'renewal_order_id'      => 0,
            'payment_charging_days' => 0,
                ) );

        if ( ! self::is_subscription_valid( $args[ 'subscription_id' ] ) ) {
            return;
        }

        $renewal_order = wc_get_order( $args[ 'renewal_order_id' ] );
        if ( ! $renewal_order ) {
            return;
        }

        if ( sumosubs_is_order_paid( $renewal_order ) ) {
            return;
        }

        update_post_meta( $args[ 'subscription_id' ], 'sumo_get_status', 'Pending_Authorization' );

        sumo_add_subscription_note( __( 'Since the renewal payment has not been paid so far, subscription status moved to Pending Authorization.', 'sumosubscriptions' ), $args[ 'subscription_id' ], sumo_note_status( 'Pending' ), __( 'Subscription Pending Authorization', 'sumosubscriptions' ) );

        $next_due_on   = get_post_meta( $args[ 'subscription_id' ], 'sumo_get_next_payment_date', true );
        $remind_before = sumo_get_subscription_timestamp( $next_due_on ) + ( $args[ 'payment_charging_days' ] * 86400 );

        SUMOSubs_Action_Scheduler::unschedule_all_actions( $args[ 'subscription_id' ] );
        SUMOSubs_Action_Scheduler::schedule_due_payment( $args[ 'subscription_id' ], $next_due_on, $args[ 'payment_charging_days' ] );
        SUMOSubs_Action_Scheduler::schedule_subscription_reminders( $args[ 'subscription_id' ], $remind_before, $args[ 'renewal_order_id' ], 'subscription_pending_authorization' );

        /**
         * After subscription is in pending auth.
         * 
         * @since 1.0
         */
        do_action( 'sumosubscriptions_status_in_pending_authorization', $args[ 'subscription_id' ] );

        /**
         * Subscription active here.
         * 
         * @since 1.0
         */
        do_action( 'sumosubscriptions_active_subscription', $args[ 'subscription_id' ], $args[ 'renewal_order_id' ] );
    }
}

SUMOSubs_Background_Process::init();
