<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Handle Subscription cron events.
 * 
 * @class SUMOSubs_Legacy_Scheduler
 */
class SUMOSubs_Legacy_Scheduler {

    /**
     * The Subscription Cron Event post ID. Handles each Subscription Cron.
     * 
     * @var int 
     */
    public $event_id = 0;

    /**
     * The Subscription post ID
     * 
     * @var int 
     */
    public $subscription_id = 0;

    /**
     * Cron Events for the Subscription
     * 
     * @var array 
     */
    public $cron_events = array();

    /**
     * Post Type for Cron Events
     * 
     * @var string 
     */
    protected $post_type = 'sumosubs_cron_events';

    /**
     * Constructor.
     */
    public function __construct( $subscription ) {
        $this->populate( $subscription );
    }

    /**
     * Populate the Subscription Cron Event
     *
     * @param int $subscription The Subscription post ID
     */
    protected function populate( $subscription ) {
        if ( ! $subscription ) {
            return false;
        }

        $this->subscription_id = absint( $subscription );
        $this->event_id        = $this->get_event_id();
        $this->cron_events     = $this->get_cron_events();
    }

    /**
     * Get Cron Event ID
     */
    public function get_event_id() {
        if ( ! $this->subscription_id ) {
            return 0;
        }

        $query = sumosubscriptions()->query->get( array(
            'type'         => $this->post_type,
            'status'       => 'publish',
            'return'       => 'q',
            'limit'        => 1,
            'meta_key'     => '_sumo_subscription_id',
            'meta_value'   => $this->subscription_id,
            'meta_compare' => '=',
                ) );

        foreach ( $query->posts as $event ) {
            $this->event_id = $event->ID;
            break;
        }

        return $this->event_id;
    }

    /**
     * Get Cron Events associated for the Subscription
     *
     * @return array
     */
    public function get_cron_events() {
        $_cron_events = get_post_meta( $this->event_id, '_sumo_subscription_cron_events', true );

        if ( isset( $_cron_events[ $this->subscription_id ] ) && is_array( $_cron_events[ $this->subscription_id ] ) ) {
            $this->cron_events = $_cron_events[ $this->subscription_id ];
        }

        return $this->cron_events;
    }

    /**
     * Set Cron Event to Schedule. It may be elapsed by wp_schedule_event
     *
     * @param int $timestamp
     * @param string $event_name
     * @param array $args
     * @return boolean true on success
     */
    public function set_cron_event( $timestamp, $event_name, $args = array() ) {
        if ( ! is_numeric( $timestamp ) || ! $timestamp || ! is_array( $args ) ) {
            return false;
        }

        $new_arg           = array( absint( $timestamp ) => $args );
        $this->cron_events = $this->get_cron_events();

        if ( $this->cron_events ) {
            //may the Event has multiple timestamps so that we are doing this way 
            if ( isset( $this->cron_events[ $event_name ] ) && is_array( $this->cron_events[ $event_name ] ) ) {
                $this->cron_events[ $event_name ] += $new_arg;
            } else {
                //may the new Event is registering
                $this->cron_events[ $event_name ] = $new_arg;
            }

            if ( $this->set_events() ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Update Cron events
     *
     * @return boolean true on success
     */
    public function set_events() {
        if ( ! is_array( $this->cron_events ) ) {
            return false;
        }

        update_post_meta( $this->event_id, '_sumo_subscription_cron_events', array(
            $this->subscription_id => $this->cron_events,
        ) );

        return true;
    }

    /**
     * UnSchedule Cron events.
     *
     * @param array $events unsetting every Cron events if left empty
     */
    public function unset_events( $events = array() ) {
        if ( empty( $events ) ) {
            $events = array(
                'start_subscription',
                'create_renewal_order',
                'notify_invoice_reminder',
                'notify_expiry_reminder',
                'notify_overdue',
                'notify_suspend',
                'notify_cancel',
                'notify_expire',
                'automatic_pay',
                'automatic_resume',
                'switch_to_manual_pay_mode',
                'retry_automatic_pay_in_overdue',
                'retry_automatic_pay_in_suspended',
            );
        }

        $events   = ( array ) $events;
        $events[] = 'switch_to_manual_pay_mode'; // deprecated

        if ( $this->cron_events ) {
            foreach ( $this->cron_events as $event_name => $event_args ) {
                if ( in_array( $event_name, $events ) ) {
                    unset( $this->cron_events[ $event_name ] );
                }
            }

            $this->set_events();
        }
    }

    /**
     * Map an array of action scheduler hooks to events.
     * 
     * @param string|array $hooks
     * @return array
     */
    public function map_events_from_hooks( $hooks = array() ) {
        $events = array();

        if ( ! empty( $hooks ) ) {
            if ( ! is_array( $hooks ) ) {
                $hooks = array( $hooks );
            }

            foreach ( $hooks as $hook ) {
                switch ( $hook ) {
                    case 'sumosubs_scheduled_subscription_activation':
                        $events[] = 'start_subscription';
                        break;
                    case 'sumosubs_scheduled_subscription_reactivation':
                        $events[] = 'automatic_resume';
                        break;
                    case 'sumosubs_scheduled_subscription_due_order_creation':
                        $events[] = 'create_renewal_order';
                        break;
                    case 'sumosubs_scheduled_subscription_due_payment_automatic':
                        $events[] = 'automatic_pay';
                        break;
                    case 'sumosubs_scheduled_subscription_due_payment_retry':
                        $events[] = 'retry_automatic_pay_in_overdue';
                        $events[] = 'retry_automatic_pay_in_suspended';
                        break;
                    case 'sumosubs_scheduled_subscription_reminder':
                        $events[] = 'notify_invoice_reminder';
                        $events[] = 'notify_expiry_reminder';
                        break;
                    case 'sumosubs_scheduled_subscription_overdue':
                        $events[] = 'notify_overdue';
                        break;
                    case 'sumosubs_scheduled_subscription_suspend':
                        $events[] = 'notify_suspend';
                        break;
                    case 'sumosubs_scheduled_subscription_cancel':
                        $events[] = 'notify_cancel';
                        break;
                    case 'sumosubs_scheduled_subscription_expiration':
                        $events[] = 'notify_expire';
                        break;
                }
            }
        }

        return $events;
    }

    /**
     * Gets the scheduled status args.
     * 
     * @return array
     */
    public function get_scheduled_status_args() {
        $args = array();

        if ( $this->cron_events ) {
            foreach ( $this->cron_events as $event_name => $event_args ) {
                if ( ! is_array( $event_args ) ) {
                    continue;
                }

                $timestamps = array_keys( $event_args );
                if ( ! $timestamps ) {
                    continue;
                }

                rsort( $timestamps );

                $scheduled_time      = current( $timestamps );
                $args                = isset( $event_args[ $scheduled_time ] ) ? $event_args[ $scheduled_time ] : array();
                $args[ 'timestamp' ] = $scheduled_time;

                switch ( $event_name ) {
                    case 'notify_overdue':
                        $args[ 'next_eligible_status' ] = 'Overdue';
                        break 2;
                    case 'notify_suspend':
                        $args[ 'next_eligible_status' ] = 'Suspended';
                        break 2;
                    case 'notify_cancel':
                        $args[ 'next_eligible_status' ] = 'Cancelled';
                        break 2;
                    case 'notify_expire':
                        $args[ 'next_eligible_status' ] = 'Expired';
                        break 2;
                    case 'automatic_pay':
                    case 'retry_automatic_pay_in_overdue':
                    case 'retry_automatic_pay_in_suspended':
                        break 2;
                }
            }
        }

        return $args;
    }
}
